// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.BooleanConsumer;

import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.PrimitiveIterator;
import java.util.function.Consumer;

/**
 * An iterator specialized for {@code boolean} values.
 * <p>
 * This is missing in the standard library: an iterator
 * for raw {@code boolean} values which does everything what
 * {@link OfInt} does for
 * [@code int}s.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 27, 2020
 */
public interface PrimitiveBooleanIterator
        extends PrimitiveIterator<Boolean, BooleanConsumer>
{
  /** Empty iterator of primitive boolean values. */
  PrimitiveBooleanIterator EMPTY = new PrimitiveBooleanIterator()
  {
    @Override
    public boolean nextBool()
    {
      throw new NoSuchElementException("Empty boolean iterator has no elements!");
    }

    @Override
    public boolean hasNext()
    {
      return false;
    }

    @Override
    public void forEachRemaining(@NotNull BooleanConsumer action)
    {
    }

    @Override
    public void forEachRemaining(@NotNull Consumer<? super Boolean> action)
    {
    }
  };

  /**
   * Returns the next {@code boolean} element in the iteration.
   *
   * @return the next {@code boolean} element in the iteration
   * @throws NoSuchElementException if the iteration has no more elements
   */
  boolean nextBool();

  /**
   * Performs the given action for each remaining element until all elements
   * have been processed or the action throws an exception.  Actions are
   * performed in the order of iteration, if that order is specified.
   * Exceptions thrown by the action are relayed to the caller.
   *
   * <p>The default implementation behaves as if:
   * <pre>{@code
   *     while (hasNext())
   *         action.accept(nextBoolean());
   * }</pre>
   *
   * @param action The action to be performed for each element
   * @throws NullPointerException if the specified action is null
   */
  default void forEachRemaining(@NotNull BooleanConsumer action) {
    Objects.requireNonNull(action);
    while (hasNext()) {
      action.accept(nextBool());
    }
  }

  /**
   * {@inheritDoc}
   * The default implementation boxes the result of calling
   * {@link #nextBool()}, and returns that boxed result.
   */
  @Override
  default Boolean next() {
    return nextBool();
  }

  /**
   * {@inheritDoc}
   * If the action is an instance of {@code BooleanConsumer} then it is cast
   * to {@code BooleanConsumer} and passed to {@link #forEachRemaining};
   * otherwise the action is adapted to an instance of
   * {@code BooleanConsumer}, by boxing the argument of {@code BooleanConsumer},
   * and then passed to {@link #forEachRemaining}.
   */
  @Override
  default void forEachRemaining(@NotNull Consumer<? super Boolean> action) {
    if (action instanceof BooleanConsumer) {
      forEachRemaining((BooleanConsumer) action);
    }
    else {
      // The method reference action::accept is never null
      Objects.requireNonNull(action);
      forEachRemaining((BooleanConsumer) action::accept);
    }
  }
}
