// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.BooleanConsumer;
import de.caff.generics.function.FragileBooleanConsumer;

import java.util.Iterator;

/**
 * Iterable which iterates over a primitive boolean.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 13, 2021
 * @see BooleanIndexable for creating one on the fly
 */
public interface PrimitiveBooleanIterable
        extends Iterable<Boolean>
{
  /**
   * Empty iterator of primitive boolean values.
   * @see PrimitiveBooleanIterator#EMPTY
   */
  PrimitiveBooleanIterator EMPTY_BOOLEAN_ITERATOR = PrimitiveBooleanIterator.EMPTY;
  /** Empty iterable of primitive boolean values. */
  PrimitiveBooleanIterable EMPTY = () -> EMPTY_BOOLEAN_ITERATOR;

  /**
   * Get a primitive boolean iterator.
   * @return boolean iterator
   */
  @NotNull
  PrimitiveBooleanIterator booleanIterator();

  @Override
  @NotNull
  default Iterator<Boolean> iterator()
  {
    return booleanIterator();
  }

  /**
   * Invoke the given action with each boolean value in this iterable.
   * @param action action to be call for each boolean value in sequemce.
   */
  default void forEachBool(@NotNull BooleanConsumer action)
  {
    for (PrimitiveBooleanIterator it = booleanIterator();  it.hasNext(); ) {
      action.accept(it.nextBool());
    }
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   * @deprecated use {@link #forEachBoolFragile(FragileBooleanConsumer)} instead
   */
  @Deprecated
  default <E extends Exception> void forEachBooleanFragile(@NotNull FragileBooleanConsumer<E> action)
          throws E
  {
    forEachBoolFragile(action);
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   */
  default <E extends Exception> void forEachBoolFragile(@NotNull FragileBooleanConsumer<E> action)
          throws E
  {
    for (PrimitiveBooleanIterator it = booleanIterator(); it.hasNext(); ) {
      action.accept(it.nextBool());
    }
  }

  /**
   * Does this boolean iterable contain the given value?
   * This is an operation which is linear to the size of this iterable
   * and should only be used rarely.
   * @param value value to check for
   * @return {@code true} if the value is contained in this iterable, {@code false} if not
   */
  default boolean containsBool(boolean value)
  {
    for (PrimitiveBooleanIterator iterator = booleanIterator();
         iterator.hasNext();  ) {
      if (iterator.nextBool() == value) {
        return true;
      }
    }
    return false;
  }

  /**
   * Does this boolean iterable contain any {@code true} values?
   * @return {@code true} if at least one value in this iterable is {@code true}
   */
  default boolean containsTrue()
  {
    return containsBool(true);
  }

  /**
   * Does this boolean iterable contain any {@code false} values?
   * @return {@code true} if at least one value in this iterable is {@code false}
   */
  default boolean containsFalse()
  {
    return containsBool(false);
  }

  /**
   * Get the truth value resulting from combining all values with logical or.
   * @return {@code true} if this is not empty and at least one value is {@code true},<br>
   *         {@code false} otherwise
   */
  default boolean or()
  {
    return containsTrue();
  }

  /**
   * Get the truth value resulting from combining all values with logical and.
   * @return {@code true} if this is not empty and all values are {@code true},<br>
   *         {@code false} otherwise
   */
  default boolean and()
  {
    return !containsFalse();
  }

  /**
   * Get the truth value resulting from combining all values with logical xor.
   * This assumes a start value of {@code false}.
   * @return resulting truth from combining all values in this iterable,<br>
   *         {@code false} if empty
   */
  default boolean xor()
  {
    boolean result = false;
    for (PrimitiveBooleanIterator it = booleanIterator();  it.hasNext();  )  {
      result ^= it.nextBool();
    }
    return result;
  }
}
