// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Ordering;

import java.util.Comparator;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * An ordered pair with equal types for first and second.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class Pair<T>
        extends OrderedPair<T, T>
        implements Iterable<T>
{
  /**
   * Constructor.
   * @param first  first entry
   * @param second second entry
   */
  public Pair(T first, T second)
  {
    super(first, second);
  }

  /**
   * Order the elements in this pair.
   * @param ordering ordering to apply
   */
  void orderBy(@NotNull Ordering<? super T> ordering)
  {
    if (ordering.check(first, second) == Order.Descending) {
      swap();
    }
  }

  /**
   * Returns an iterator over this pair.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  public Iterator<T> iterator()
  {
    return new Iterator<T>()
    {
      int pos = 0;
      @Override
      public boolean hasNext()
      {
        return pos < 2;
      }

      @Override
      public T next()
      {
        if (pos < 2) {
          switch (pos++) {
          case 0:
            return first;
          case 1:
            return second;
          }
        }
        throw new NoSuchElementException("Cannot iterate more than twice in a Pair!");
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException("Cannot remove Pair item!");
      }
    };
  }

  /**
   * Exchange first and second.
   */
  public void swap()
  {
    final T tmp = first;
    first = second;
    second = tmp;
  }

  /**
   * Get the first or second parameter via its index.
   * This can come handy in special situations.
   * @param index either {@code 0} or {@code 1}
   * @return if {@code index} is {@code 0} the first entry,
   *         if {@code index} is {@code 1} the second entry
   */
  public T get(int index)
  {
    switch (index)
    {
    case 0:
      return first;
    case 1:
      return second;
    default:
      throw new IllegalArgumentException("Only allowed indexes are 0 or 1!");
    }
  }

  /**
   * Set the first or second parameter via its index.
   * This can come handy in special situations
   * @param index either {@code 0} or {@code 1}
   * @param item  new first if {@code index} is {@code 0},
   *              new second if {@code index} is {@code 1}
   */
  public void set(int index, T item)
  {
    switch (index)
    {
    case 0:
      first = item;
      break;
    case 1:
      second = item;
      break;
    default:
      throw new IllegalArgumentException("Only allowed indexes are 0 or 1!");
    }
  }

  /**
   * Get a comparator which compares the first entries of pairs using their natural order.
   * @param <S> pair entry type (comparable)
   * @return comparator for ordered pairs, comparing the first entry
   */
  public static <S extends Comparable<S>> Comparator<Pair<S>> getFirstEntryPairComparator()
  {
    return Comparator.comparing(p -> p.first);
  }

  /**
   * Get a comparator which compares the first entries of pairs using a comparator.
   * @param <S> pair entry type
   * @param comparator comparator for inner elements
   * @return comparator for ordered pairs, comparing the first entry
   */
  @NotNull
  public static <S> Comparator<Pair<S>> getFirstEntryPairComparator(@NotNull final Comparator<S> comparator)
  {
    return (p1, p2) -> comparator.compare(p1.first, p2.first);
  }

  /**
   * Get a comparator which compares the first entries of pairs using their natural order.
   * @param <S> pair entry type (comparable)
   * @return comparator for ordered pairs, comparing the second entry
   */
  @NotNull
  public static <S extends Comparable<S>> Comparator<Pair<S>> getSecondEntryPairComparator()
  {
    return Comparator.comparing(p -> p.second);
  }

  /**
   * Get a comparator which compares the second entries of pairs using a comparator.
   * @param <S> pair entry type
   * @param comparator comparator for inner elements
   * @return comparator for ordered pairs, comparing the second entry
   */
  @NotNull
  public static <S> Comparator<Pair<S>> getSecondEntryPairComparator(@NotNull final Comparator<S> comparator)
  {
    return (p1, p2) -> comparator.compare(p1.second, p2.second);
  }

  /**
   * Get a comparator which compares the first entries of two pairs using their natural order.,
   * and then the second entries of the pairs if the first are equal.
   * @param <S> pair entry type (comparable)
   * @return comparator for ordered pairs, comparing both entries in order
   */
  @NotNull
  public static <S extends Comparable<S>> Comparator<Pair<S>> getFirstThanSecondEntryPairComparator()
  {
    return Comparator.comparing((Pair<S> o) -> o.first).thenComparing(o -> o.second);
  }

  /**
   * Get a comparator which compares the first entries of two pairs,
   * and then the second entries of the pairs if the first are equal.
   * @param comparator comparator for inner elements
   * @param <S> pair entry type (comparable)
   * @return comparator for ordered pairs, comparing both entries in order
   */
  @NotNull
  public static <S> Comparator<Pair<S>> getFirstThanSecondEntryPairComparator(@NotNull final Comparator<S> comparator)
  {
    return (o1, o2) -> {
      int result = comparator.compare(o1.first, o2.first);
      return result == 0
              ? comparator.compare(o1.second, o2.second)
              : result;
    };
  }

  /**
   * Create a pair from two items.
   * @param f first item
   * @param s second item
   * @param <U> item type
   * @return pair of items
   */
  @NotNull
  public static <U> Pair<U> createPair(U f, U s)
  {
    return new Pair<>(f, s);
  }

}
