// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.tuple.ITuple2;

import java.util.Comparator;
import java.util.Map;
import java.util.Objects;

/**
 * A simple ordered pair.
 * <p>
 * Useful for temporary packing.
 * <p>
 * It implements also {@link java.util.Map.Entry}, and will return the first value as key and the second value as value.
 * Obviously it will not allow setting to the underlying map.
 *
 * @param <S> type of first entry
 * @param <T> type of second entry
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class OrderedPair<S, T>
        implements Map.Entry<S, T>,
                   ITuple2<S, T>
{
  /** The first element. */
  public S first;
  /** The second element. */
  public T second;

  /**
   * Constructor.
   * @param first  first element of the pair
   * @param second second element of the pair
   */
  public OrderedPair(S first, T second)
  {
    this.first = first;
    this.second = second;
  }

  /**
   * Returns the key corresponding to this entry.
   *
   * @return the key corresponding to this entry.
   * @throws IllegalStateException implementations may, but are not
   *                               required to, throw this exception if the entry has been
   *                               removed from the backing map
   */
  @Override
  public S getKey()
  {
    return first;
  }

  /**
   * Returns the value corresponding to this entry.  If the mapping
   * has been removed from the backing map (by the iterator's
   * <tt>remove</tt> operation), the results of this call are undefined.
   *
   * @return the value corresponding to this entry.
   * @throws IllegalStateException implementations may, but are not
   *                               required to, throw this exception if the entry has been
   *                               removed from the backing map
   */
  @Override
  public T getValue()
  {
    return second;
  }

  /**
   * Replaces the value corresponding to this entry with the specified
   * value (optional operation).  (Writes through to the map.)  The
   * behavior of this call is undefined if the mapping has already been
   * removed from the map (by the iterator's <tt>remove</tt> operation).
   *
   * @param value new value to be stored in this entry.
   * @return old value corresponding to the entry.
   * @throws UnsupportedOperationException if the <tt>put</tt> operation
   *                                       is not supported by the backing map.
   * @throws ClassCastException            if the class of the specified value
   *                                       prevents it from being stored in the backing map.
   * @throws IllegalArgumentException      if some aspect of this value
   *                                       prevents it from being stored in the backing map.
   * @throws NullPointerException          if the backing map does not permit
   *                                       <tt>null</tt> values, and the specified value is
   *                                       <tt>null</tt>.
   * @throws IllegalStateException         implementations may, but are not
   *                                       required to, throw this exception if the entry has been
   *                                       removed from the backing map
   */
  @Override
  public T setValue(T value)
  {
    throw new UnsupportedOperationException("Not setting for ordered pairs used as map entries!");
  }

  @Override
  public S _1()
  {
    return first;
  }

  @Override
  public T _2()
  {
    return second;
  }

  /**
   * Is this pair equal to another?
   * This just checks whether both first and second are equal in each pair.
   * @param o other pair
   * @return {@code true} if both elements are equal<br>
   *         {@code false} otherwise
   */
  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    final OrderedPair<?,?> that = (OrderedPair<?,?>)o;

    return Objects.deepEquals(first, that.first) &&
           Objects.deepEquals(second, that.second);
  }

  /**
   * Get a hash code.
   * @return hash code
   */
  @Override
  public int hashCode()
  {
    return Objects.hash(first, second);
  }

  /**
   * Get a ordered pair with inverted order.
   * @return pair with swapped {@link #first} and {@link #second} entries
   */
  @NotNull
  public OrderedPair<T, S> swapped()
  {
    return create(second, first);
  }

  @Override
  public String toString()
  {
    return String.format("Pair(%s,%s)", first, second);
  }

  /**
   * Get a comparator which compares the first entries of ordered pairs using their natural order.
   * @param <U> first entry type
   * @param <V> second entry type
   * @return comparator for ordered pairs, comparing the first entry
   */
  @NotNull
  public static <U extends Comparable<U>, V> Comparator<OrderedPair<U, V>> getFirstEntryOrderedPairComparator()
  {
    return Comparator.comparing(p -> p.first);
  }

  /**
   * Get a comparator which compares the first entries of ordered pairs using a comparator.
   * @param <U> first entry type
   * @param <V> second entry type
   * @param comparator comparator for first entry
   * @return comparator for ordered pairs, comparing the first entry
   */
  @NotNull
  public static <U, V> Comparator<OrderedPair<U, V>> getFirstEntryOrderedPairComparator(final Comparator<U> comparator)
  {
    return (p1, p2) -> comparator.compare(p1.first, p2.first);
  }

  /**
   * Get a comparator which compares the second entries of ordered pairs using their natural order.
   * @param <U> first entry type
   * @param <V> second entry type
   * @return comparator for ordered pairs, comparing the second entry
   */
  @NotNull
  public static <U, V extends Comparable<V>> Comparator<OrderedPair<U, V>> getSecondEntryOrderedPairComparator()
  {
    return Comparator.comparing(p -> p.second);
  }

  /**
   * Get a comparator which compares the second entries of ordered pairs using a comparator.
   * @param <U> first entry type
   * @param <V> second entry type
   * @param comparator comparator for first entry
   * @return comparator for ordered pairs, comparing the first entry
   */
  @NotNull
  public static <U, V> Comparator<OrderedPair<U, V>> getSecondEntryOrderedPairComparator(final Comparator<V> comparator)
  {
    return (p1, p2) -> comparator.compare(p1.second, p2.second);
  }

  /**
   * Create an ordered pair from two values.
   * @param firstValue first value
   * @param secondValue second value
   * @param <U> first value type
   * @param <V> second value type
   * @return ordered pair
   */
  @NotNull
  public static <U, V> OrderedPair<U, V> create(U firstValue, V secondValue)
  {
    return new OrderedPair<>(firstValue, secondValue);
  }

  /**
   * Create an ordered pair from a map entry.
   * @param mapEntry map entry
   * @param <U>      first value (map key) type
   * @param <V>      second value (map value) type
   * @return ordered pair
   */
  @NotNull
  public static <U, V> OrderedPair<U, V> create(Map.Entry<U, V> mapEntry)
  {
    return create(mapEntry.getKey(), mapEntry.getValue());
  }
}
