// ============================================================================
// File:               Order
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            1/11/23 4:00 PM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

/**
 * Order of two values.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 11, 2023
 */
public enum Order
{
  /** First value comes before second value. */
  Ascending(-1, true, false, false),
  /** Values are the same. */
  Same(0, false, false, true),
  /** First value comes after second value. */
  Descending(1, false, true, false);

  static {
    // compare https://caff.de/posts/enum_siblings_access/
    Ascending.inverse  = Descending;
    Same.inverse       = Same;
    Descending.inverse = Ascending;
  }

  /** The equivalent of this order when used for standard compare. */
  public final int comparison;
  /**
   * Is this order ascending?
   * This means that for two values {@code v1, v2} this order
   * says that {@code v1 < v2}.
   */
  public final boolean ascending;
  /**
   * Is this order ascending or the same?
   * This means that for two values {@code v1, v2} this order
   * says that {@code v1 <= v2}.
   */
  public final boolean ascendingOrSame;
  /**
   * Is this order descending?
   * This means that for two values {@code v1, v2} this order
   * says that {@code v2 > v1}.
   */
  public final boolean descending;
  /**
   * Is this order descending or the same?
   * This means that for two values {@code v1, v2} this order
   * says that {@code v1 >= v2}.
   */
  public final boolean descendingOrSame;
  /**
   * Is this order considering two values equal?
   * This means that for two values {@code v1, v2} this order
   * says that {@code v1 == v2}.
   */
  public final boolean same;
  /**
   * Is this order considering two values equal?
   * This means that for two values {@code v1, v2} this order
   * says that {@code v1 != v2}.
   */
  public final boolean different;

  /** Inverse order, set by static initializer. */
  private Order inverse;

  /**
   * Constructor.
   * @param comparison equivalent comparison result
   */
  Order(int comparison, boolean asc, boolean desc, boolean same)
  {
    this.comparison = comparison;
    ascending = asc;
    ascendingOrSame = asc || same;
    descending = desc;
    descendingOrSame = desc || same;
    this.same = same;
    this.different = !same;
  }

  /**
   * The values of this enum.
   * Use this instead of {@link #values()} because it does not require internal array copying.
   */
  @NotNull
  public static final Indexable<Order> VALUES = Indexable.viewArray(values());

  /**
   * Get the order which is the inverse of this one.
   * @return inverse order
   */
  @NotNull
  public Order inverse()
  {
    return inverse;
  }

  /**
   * Get the Order enum associated with the given internal value.
   *
   * @param internalValue internal value
   * @return associated enum value, or {@code null} if there is none
   */
  @Nullable
  public static Order fromInternal(int internalValue)
  {
    return Enums.getEnumFromOrdinal(VALUES, internalValue);
  }

  /**
   * Get the Order enum associated with the given internal value.
   *
   * @param internalValue internal value
   * @param defaultValue  default value
   * @return associated enum value, or {@code defaultValue} if there is none
   */
  @NotNull
  public static Order fromInternal(int internalValue, @NotNull Order defaultValue)
  {
    return Enums.getEnumFromOrdinal(VALUES, internalValue, defaultValue);
  }

  /**
   * Get the order from a standard {@link java.util.Comparator} or
   * {@link Comparable} result.
   * @param value comparison value, negative means ascending, positive means descending,
   *              {@code 0} means same
   * @return order equivalent
   */
  @NotNull
  public static Order fromCompare(int value)
  {
    return value < 0
            ? Ascending
            : (value > 0
                       ? Descending
                       : Same);
  }
}
