// ============================================================================
// File:               Union
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            7/26/24 2:02 PM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.FragileProcedure1;

import java.util.Objects;
import java.util.function.Consumer;

/**
 * This class defines a union structure for two values, where only one of them
 * exists.
 * This is sometimes useful to transport errors from one part to the other, when
 * throwing them is no option. This is a type-safe implementation of a 2-member {@code union}
 * structure as defined in the C programming language.
 * <p>
 * This class allows no overriding to guarantee the following invariances:
 * <ul>
 *   <li>{@code u.has1() ^ u.has2()} is always {@code true}</li>
 * </ul>
 * A union is immutable in the sense that it itself does not change.
 * The contained values may change if they are mutable.
 * <p>
 * There is exactly one way to construct a union of either type with valid values:
 * <ul>
 *   <li>{@link #t1(Object)} will construct a union which contains a value of type {@code T1}</li>
 *   <li>{@link #t2(Object)} will construct a union which contains a value of type {@code T2}</li>
 * </ul>
 * And two more for {@code null} values:
 * <ul>
 *   <li>{@link #nul1()} will return a static union which contains a {@code null} value of type {@code T1}</li>
 *   <li>{@link #nul2()} will return a static union which contains a {@code null} value of type {@code T2}</li>
 * </ul>
 * This implementation allows for either type to be {@code null}.
 * Any consumers should be prepared for {@code null} values.
 * See {@link Union} for an implementation which does not allow for {@code null} values.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 26, 2024
 * @param <T1> type of the first possible value
 * @param <T2> type of the second possible value
 */
public abstract class NullableUnion<T1, T2>
        extends UnionBase<T1, T2>
{
  /** Fix value for a union of first type with value {@code null}. */
  private static final NullableUnion<Object, Object> NULL1 = new NullableUnion<Object, Object>()
  {
    @Override
    public boolean has1()
    {
      return true;
    }

    @Override
    public boolean has2()
    {
      return false;
    }

    @Override
    public Object get1() throws IllegalStateException
    {
      return null;
    }

    @Override
    public Object get2() throws IllegalStateException
    {
      throw new IllegalStateException("Requesting type 2 from a union containing type 1!");
    }

    @Override
    public Object getValue()
    {
      return null;
    }

    @Override
    public void dispose(@NotNull Consumer<? super Object> handler1, @NotNull Consumer<? super Object> handler2)
    {
      handler1.accept(null);
    }

    @NotNull
    @Override
    public NullableUnion<Object, Object> disposeOnly1(@NotNull Consumer<? super Object> handler)
    {
      handler.accept(null);
      return this;
    }

    @NotNull
    @Override
    public NullableUnion<Object, Object> disposeOnly2(@NotNull Consumer<? super Object> handler)
    {
      return this;
    }

    @NotNull
    @Override
    public <E extends Exception> NullableUnion<Object, Object> disposeFragile1(
            @NotNull FragileProcedure1<E, ? super Object> handler) throws E
    {
      handler.apply(null);
      return this;
    }

    @NotNull
    @Override
    public <E extends Exception> NullableUnion<Object, Object> disposeFragile2(
            @NotNull FragileProcedure1<E, ? super Object> handler) throws E
    {
      return this;
    }

    @Override
    public int hashCode()
    {
      return 0;
    }

    @Override
    public boolean equals(Object obj)
    {
      return UnionBase.isEqual1(this, obj);
    }

    @Override
    @NotNull
    public String toString()
    {
      return "Union{1:null}";
    }
  };

  /** Fix value for a union of second type with value {@code null}. */
  private static final NullableUnion<Object, Object> NULL2 = new NullableUnion<Object, Object>()
  {
    @Override
    public boolean has1()
    {
      return false;
    }

    @Override
    public boolean has2()
    {
      return true;
    }

    @Override
    public Object get1() throws IllegalStateException
    {
      throw new IllegalStateException("Requesting type 1 from a union containing type 2!");
    }

    @Override
    public Object get2() throws IllegalStateException
    {
      return null;
    }

    @Override
    public Object getValue()
    {
      return null;
    }

    @Override
    public void dispose(@NotNull Consumer<? super Object> handler1, @NotNull Consumer<? super Object> handler2)
    {
      handler2.accept(null);
    }

    @NotNull
    @Override
    public NullableUnion<Object, Object> disposeOnly1(@NotNull Consumer<? super Object> handler)
    {
      return this;
    }

    @NotNull
    @Override
    public NullableUnion<Object, Object> disposeOnly2(@NotNull Consumer<? super Object> handler)
    {
      handler.accept(null);
      return this;
    }

    @NotNull
    @Override
    public <E extends Exception> NullableUnion<Object, Object> disposeFragile1(
            @NotNull FragileProcedure1<E, ? super Object> handler) throws E
    {
      return this;
    }

    @NotNull
    @Override
    public <E extends Exception> NullableUnion<Object, Object> disposeFragile2(
            @NotNull FragileProcedure1<E, ? super Object> handler) throws E
    {
      handler.apply(null);
      return this;
    }

    @Override
    public int hashCode()
    {
      return 0;
    }

    @Override
    public boolean equals(Object obj)
    {
      return UnionBase.isEqual2(this, obj);
    }

    @Override
    @NotNull
    public String toString()
    {
      return "Union{2:null}";
    }
  };

  /**
   * Don't allow construction from elsewhere.
   */
  private NullableUnion() {}

  /**
   * Get the value if it is of type {@code T1}.
   * Throw an exception otherwise.
   *
   * @return value of type {@code T1}, possibly {@code null}
   * @throws IllegalStateException if this union does contain a value of type {@code T2}
   * @see #has1()
   */
  @Nullable
  @Override
  public abstract T1 get1() throws IllegalStateException;

  /**
   * Get the value if it is of type {@code T2}.
   * Throw an exception otherwise.
   *
   * @return value of type {@code T2}, possibly {@code null}
   * @throws IllegalStateException if this union does contain a value of type {@code T1}
   * @see #has1()
   */
  @Nullable
  @Override
  public abstract T2 get2() throws IllegalStateException;

  /**
   * Get the value contained in this union.
   * @return the value of this union, having either type {@code T1} or {@code T2},
   *         possibly {@code null}
   */
  @Nullable
  @Override
  public Object getValue()
  {
    return null;
  }

  /**
   * Create a union containing a value of type V1.
   * @param v1 value contained in this union
   * @return union with value {@code v1}
   * @param <V1> value type of contained value
   * @param <V2> implicit
   */
  @NotNull
  public static <V1, V2> NullableUnion<V1, V2> t1(@Nullable V1 v1)
  {
    if (v1 == null) {
      return nul1();
    }
    return new NullableUnion<V1, V2>() {
      @Override
      public boolean has1()
      {
        return true;
      }

      @Override
      public boolean has2()
      {
        return false;
      }

      @Override
      public V1 get1() throws IllegalStateException
      {
        return v1;
      }

      @Override
      public V2 get2() throws IllegalStateException
      {
        throw new IllegalStateException("Requesting type 2 from a union containing type 1!");
      }

      @Override
      public Object getValue()
      {
        return v1;
      }

      @Override
      public void dispose(@NotNull Consumer<? super V1> handler1, @NotNull Consumer<? super V2> handler2)
      {
        handler1.accept(v1);
      }

      @NotNull
      @Override
      public NullableUnion<V1, V2> disposeOnly1(@NotNull Consumer<? super V1> handler)
      {
        handler.accept(v1);
        return this;
      }

      @NotNull
      @Override
      public NullableUnion<V1, V2> disposeOnly2(@NotNull Consumer<? super V2> handler)
      {
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> NullableUnion<V1, V2> disposeFragile1(@NotNull FragileProcedure1<E, ? super V1> handler)
              throws E
      {
        handler.apply(v1);
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> NullableUnion<V1, V2> disposeFragile2(@NotNull FragileProcedure1<E, ? super V2> handler)
              throws E
      {
        return this;
      }

      @Override
      public int hashCode()
      {
        return Objects.hash(v1);
      }

      @Override
      public boolean equals(Object obj)
      {
        return UnionBase.isEqual1(this, obj);
      }

      @Override
      @NotNull
      public String toString()
      {
        return "Union{1:"+v1+"}";
      }
    };
  }

  /**
   * Create a union containing a value of type V2.
   * @param v2 value contained in this union
   * @return union with value {@code v2}
   * @param <V1> implicit
   * @param <V2> value type of contained value
   */
  @NotNull
  public static <V1, V2> NullableUnion<V1, V2> t2(@Nullable V2 v2)
  {
    if (v2 == null) {
      return nul2();
    }
    return new NullableUnion<V1, V2>() {
      @Override
      public boolean has1()
      {
        return false;
      }

      @Override
      public boolean has2()
      {
        return true;
      }

      @Override
      public V1 get1() throws IllegalStateException
      {
        throw new IllegalStateException("Requesting type 1 from a union containing type 2!");
      }

      @Override
      public V2 get2() throws IllegalStateException
      {
        return v2;
      }

      @Override
      public Object getValue()
      {
        return v2;
      }

      @Override
      public void dispose(@NotNull Consumer<? super V1> handler1, @NotNull Consumer<? super V2> handler2)
      {
        handler2.accept(v2);
      }

      @NotNull
      @Override
      public NullableUnion<V1, V2> disposeOnly1(@NotNull Consumer<? super V1> handler)
      {
        return this;
      }

      @NotNull
      @Override
      public NullableUnion<V1, V2> disposeOnly2(@NotNull Consumer<? super V2> handler)
      {
        handler.accept(v2);
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> NullableUnion<V1, V2> disposeFragile1(@NotNull FragileProcedure1<E, ? super V1> handler)
              throws E
      {
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> NullableUnion<V1, V2> disposeFragile2(@NotNull FragileProcedure1<E, ? super V2> handler)
              throws E
      {
        handler.apply(v2);
        return this;
      }

      @Override
      public int hashCode()
      {
        return Objects.hash(v2);
      }

      public boolean equals(Object obj)
      {
        return UnionBase.isEqual2(this, obj);
      }

      @Override
      @NotNull
      public String toString()
      {
        return "Union{2:"+v2+"}";
      }
    };
  }

  /**
   * Get a union with a first type of {@code null}.
   * @return union of type {@code V1}, with a value of {@code null}
   * @param <V1> first possible type
   * @param <V2> second possible type
   */
  @NotNull
  @SuppressWarnings("unchecked") // as returned union is immutable
  public static <V1, V2> NullableUnion<V1, V2> nul1()
  {
    return (NullableUnion<V1, V2>)NULL1;
  }

  /**
   * Get a union with a second type of {@code null}.
   * @return union of type {@code V2}, with a value of {@code null}
   * @param <V1> first possible type
   * @param <V2> second possible type
   */
  @NotNull
  @SuppressWarnings("unchecked") // as returned union is immutable
  public static <V1, V2> NullableUnion<V1, V2> nul2()
  {
    return (NullableUnion<V1, V2>)NULL2;
  }
}
