// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2022-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.tuple.ITuple2;

/**
 * A nullable value with a name.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 25, 2022
 */
public interface NamedNullableValue<T>
        extends ITuple2<String, T>
{
  /**
   * Get the name of the named value.
   * @return the name
   */
  @NotNull
  String getName();

  /**
   * Get the value.
   * @return the value
   */
  @Nullable
  T getValue();

  @NotNull
  default String _1()
  {
    return getName();
  }

  @Nullable
  default T _2()
  {
    return getValue();
  }

  /**
   * Create a named value.
   * @param name  name
   * @param value value
   * @return named value combination
   * @param <V> value type
   */
  @NotNull
  static <V> NamedNullableValue<V> of(@NotNull String name, @Nullable V value)
  {
    return new NamedNullableValue<V>()
    {
      @NotNull
      @Override
      public String getName()
      {
        return name;
      }

      @Nullable
      @Override
      public V getValue()
      {
        return value;
      }

      @Override
      public int hashCode()
      {
        return ITuple2.hash(this);
      }

      @Override
      public boolean equals(Object obj)
      {
        return ITuple2.equals(this, obj);
      }

      @Override
      public String toString()
      {
        return NamedNullableValue.toString(this);
      }
    };
  }

  /**
   * Default implementation for creating a string representation
   * for a named value.
   * @param nv named value
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull NamedNullableValue<?> nv)
  {
    return nv.getName()+": "+nv.getValue();
  }
}
