// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.algorithm.TimSort;
import de.caff.generics.function.*;

import java.lang.reflect.Array;
import java.util.*;
import java.util.function.*;

/**
 * Indexable with fixed length but mutable content.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public interface MutableIndexable<T>
        extends Indexable<T>
{
  /**
   * Set the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @param elem  element to put to the given index
   */
  void set(int index, T elem);

  /**
   * Pythonesque set.
   * This allows to access elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @param elem element to put to the given index
   */
  default void syt(int index, T elem)
  {
    set(Pythonesque.mapX(index, this), elem);
  }

  /**
   * Set the elements of this mutable indexable one after
   * the other from the given iterable until either the
   * iterable is emptied or {@link #size()} elements are set.
   * @param iterable iterable from which this indexable is filled
   * @return number of elements set
   */
  default int setFrom(@NotNull Iterable<? extends T> iterable)
  {
    return setFrom(iterable, 0, size());
  }

  /**
   * Set the elements of this mutable indexable one after
   * the other from the given iterable until either the
   * iterable is emptied or {@code numElements} elements are set.
   * @param iterable     iterable from which this indexable is filled
   * @param startIndex   start index where the setting begins
   * @param numElements  number of elements to set
   * @return number of elements set
   */
  default int setFrom(@NotNull Iterable<? extends T> iterable,
                      int startIndex, int numElements)
  {
    if (startIndex > size() - numElements) {
      numElements = size() - startIndex;
    }

    int n = 0;

    for (T elem : iterable) {
      set(startIndex + n++, elem);
      if (n == numElements) {
        break;
      }
    }
    return n;
  }

  /**
   * Set consecutive entries from an array.
   * This will set less elements
   * @param array       array with elements to set from
   * @param arrayIndex  start index in the array
   * @param startIndex  start index in this mutable indexable
   * @param numElements number of elements to copy
   * @throws IndexOutOfBoundsException if this mutable indexable would overflow by this operation
   * @throws ArrayIndexOutOfBoundsException if array overflows by this operation
   */
  default void setFromArray(@NotNull T[] array,
                           int arrayIndex,
                           int startIndex,
                           int numElements)
  {
    if (startIndex + numElements > size()) {
      throw new IndexOutOfBoundsException("Overflow: numElements is too large!");
    }

    for (int n = 0;  n < numElements;  ++n) {
      set(startIndex + n, array[arrayIndex++]);
    }
  }

  /**
   * Set multiple elements to the same value.
   * @param from  first index to be set (Pythonesque)
   * @param len   number of elements to be set (non-negative)
   * @param value value to be set
   */
  default void setMulti(int from, int len, T value)
  {
    from = Pythonesque.mapX(from, this);
    if (len < 0) {
      throw new IllegalArgumentException("len has to be non-negative: "+len);
    }
    final int end = from + len;
    if (end > size()) {
      throw new IndexOutOfBoundsException(String.format("from + len exceed size(): %d + %d > %d", from, len, size()));
    }
    for (int i = from;  i < end;  ++i) {
      set(i, value);
    }
  }

  /**
   * Copy elements inside this indexable.
   * @param fromIndex   start index from where elements are copied (Pythonesque)
   * @param toIndex     start index to where elements are copied (Pythonesque)
   * @param numElements number of elements to copy
   */
  default void copyInternally(int fromIndex, int toIndex, int numElements)
  {
    if (numElements == 0) {
      return;
    }
    if (numElements < 0) {
      throw new IllegalArgumentException("numElements must not be negative!");
    }
    final int size = size();
    int from = Pythonesque.mapX(fromIndex, size);
    int to   = Pythonesque.mapX(toIndex, size);
    if (from + numElements > size) {
      throw new IndexOutOfBoundsException(String.format("Reading would overflow: from + numElements > size (%d + %d > %d)!",
                                                        from, numElements, size));
    }
    if (to + numElements > size) {
      throw new IndexOutOfBoundsException(String.format("Writing would overflow: to + numElements > size (%d + %d > %d)!",
                                                        to, numElements, size));
    }
    if (from == to) {
      return;
    }
    if (from < to  &&  from + numElements > to) {
      // have to go backward
      from += numElements;
      to   += numElements;
      while (--numElements >= 0) {
        set(--to, get(--from));
      }
    }
    else {
      while (--numElements >= 0) {
        set(to++, get(from++));
      }
    }
  }

  /**
   * Fill this indexable from a provider.
   * @param provider provider which is called for each index and provides the element for that index
   */
  default void fillFrom(@NotNull IntFunction<? extends T> provider)
  {
    final int size = size();
    for (int p = 0;  p < size;  ++p) {
      set(p, provider.apply(p));
    }
  }

  /**
   * Swap the values at two indices.
   * @param idx1 first index
   * @param idx2 second index
   */
  default void swap(int idx1, int idx2)
  {
    if (idx1 == idx2) {
      return;
    }
    final T tmp = get(idx1);
    set(idx1, get(idx2));
    set(idx2, tmp);
  }

  /**
   * Swap the values at two indices using Pythonesque indices.
   * @param idx1 first index ({@link Pythonesque})
   * @param idx2 second index ({@link Pythonesque})
   */
  default void swyp(int idx1, int idx2)
  {
    if (idx1 == idx2) {
      return;
    }
    swap(Pythonesque.mapX(idx1, this),
         Pythonesque.mapX(idx2, this));
  }

  @NotNull
  @Override
  default MutableIndexable<T> subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0) {
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    }
    if (toIndex > size()) {
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    }
    if (fromIndex > toIndex) {
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    }
    final int length = toIndex - fromIndex;

    return new Base<T>()
    {
      @Override
      public void set(int index, T elem)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("index = "+index);
        }
        MutableIndexable.this.set(index + fromIndex,
                                  elem);
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public T get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("index = "+index);
        }
        return MutableIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public MutableIndexable<T> subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > size()) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return MutableIndexable.this.subSet(fromIndex + fromIdx,
                                            fromIndex + toIdx - fromIdx);
      }

      @Override
      public void copyInternally(int fromIdx, int toIdx, int numElements)
      {
        final int from = Pythonesque.mapX(fromIndex, length);
        final int to   = Pythonesque.mapX(toIndex, length);
        if (from + numElements > length) {
          throw new IndexOutOfBoundsException(String.format("Reading would overflow: from + numElements > size (%d + %d > %d)!",
                                                            from, numElements, length));
        }
        if (to + numElements > length) {
          throw new IndexOutOfBoundsException(String.format("Writing would overflow: to + numElements > size (%d + %d > %d)!",
                                                            to, numElements, length));
        }
        // now forward to wrapped
        MutableIndexable.this.copyInternally(fromIndex + fromIdx,
                                             fromIndex + toIdx,
                                             numElements);
      }
    };
  }

  @NotNull
  @Override
  default MutableIndexable<T> sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  @NotNull
  @Override
  default MutableIndexable<T> tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  @NotNull
  @Override
  default MutableIndexable<T> headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Overridden to allow setting via the returned iterator.
   * @return list iterator which allows {@link ListIterator#set(Object) setting},
   *         but not {@link ListIterator#add(Object) adding}
   *         or {@link ListIterator#remove() deleting}
   */
  @NotNull
  @Override
  default ListIterator<T> listIterator()
  {
    return new ListIterator<T>() {
      private int index = 0;
      private int lastRet = -1;  // last returned element index, necessary for wet()

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public T next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(lastRet = index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public T previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(lastRet = --index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(T t)
      {
        if (lastRet < 0) {
          throw new IllegalStateException();
        }
        MutableIndexable.this.set(lastRet, t);
      }

      @Override
      public void add(T t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  @NotNull
  @Override
  default MutableIndexable<T> reverse()
  {
    return new MutableIndexable<T>()
    {
      @Override
      public void set(int index, T elem)
      {
        MutableIndexable.this.set(size() - index - 1, elem);
      }

      @Override
      public int size()
      {
        return MutableIndexable.this.size();
      }

      @Override
      public T get(int index)
      {
        return MutableIndexable.this.get(size() - index - 1);
      }

      @NotNull
      @Override
      public MutableIndexable<T> reverse()
      {
        return MutableIndexable.this;
      }
    };
  }

  /**
   * Revert the order of all elements in the given range.
   * Attention: the {@code to} element is also included in the reversion to allow easy
   * reversion at the end by using {@code -1}.
   * @param from start element of reversion (Pythonesque), included
   * @param to   end element of reversion (Pythonesque), included
   */
  default void revert(int from, int to)
  {
    final int size = size();
    int up = Pythonesque.map(from, size);
    int down = Pythonesque.map(to, size);

    while (up < down) {
      swap(up++, down--);
    }
  }

  /**
   * Revert the elements in this indexable.
   * This reverts the order of all elements.
   */
  default void revert()
  {
    if (size() <= 1) {
      return;
    }
    revert(0, -1);
  }

  /**
   * Order the elements in this indexable by the given ordering.
   * After this call the elements of this indexable are reordered
   * so that {@link #isOrdered(Ordering)} returns {@code true}
   * for the given {@code ordering}.
   * <p>
   * Introduced by the overhead due to accessing elements via this interface
   * ordering via this method is some 1% slower than sorting an array
   * with standard {@link Arrays#sort(Object[], Comparator)}. Sort the array
   * first, if this is a problem for you.
   * @param ordering ordering to apply
   */
  default void order(@NotNull Ordering<? super T> ordering)
  {
    TimSort.sort(this, ordering);
  }

  /**
   * Randomize the content  of this mutable indexable.
   * This will reorder the elements randomly.
   * @param random random number generator
   */
  default void shuffle(@NotNull Random random)
  {
    final int size = size();
    for (int i = size - 1;  i >= 0;  --i) {
      swap(i, random.nextInt(i + 1));
    }
  }

  /**
   * View this indexable as a standard list.
   * Overwritten to allow for modifications.
   * The returned list will allow usage of the {@link List#set(int, Object)} method,
   * but neither any adding nor deleting methods. Standard sorting algorithms will work
   * on the returned list as a sorting algorithm is expected to neither add nor delete
   * elements.
   * @return list allowing setting elements
   */
  @NotNull
  @Override
  default List<T> asList()
  {
    return new ListView<>(this);
  }

  /**
   * Apply the given operator on each element.
   * @param operator operator to apply
   */
  default void applyOnEach(@NotNull Function<? super T, ? extends T> operator)
  {
    for (int i = size() - 1;  i >= 0;  --i) {
      set(i, operator.apply(get(i)));
    }
  }

  /**
   * Helper class for viewing a mutable indexable as a standard list.
   * The main purpose of this class is providing the RandomAccess modifier.
   */
  class ListView<T> extends AbstractList<T> implements RandomAccess
  {
    @NotNull
    private final MutableIndexable<T> indexable;

    public ListView(@NotNull MutableIndexable<T> indexable)
    {
      this.indexable = indexable;
    }

    public T get(int index)
    {
      return indexable.get(index);
    }

    @Override
    public int size()
    {
      return indexable.size();
    }

    @Override
    public T set(int index, T element)
    {
      indexable.set(index, element);
      return element;
    }

    @Override
    public boolean isEmpty()
    {
      return indexable.isEmpty();
    }

    @Override
    public Iterator<T> iterator()
    {
      return indexable.iterator();
    }
  }

  /**
   * Get a flat copy of this mutable indexable.
   * This will not copy elements, if this is required use {@link #getCopy(Function)}
   * which allows to define how elements are copied.
   * @return a copy of this mutable indexable
   */
  @NotNull
  default MutableIndexable<T> getCopy()
  {
    return getCopy(null);
  }

  /**
   * Get a copy of this object, allowing to define a copy method for the elements.
   * @param elementCloner function for cloning each element before adding it to the copy,
   *                      if {@code null} elements are copied directly, which will couple
   *                      this mutable indexable and the returned copy if the elemtns are mutable:
   *                      when an element is changed this is reflected in both.
   * @return copy of this mutable indexable
   */
  @NotNull
  @SuppressWarnings("unchecked")  // evil cast is okay because only elements of T will be used in the array
  default MutableIndexable<T> getCopy(@Nullable Function<? super T, ? extends T> elementCloner)
  {
    final T[] array = (T[])toArray();
    if (elementCloner != null) {
      for (int i = array.length - 1; i >= 0; --i) {
        array[i] = elementCloner.apply(array[i]);
      }
    }
    return MutableIndexable.viewArray(array);
  }

  /**
   * Initialize a mutable indexable to a given size.
   * @param size     size of indexable
   * @param creator  creator for the initial elements, called in order for each element
   * @param <E> element type
   * @return mutable indexable with the given size
   */
  @NotNull
  @SuppressWarnings("unchecked") // okay because array restrictions are kept
  static <E> MutableIndexable<E> init(int size, @NotNull Supplier<E> creator)
  {
    final E[] array = (E[])new Object[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = creator.get();
    }
    return viewArray(array);
  }

  /**
   * Initialize a mutable indexable to a given size while allowing exceptions.
   * @param size     size of indexable
   * @param creator  creator for the initial elements, called in order for each element, may throw an exception
   * @param <E> element type
   * @param <X> exception type
   * @return mutable indexable with the given size
   * @throws X if {@code creator} throws
   */
  @NotNull
  @SuppressWarnings("unchecked") // okay because array restrictions are kept
  static <E, X extends Exception> MutableIndexable<E> initFragile(int size,
                                                                  @NotNull FragileFunction0<E, X> creator)
          throws X
  {
    final Object[] array = new Object[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = creator.apply();
    }
    return (MutableIndexable<E>)viewArray(array);
  }

  /**
   * Create a mutable indexable of a given size, and initialize its elements by index.
   * @param size     size of returned indexable
   * @param producer element producer which will be called for each index
   * @param <E> element type
   * @return mutable indexable with the given size filled from the provider
   */
  @NotNull
  @SuppressWarnings("unchecked") // array is hidden, and able to contain any element
  static <E> MutableIndexable<E> initByIndex(int size, @NotNull IntFunction<? extends E> producer)
  {
    final Object[] array = new Object[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = producer.apply(i);
    }
    return (MutableIndexable<E>)fromArray(array);
  }

  /**
   * Initialize a mutable indexable from a provider.
   * @param size     size of returned indexable
   * @param provider provider called for each index to set the element at that index
   * @param <E> element type
   * @return mutable indexable with the given size filled from the provider
   * @deprecated inconsistently named, use {@link #initByIndex(int, IntFunction)} instead
   */
  @NotNull
  @Deprecated
  static <E> MutableIndexable<E> filled(int size, @NotNull IntFunction<? extends E> provider)
  {
    return fromIndexable(new Indexable.Base<E>()
    {
      @Override
      public E get(int index)
      {
        return provider.apply(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    });
  }

  /**
   * Get a mutable indexable of a given size filled with {@code null} values.
   * @param size size of the returned indexable
   * @param <E> element type of the returned indexable
   * @return mutable indexable of the given size, where all elements are {@code null}
   */
  @NotNull
  static <E> MutableIndexable<E> nulled(int size)
  {
    if (size == 0) {
      return empty();
    }
    return init(size, Function0.alwaysNull());
  }

  /**
   * Create a mutable indexable which is intialized from a
   * given collection.
   * @param collection collection
   * @param <E> list and result element type
   * @return mutable indexable
   */
  @NotNull
  static <E> MutableIndexable<E> copyOf(@NotNull Collection<E> collection)
  {
    return viewList(new ArrayList<>(collection));
  }

  /**
   * Create a mutable indexable which is initialized from copied elements of
   * a given collection.
   * @param collection collection
   * @param copier     element copier
   * @param <E>        result element type
   * @param <IN>       incoming element type
   * @return mutable indexable
   */
  @NotNull
  static <E, IN> MutableIndexable<E> copy(@NotNull Collection<IN> collection,
                                          @NotNull Function<IN, E> copier)
  {
    return viewList(Types.map(collection, copier));
  }

  /**
   * Get an empty mutable indexable.
   * @param <E> element type
   * @return mutable indexable with no elements, always the same object
   */
  @NotNull
  @SuppressWarnings("unchecked") // beause result is immutable
  static <E> MutableIndexable<E> empty()
  {
    return (MutableIndexable<E>)EMPTY;
  }

  /**
   * Create a mutable indexable which is initialized from elements
   * of the given array.
   * @param elements  elements
   * @param <E> element type
   * @return mutable indexable independent of {@code elements}
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  @NotNull
  static <E> MutableIndexable<E> fromArray(@NotNull E... elements)
  {
    return fromArray(elements, 0, elements.length);
  }

  /**
   * Create a mutable indexable which is initialized from elements
   * of the given array.
   * @param elements    array of elements
   * @param startIndex  index of first element used in the returned indxable
   * @param length      length number of elements used in the returned indexable
   * @param <E>         element type
   * @return mutable indexable independnet of {@code eleements}
   */
  @NotNull
  static <E> MutableIndexable<E> fromArray(@NotNull E[] elements,
                                           int startIndex, int length)
  {
    if (startIndex < 0  ||  startIndex + length > elements.length)  {
      throw new IllegalArgumentException("Indexes out of bounds!");
    }
    return viewArray(Arrays.copyOfRange(elements, startIndex, startIndex + length));
  }

  /**
   * Mutable indexable view which operates on the given array.
   * Any changes done by the returned indexable will be forwarded to the array.
   * @param array array
   * @param <E> element type
   * @return mutable indexable view of the given array
   */
  @NotNull
  @SafeVarargs
  static <E> MutableIndexable<E> viewArray(@NotNull E... array)
  {
    return new Base<E>()
    {
      @Override
      public void set(int index, E elem)
      {
        array[index] = elem;
      }

      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public E get(int index)
      {
        return array[index];
      }

      @NotNull
      @Override
      public MutableIndexable<E> subSet(int fromIndex, int toIndex)
      {
        return MutableIndexable.viewArray(array, fromIndex, toIndex - fromIndex);
      }

      @Override
      public void copyInternally(int fromIndex, int toIndex, int numElements)
      {
        System.arraycopy(array, fromIndex, array, toIndex, numElements);
      }
    };
  }

  /**
   * View a slice of an array.
   * @param array array to view
   * @param start start of array slice represented by the returned indexable
   * @param len   length of array slice represented by the returned indexable
   * @return indexable view of the given array slice, basically the same
   *         as {@code MutableIndexable.viewArray(array).subset(start, start + len)}
   * @param <E> array type
   */
  @NotNull
  static <E> MutableIndexable<E> viewArray(@NotNull E[] array,
                                           int start,
                                           int len)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start has to be non-negative, but is "+start);
    }
    if (len < 0) {
      throw new IllegalArgumentException("len has to be non-negative, but is "+len);
    }
    if (start + len > array.length) {
      throw new IndexOutOfBoundsException(String.format("end will be outside array: %d + %d > %d",
                                                        start, len, array.length));
    }
    if (len == 0) {
      return empty();
    }
    return new MutableIndexable.Base<E>()
    {
      private void checkIndex(int index)
      {
        if (index < 0  ||  index >= len) {
          throw new IndexOutOfBoundsException("Index: "+index);
        }
      }

      @Override
      public void set(int index, E elem)
      {
        checkIndex(index);
        array[start + index] = elem;
      }

      @Override
      public E get(int index)
      {
        checkIndex(index);
        return array[start + index];
      }

      @Override
      public int size()
      {
        return len;
      }

      @NotNull
      @Override
      public MutableIndexable<E> subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > len) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return MutableIndexable.viewArray(array,
                                          fromIndex + start,
                                          toIndex - fromIndex);
      }

      @Override
      public void copyInternally(int fromIndex, int toIndex, int numElements)
      {
        if (numElements == 0) {
          return;
        }
        if (numElements < 0) {
          throw new IllegalArgumentException("numElements must not be negative!");
        }
        final int from = Pythonesque.mapX(fromIndex, len);
        final int to   = Pythonesque.mapX(toIndex, len);
        if (from + numElements > len) {
          throw new IndexOutOfBoundsException(String.format("Reading would overflow: from + numElements > size (%d + %d > %d)!",
                                                            from, numElements, len));
        }
        if (to + numElements > len) {
          throw new IndexOutOfBoundsException(String.format("Writing would overflow: to + numElements > size (%d + %d > %d)!",
                                                            to, numElements, len));
        }
        if (from == to) {
          return;
        }
        System.arraycopy(array, fromIndex + start, array, toIndex + start, numElements);
      }
    };
  }

  /**
   * Mutable indexable view which operates on the given array.
   * Any changes done by the returned indexable will be forwarded to the list.
   * @param list list
   * @param <E> element type
   * @return mutable indexable view of the given list
   */
  @NotNull
  static <E> MutableIndexable<E> viewList(@NotNull List<E> list)
  {
    return new Base<E>()
    {
      @Override
      public void set(int index, E elem)
      {
        list.set(index, elem);
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public E get(int index)
      {
        return list.get(index);
      }
    };
  }

  /**
   * Create a mutable indexable which is the copy of a standard indexable.
   * @param indexable standard indexable
   * @param <E> element type
   * @return mutable indexable initialize from the standard indexable
   */
  @NotNull
  static <E> MutableIndexable<E> fromIndexable(@NotNull Indexable<E> indexable)
  {
    return copyOf(indexable.asCollection());
  }

  /**
   * Create a mutable indexable which is the copy of a standard indexable.
   * This especieally allows to copy each element during the initialization.
   * @param indexable  base indexable
   * @param converter  converter from base indexable type to result element type,
   * @param <E> element type of returned mutable indexable
   * @param <B> element type of incoming idexable
   * @return mutable indexable
   */
  @NotNull
  static <E, B> MutableIndexable<E> fromIndexable(@NotNull Indexable<B> indexable,
                                                  @NotNull Function<B, E> converter)
  {
    return viewList(Types.map(indexable.asList(), converter));
  }

  /**
   * View a generic indexable as a mutable indexable by accessing
   * a property of the elements of the underlying generic indexable.
   * <p>
   * This is useful if you have complex items, but are interested
   * into only one property of each item.
   * @param indexable underlying generic indexable
   * @param getter    function used to extract the property of interest
   * @param setter    procedure used to set the property of interest
   * @param <IT>      element type of the underlying indexable
   * @param <OT>      element type of the returned indexable
   * @return mutable indexable view of the underlying indexable
   */
  @NotNull
  static <IT, OT> MutableIndexable.Base<OT> viewIndexable(@NotNull Indexable<IT> indexable,
                                                          @NotNull Function<? super IT, OT> getter,
                                                          @NotNull BiConsumer<? super IT, ? super OT> setter)
  {
    return new MutableIndexable.Base<OT>()
    {
      @Override
      public void set(int index, OT value)
      {
        setter.accept(indexable.get(index), value);
      }

      @Override
      public OT get(int index)
      {
        return getter.apply(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }
    };
  }

  /**
   * Create a mutable indexable from an iterable or a part of it.
   * @param size maximum size of returned indexable
   * @param iter iterable used for initializing the indexable
   * @param <E> element type
   * @return mutable indexable
   */
  @NotNull
  static <E> MutableIndexable<E> fromIterable(int size, @NotNull Iterable<E> iter)
  {
    final ArrayList<E> list = new ArrayList<>(size);
    int i = 0;
    for (E elem : iter) {
      list.add(elem);
      if (++i == size) {
        break;
      }
    }
    list.trimToSize();
    return viewList(list);
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   */
  abstract class Base<TT>
          extends Indexable.Base<TT>
          implements MutableIndexable<TT>
  {
  }

  /**
   * Empty mutable indexable.
   * Use {@link #empty()} instead.
   */
  MutableIndexable<Object> EMPTY = new MutableIndexable.Base<Object>()
  {
    @Override
    public void set(int index, Object elem)
    {
      throw new IndexOutOfBoundsException("Cannot set in empty indexable!");
    }

    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public Object get(int index)
    {
      throw new IndexOutOfBoundsException("Cannot get from empty indexable!");
    }

    @NotNull
    @Override
    public ListIterator<Object> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public MutableIndexable<Object> reverse()
    {
      return this;
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public Iterator<Object> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public Collection<Object> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Object> asList()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return Types.emptyIterable();
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public Indexable<Object> frozen()
    {
      return Indexable.emptyIndexable();
    }

    @NotNull
    @Override
    public Indexable<Object> frozen(@NotNull Function<? super Object, ?> elementCloner, boolean cloneOnGet)
    {
      return Indexable.emptyIndexable();
    }

    @NotNull
    @Override
    public MutableIndexable<Object> getCopy(@Nullable Function<? super Object, ?> elementCloner)
    {
      return this;
    }

    @NotNull
    @Override
    public Indexable<Object> sorted(@NotNull Comparator<? super Object> order)
    {
      return this;
    }

    @NotNull
    @Override
    public Object[] toArray()
    {
      return Empty.OBJECT_ARRAY;
    }

    @NotNull
    @Override
    public Object[] toArray(@NotNull Class<Object> type)
    {
      return (Object[])Array.newInstance(type, 0);
    }

    @Override
    public int addToArray(@NotNull Object[] array, int arrayPos, int index, int length)
    {
      return arrayPos;
    }

    @Override
    public int addToArray(@NotNull Object[] array, int arrayPos)
    {
      return arrayPos;
    }

    @Override
    public void forEachEntry(@NotNull BiConsumer<Integer, ? super Object> action)
    {
    }

    @Override
    public <E extends Exception> void forEachEntryFragile(@NotNull FragileProcedure2<E, Integer, ? super Object> action)
            throws E
    {
    }

    @Override
    public void forEach(Consumer<? super Object> action)
    {
    }

    @Override
    public void addAllTo(@NotNull Collection<? super Object> collection)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> filtered(@NotNull Predicate<? super Object> filter)
    {
      return Types.emptyIterable();
    }

    @Override
    public <E extends Exception> void forEachFragile(@NotNull FragileProcedure1<E, ? super Object> consumer) throws E
    {
    }

    @Override
    public <V> V foldLeft(V initialValue, @NotNull BiFunction<? super V, ? super Object, ? extends V> folder)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public <E> Indexable<E> view(@NotNull Function<? super Object, ? extends E> mapper)
    {
      return Indexable.emptyIndexable();
    }

    @NotNull
    @Override
    public <R, E extends Exception> Indexable<R> viewFragile(
            @NotNull FragileFunction1<? extends R, E, ? super Object> mapper)
    {
      return Indexable.emptyIndexable();
    }

    @Override
    @Deprecated
    public void addToCollection(@NotNull Collection<? super Object> collection)
    {
    }

    @NotNull
    @Override
    public Indexable<Object> findAll(@NotNull Predicate<? super Object> filter)
    {
      return Indexable.emptyIndexable();
    }

    @Override
    public int setFrom(@NotNull Iterable<?> iterable)
    {
      return 0;
    }

    @Override
    public void fillFrom(@NotNull IntFunction<?> provider)
    {
    }
  };
}
