// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.algorithm.DualPivotQuicksort;
import de.caff.generics.algorithm.TimSortFloat;
import de.caff.generics.function.*;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Primitive float indexable with fixed length but mutable content.
 * This may be used as a substitute for {@code float[]} as it eg allows
 * read-only views or transparent views of subsets of the array.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public interface MutableFloatIndexable
        extends FloatIndexable,
                Copyable<MutableFloatIndexable>
{
  /**
   * Set the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @param value value to put to the given index
   */
  void set(int index, float value);

  /**
   * Pythonesque set.
   * This allows to access elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @param value value to put to the given index
   */
  default void syt(int index, float value)
  {
    if (index >= 0) {
      set(index, value);
    }
    final int realIndex = index + size();
    if (realIndex < 0) {
      throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                         index, size()));
    }
    set(realIndex, value);
  }

  /**
   * Set the values of this mutable float indexable one after
   * the other from the given Number iterable until either the
   * iterable is emptied or {@link #size()} elements are set.
   * @param iterable iterable from which this indexable is filled
   * @return number of elements set
   */
  default int setFrom(@NotNull Iterable<? extends Number> iterable)
  {
    return setFrom(iterable, 0, size());
  }

  /**
   * Set the values of this mutable float indexable one after
   * the other from the given iterable until either the
   * iterable is emptied or {@code numElements} elements are set.
   * @param iterable     iterable from which this indexable is filled
   * @param startIndex   start index where the setting begins
   * @param numElements  number of elements to set
   * @return number of elements set
   */
  default int setFrom(@NotNull Iterable<? extends Number> iterable,
                      int startIndex, int numElements)
  {
    if (startIndex > size() - numElements) {
      numElements = size() - startIndex;
    }

    int n = 0;

    for (Number elem : iterable) {
      set(startIndex + n++, elem.floatValue());
      if (n == numElements) {
        break;
      }
    }
    return n;
  }

  /**
   * Set consecutive entries from an array.
   * This will set less elements
   * @param array       array with elements to set from
   * @param arrayIndex  start index in the array
   * @param startIndex  start index in this mutable indexable
   * @param numElements number of elements to copy
   * @throws IndexOutOfBoundsException if this mutable indexable would overflow by this operation
   * @throws ArrayIndexOutOfBoundsException if array overflows by this operation
   */
  default void setFromArray(@NotNull float[] array,
                            int arrayIndex,
                            int startIndex,
                            int numElements)
  {
    if (startIndex + numElements > size()) {
      throw new IndexOutOfBoundsException("Overflow: numElements is too large!");
    }

    for (int n = 0;  n < numElements;  ++n) {
      set(startIndex + n, array[arrayIndex++]);
    }
  }

  /**
   * Set multiple elements to the same value.
   * @param from  first index to be set (Pythonesque)
   * @param len   number of elements to be set (non-negative)
   * @param value value to be set
   */
  default void setMulti(int from, int len, float value)
  {
    from = Pythonesque.mapX(from, this);
    if (len < 0) {
      throw new IllegalArgumentException("len has to be non-negative: "+len);
    }
    final int end = from + len;
    if (end > size()) {
      throw new IndexOutOfBoundsException(String.format("from + len exceed size(): %d + %d > %d", from, len, size()));
    }
    for (int i = from;  i < end;  ++i) {
      set(i, value);
    }
  }

  /**
   * Copy elements inside this indexable.
   * @param fromIndex   start index from where elements are copied (Pythonesque)
   * @param toIndex     start index to where elements are copied (Pythonesque)
   * @param numElements number of elements to copy
   */
  default void copyInternally(int fromIndex, int toIndex, int numElements)
  {
    if (numElements == 0) {
      return;
    }
    if (numElements < 0) {
      throw new IllegalArgumentException("numElements must not be negative!");
    }
    final int size = size();
    int from = Pythonesque.mapX(fromIndex, size);
    int to   = Pythonesque.mapX(toIndex, size);
    if (from + numElements > size) {
      throw new IndexOutOfBoundsException(String.format("Reading would overflow: from + numElements > size (%d + %d > %d)!",
                                                        from, numElements, size));
    }
    if (to + numElements > size) {
      throw new IndexOutOfBoundsException(String.format("Writing would overflow: to + numElements > size (%d + %d > %d)!",
                                                        to, numElements, size));
    }
    if (from == to) {
      return;
    }
    if (from < to  &&  from + numElements > to) {
      // have to go backward
      from += numElements;
      to   += numElements;
      while (--numElements >= 0) {
        set(--to, get(--from));
      }
    }
    else {
      while (--numElements >= 0) {
        set(to++, get(from++));
      }
    }
  }

  /**
   * Fill this indexable from a provider.
   * @param provider provider which is called for each index and provides the element for that index
   */
  default void fillFrom(@NotNull IntToFloatFunction1 provider)
  {
    final int size = size();
    for (int p = 0;  p < size;  ++p) {
      set(p, provider.applyAsFloat(p));
    }
  }

  /**
   * Swap the values at two indices.
   * @param idx1 first index
   * @param idx2 second index
   */
  default void swap(int idx1, int idx2)
  {
    if (idx1 == idx2) {
      return;
    }
    final float tmp = get(idx1);
    set(idx1, get(idx2));
    set(idx2, tmp);
  }

  /**
   * Swap the values at two indices using Pythonesque indices.
   * @param idx1 first index ({@link Pythonesque})
   * @param idx2 second index ({@link Pythonesque})
   */
  default void swyp(int idx1, int idx2)
  {
    if (idx1 == idx2) {
      return;
    }
    swap(Pythonesque.mapX(idx1, this),
         Pythonesque.mapX(idx2, this));
  }

  @NotNull
  @Override
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0) {
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    }
    if (toIndex > size()) {
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    }
    if (fromIndex > toIndex) {
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    }
    final int length = toIndex - fromIndex;

    return new Base()
    {
      @Override
      public void set(int index, float elem)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("index = "+index);
        }
        MutableFloatIndexable.this.set(index + fromIndex,
                                       elem);
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public float get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("index = "+index);
        }
        return MutableFloatIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public MutableFloatIndexable.Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > size()) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return MutableFloatIndexable.this.subSet(fromIndex + fromIdx,
                                                  fromIndex + toIdx - fromIdx);
      }

      @Override
      public void copyInternally(int fromIdx, int toIdx, int numElements)
      {
        final int from = Pythonesque.mapX(fromIndex, length);
        final int to   = Pythonesque.mapX(toIndex, length);
        if (from + numElements > length) {
          throw new IndexOutOfBoundsException(String.format("Reading would overflow: from + numElements > size (%d + %d > %d)!",
                                                            from, numElements, length));
        }
        if (to + numElements > length) {
          throw new IndexOutOfBoundsException(String.format("Writing would overflow: to + numElements > size (%d + %d > %d)!",
                                                            to, numElements, length));
        }
        // now forward to wrapped
        MutableFloatIndexable.this.copyInternally(fromIndex + fromIdx,
                                                  fromIndex + toIdx,
                                                  numElements);
      }
    };
  }

  @NotNull
  @Override
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(fromIndex < 0
                          ? size() + fromIndex
                          : fromIndex,
                  toIndex < 0
                          ? size() + toIndex
                          : toIndex);
  }

  @NotNull
  @Override
  default Base tailSet(int fromIndex)
  {
    return subSet(fromIndex < 0
                          ? size() + fromIndex
                          : fromIndex,
                  size());
  }

  @NotNull
  @Override
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  toIndex < 0
                          ? size() + toIndex
                          : toIndex);
  }

  @NotNull
  @Override
  default Base reverse()
  {
    return new MutableFloatIndexable.Base()
    {
      @Override
      public void set(int index, float elem)
      {
        MutableFloatIndexable.this.set(size() - index - 1, elem);
      }

      @Override
      public int size()
      {
        return MutableFloatIndexable.this.size();
      }

      @Override
      public float get(int index)
      {
        return MutableFloatIndexable.this.get(size() - index - 1);
      }

      @NotNull
      @Override
      public MutableFloatIndexable.Base reverse()
      {
        return based(MutableFloatIndexable.this);
      }
    };
  }

  /**
   * Revert the order of all elements in the given range.
   * Attention: the {@code to} element is also included in the reversion to allow easy
   * reversion at the end by using {@code -1}.
   * @param from start element of reversion (Pythonesque), included
   * @param to   end element of reversion (Pythonesque), included
   */
  default void revert(int from, int to)
  {
    final int size = size();
    int up = Pythonesque.map(from, size);
    int down = Pythonesque.map(to, size);

    while (up < down) {
      swap(up++, down--);
    }
  }

  /**
   * Revert the elements in this indexable.
   * This reverts the order of all elements.
   */
  default void revert()
  {
    if (size() <= 1) {
      return;
    }
    revert(0, -1);
  }

  /**
   * Order the elements in this indexable by the given ordering.
   * After this call the elements of this indexable are reordered
   * so that {@link #isOrdered(FloatOrdering)} returns {@code true}
   * for the given {@code ordering}.
   * <p>
   * If natural ordering is required use {@link #order()}
   * because in many cases this can avoid the overhead of an ordering
   * and sort much faster (typically 1.5x as fast).
   * @param ordering ordering to apply
   */
  default void order(@NotNull FloatOrdering ordering)
  {
    TimSortFloat.sort(this, ordering);
  }

  /**
   * Order the elements in this indexable in their standard Java order.
   * This method is typically 1.5 times faster than
   * {@link #order(FloatOrdering)} with {@link FloatOrdering#STANDARD_ASCENDING natural order}.
   */
  default void order()
  {
    order(FloatOrdering.NATURAL_ASCENDING);
  }

  /**
   * Randomize the content  of this mutable indexable.
   * This will reorder the elements randomly.
   * @param random random number generator
   */
  default void shuffle(@NotNull Random random)
  {
    final int size = size();
    for (int i = size - 1;  i >= 0;  --i) {
      swap(i, random.nextInt(i + 1));
    }
  }

  /**
   * Set all values of this float indexable by index.
   * @param setter setter which provides the value to be set for a given index
   * @deprecated use {@link #fillFrom(IntToFloatFunction1)} instead
   */
  @Deprecated
  default void initByIndex(@NotNull IntToFloatFunction1 setter)
  {
    final int size = size();
    for (int i = 0; i < size; ++i) {
      set(i, setter.applyAsFloat(i));
    }
  }

  /**
   * View this indexable as a standard list.
   * Overwritten to allow for modifications.
   * The returned list will allow usage of the {@link List#set(int, Object)} method,
   * but neither any adding nor deleting methods. Standard sorting algorithms will work
   * on the returned list as a sorting algorithm is expected to neither move nor add
   * elements.
   * @return list allowing setting elements
   */
  @NotNull
  @Override
  default List<Float> asList()
  {
    return new AbstractList<Float>()
    {
      @Override
      public Float get(int index)
      {
        return MutableFloatIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return MutableFloatIndexable.this.size();
      }

      @Override
      public Float set(int index, Float element)
      {
        MutableFloatIndexable.this.set(index, element);
        return element;
      }

      @Override
      public Iterator<Float> iterator()
      {
        return MutableFloatIndexable.this.iterator();
      }
    };
  }

  @NotNull
  @Override
  default MutableFloatIndexable getCopy()
  {
    return viewArray(toArray());
  }

  /**
   * View any mutable float indexable as a base float indexable.
   * The latter provides various standard methods for comparison, output and hanshing.
   * @param indexable mutqable float indexable to view as a {@link Base} mutable float indexable
   * @return base view of given indexable, or the indexable itself if it is already a Base
   */
  @NotNull
  static Base based(@NotNull MutableFloatIndexable indexable)
  {
    if (indexable instanceof Base) {
      return (Base)indexable;
    }
    return new Base()
    {
      @Override
      public void set(int index, float value)
      {
        indexable.set(index, value);
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public float get(int index)
      {
        return indexable.get(index);
      }
    };
  }

  /**
   * Initialize a mutable indexable to a given size.
   * @param size     size of indexable
   * @param creator  creator for the initial elements
   * @return mutable indexable with the given size
   * @see #zeroed(int)
   * @see #init(int, float)
   */
  @NotNull
  static Base init(int size, @NotNull Supplier<? extends Number> creator)
  {
    final float[] array = new float[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = creator.get().floatValue();
    }
    return viewArray(array);
  }

  /**
   * Create a mutable indexable which is initialized from a
   * given collection.
   * @param collection collection
   * @return mutable indexable
   */
  @NotNull
  static Base copyOf(@NotNull Collection<? extends Number> collection)
  {
    final float[] array = new float[collection.size()];
    int i = 0;
    for (Number n : collection) {
      array[i++] = n.floatValue();
    }
    return viewArray(array);
  }

  /**
   * Create a mutable indexable which is initialized from copied elements of
   * a given collection.
   * @param collection collection
   * @param copier     element copier
   * @param <IN>       incoming element type
   * @return mutable indexable
   */
  @NotNull
  static <IN> Base copy(@NotNull Collection<IN> collection,
                        @NotNull Function<IN, Number> copier)
  {
    final float[] array = new float[collection.size()];
    int i = 0;
    for (IN elem : collection) {
      array[i++] = copier.apply(elem).floatValue();
    }
    return viewArray(array);
  }

  @NotNull
  static Base empty()
  {
    return EMPTY;
  }

  /**
   * Create a mutable indexable which is initialized from elements
   * of the given array.
   * @param elements  elements
   * @return mutable indexable independent of {@code elements}
   */
  @NotNull
  static Base fromArray(@NotNull float... elements)
  {
    return fromArray(elements, 0, elements.length);
  }

  /**
   * Create a mutable indexable which is initialized from elements
   * of the given array.
   * @param elements    array of elements
   * @param startIndex  index of first element used in the returned indxable
   * @param length      length number of elements used in the returned indexable
   * @return mutable indexable independent of {@code elements}
   */
  @NotNull
  static Base fromArray(@NotNull float[] elements,
                        int startIndex, int length)
  {
    if (startIndex < 0  ||  startIndex + length > elements.length)  {
      throw new IllegalArgumentException("Indexes out of bounds!");
    }
    return viewArray(Arrays.copyOfRange(elements, startIndex, startIndex + length));
  }

  /**
   * Mutable indexable view which operates on the given array.
   * Any changes done by the returned indexable will be forwarded to the array.
   * @param array array
   * @return mutable indexable view of the given array
   */
  @NotNull
  static Base viewArray(@NotNull float[] array)
  {
    return new Base()
    {
      @Override
      public void set(int index, float elem)
      {
        array[index] = elem;
      }

      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public float get(int index)
      {
        return array[index];
      }

      @NotNull
      @Override
      public MutableFloatIndexable.Base subSet(int fromIndex, int toIndex)
      {
        return MutableFloatIndexable.viewArray(array, fromIndex, toIndex - fromIndex);
      }

      @Override
      public void copyInternally(int fromIndex, int toIndex, int numElements)
      {
        System.arraycopy(array, fromIndex, array, toIndex, numElements);
      }

      @Override
      public void order()
      {
        Arrays.sort(array);
      }

      @Override
      public void order(@NotNull FloatOrdering ordering)
      {
        DualPivotQuicksort.sort(array, ordering);
      }
    };
  }

  /**
   * Mutable indexable view which operates on a part of the given array.
   * @param array  array
   * @param start  start position in array
   * @param length length of returned indexable
   * @return mutable float indexable view of the given part of the array
   */
  @NotNull
  static MutableFloatIndexable.Base viewArray(@NotNull float[] array, int start, int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start has to be non-negative, but is "+start);
    }
    if (length < 0) {
      throw new IllegalArgumentException("len has to be non-negative, but is "+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException(String.format("end will be outside array: %d + %d > %d",
                                                        start, length, array.length));
    }
    if (length == 0) {
      return empty();
    }
    return new MutableFloatIndexable.Base()
    {
      @Override
      public void set(int index, float value)
      {
        array[index + start] = value;
      }

      @Override
      public float get(int index)
      {
        return array[index + start];
      }

      @Override
      public int size()
      {
        return length;
      }

      @NotNull
      @Override
      public MutableFloatIndexable.Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return MutableFloatIndexable.viewArray(array,
                                                fromIndex + start,
                                                toIndex - fromIndex);
      }

      @Override
      public void copyInternally(int fromIndex, int toIndex, int numElements)
      {
        if (numElements == 0) {
          return;
        }
        if (numElements < 0) {
          throw new IllegalArgumentException("numElements must not be negative!");
        }
        final int from = Pythonesque.mapX(fromIndex, length);
        final int to   = Pythonesque.mapX(toIndex, length);
        if (from + numElements > length) {
          throw new IndexOutOfBoundsException(String.format("Reading would overflow: from + numElements > size (%d + %d > %d)!",
                                                            from, numElements, length));
        }
        if (to + numElements > length) {
          throw new IndexOutOfBoundsException(String.format("Writing would overflow: to + numElements > size (%d + %d > %d)!",
                                                            to, numElements, length));
        }
        if (from == to) {
          return;
        }
        System.arraycopy(array, fromIndex + start, array, toIndex + start, numElements);
      }

      @Override
      public void order()
      {
        Arrays.sort(array, start, start + length);
      }

      @Override
      public void order(@NotNull FloatOrdering ordering)
      {
        DualPivotQuicksort.sort(array, start, start + length - 1, ordering);
      }
    };
  }

  /**
   * Mutable indexable view which operates on the given array.
   * Any changes done by the returned indexable will be forwarded to the list.
   * @param list list
   * @return mutable indexable view of the given list
   */
  @NotNull
  static Base viewList(@NotNull List<Float> list)
  {
    return new Base()
    {
      @Override
      public void set(int index, float elem)
      {
        list.set(index, elem);
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public float get(int index)
      {
        return list.get(index);
      }
    };
  }

  /**
   * Create a mutable indexable which is the copy of a standard indexable.
   * @param indexable standard indexable
   * @return mutable indexable initialize from the standard indexable
   */
  @NotNull
  static Base fromIndexable(@NotNull Indexable<? extends Number> indexable)
  {
    return copyOf(indexable.asCollection());
  }

  /**
   * Create a mutable indexable which is the copy of a standard indexable.
   * This especieally allows to copy each element during the initialization.
   * @param indexable  base indexable
   * @param converter  converter from base indexable type to result element type,
   * @param <B> element type of incoming idexable
   * @return mutable indexable
   */
  @NotNull
  static <B> Base fromIndexable(@NotNull Indexable<B> indexable,
                                @NotNull Function<? super B, ? extends Number> converter)
  {
    final float[] array = new float[indexable.size()];
    int i = 0;
    for (B elem : indexable) {
      array[i++] = converter.apply(elem).floatValue();
    }
    return viewArray(array);
  }

  /**
   * Return a mutable float indexable initialized with the content of a
   * standard double indexable. This will cast the incoming double
   * values and result in a loss of accuracy.
   * @param indexable double indexable
   * @return mutable indexable initialized with the values of the given double indexable
   */
  @NotNull
  static Base fromDoubleIndexable(@NotNull DoubleIndexable indexable)
  {
    return initByIndex(indexable.size(), i -> (float)indexable.get(i));
  }

  /**
   * Return a mutable float indexable initialized with the content of a
   * standard float indexable.
   * @param indexable float indexable
   * @return mutable indexable initialized with the values of the given float indexable
   */
  @NotNull
  static Base fromFloatIndexable(@NotNull FloatIndexable indexable)
  {
    return initByIndex(indexable.size(), indexable::get);
  }

  /**
   * Return a mutable float indexable initialized with the content of a
   * standard long indexable. This will result in a loss of accuracy for
   * many values.
   * @param indexable long indexable
   * @return mutable indexable initialized with the values of the given long indexable
   */
  @NotNull
  static Base fromLongIndexable(@NotNull LongIndexable indexable)
  {
    return initByIndex(indexable.size(), indexable::get);
  }

  /**
   * Return a mutable float indexable initialized with the content of a
   * standard int indexable. This will result in a loss of accuracy for
   * various values.
   * @param indexable int indexable
   * @return mutable indexable initialized with the values of the given int indexable
   */
  @NotNull
  static Base fromIntIndexable(@NotNull IntIndexable indexable)
  {
    return initByIndex(indexable.size(), indexable::get);
  }

  /**
   * Return a mutable long indexable initialized with the content of a
   * standard short indexable.
   * @param indexable short indexable
   * @return mutable indexable initialized with the values of the given short indexable
   */
  @NotNull
  static Base fromShortIndexable(@NotNull ShortIndexable indexable)
  {
    return initByIndex(indexable.size(), indexable::get);
  }

  /**
   * Return a mutable double indexable initialized with the content of a
   * standard byte indexable.
   * @param indexable long indexable
   * @return mutable indexable initialized with the values of the given byte indexable
   */
  @NotNull
  static Base fromByteIndexable(@NotNull ByteIndexable indexable)
  {
    return initByIndex(indexable.size(), indexable::get);
  }

  /**
   * Create a mutable indexable from an iterable or a part of it.
   * @param size maximum size of returned indexable
   * @param iter iterable used for initializing the indexable
   * @return mutable indexable
   */
  @NotNull
  static Base fromIterable(int size, @NotNull Iterable<? extends Number> iter)
  {
    final float[] array = new float[size];
    int i = 0;
    for (Number elem : iter) {
      array[i++] = elem.floatValue();
      if (i == size) {
        break;
      }
    }
    return viewArray(array);
  }

  /**
   * View a generic indexable as a mutable float indexable by accessing
   * a float property of the elements of the underlying generic indexable.
   * <p>
   * This is useful if you have complex items, but are interested
   * into only one (float) property of each item.
   * @param indexable underlying generic indexable
   * @param getter    function used to extract the property of interest
   * @param setter    procedure used to set the property of interest
   * @param <T>       element type of the underlying indexable
   * @return mutable float indexable view of the underlying indexable
   */
  @NotNull
  static <T> MutableFloatIndexable.Base viewIndexable(@NotNull Indexable<T> indexable,
                                                      @NotNull ToFloatFunction<? super T> getter,
                                                      @NotNull FloatSetter<? super T> setter)
  {
    return new MutableFloatIndexable.Base()
    {
      @Override
      public void set(int index, float value)
      {
        setter.set(indexable.get(index), value);
      }

      @Override
      public float get(int index)
      {
        return getter.applyAsFloat(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }
    };
  }

  /**
   * Create a mutable float indexable with {@code size}
   * {@code 0.0} elements.
   * @param size required size of returned mutable float indexable
   * @return mutable float indexable of the given size initialized with
   *         {@code 0.0}
   * @see #init(int, float)
   * @see #init(int, Supplier)
   */
  @NotNull
  static Base zeroed(int size)
  {
    return viewArray(new float[size]);
  }

  /**
   * Create a mutable
   * @param size  required size of returned mutable float indexable
   * @param value initial value of all elements
   * @return mutable float indexable of the given size initialized with
   *         {@code value}
   * @see #init(int, Supplier)
   * @see #zeroed(int)
   */
  @NotNull
  static Base init(int size, float value)
  {
    final float[] array = new float[size];
    Arrays.fill(array, value);
    return viewArray(array);
  }

  /**
   * Get a mutable float indexable of a given size which contains elements created by index.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return a mutable boolean indexable with the given size with elements initialized by calling the {@code producer}
   */
  @NotNull
  static Base initByIndex(int size, @NotNull IntToFloatFunction1 producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    final float[] array = new float[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = producer.applyAsFloat(i);
    }
    return viewArray(array);
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   * @see Indexable#asBase()
   */
  abstract class Base extends FloatIndexable.Base implements MutableFloatIndexable
  {
  }

  /**
   * Empty mutable indexable.
   * Use {@link #empty()} instead.
   */
  Base EMPTY = new MutableFloatIndexable.Base()
  {
    @Override
    public void set(int index, float value)
    {
      throw new IndexOutOfBoundsException("Cannot set in empty indexable!");
    }

    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public float get(int index)
    {
      throw new IndexOutOfBoundsException("Cannot get from empty indexable!");
    }

    @NotNull
    @Override
    public ListIterator<Float> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public MutableFloatIndexable.Base reverse()
    {
      return this;
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public Iterator<Float> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public PrimitiveFloatIterator floatIterator()
    {
      return Types.EMPTY_FLOAT_ITERATOR;
    }

    @NotNull
    @Override
    public PrimitiveIterator.OfDouble doubleIterator()
    {
      return Types.EMPTY_DOUBLE_ITERATOR;
    }

    @NotNull
    @Override
    public Collection<Float> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Float> asList()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return Types.emptyIterable();
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public float[] toArray()
    {
      return Empty.FLOAT_ARRAY;
    }

    @Override
    public int addToArray(@NotNull float[] array, int pos)
    {
      return pos;
    }

    @NotNull
    @Override
    public DoubleIndexable.Base asDoubleIndexable()
    {
      return DoubleIndexable.EMPTY;
    }

    @Override
    public float foldLeft(float initialValue, @NotNull FloatOperator2 foldOperation)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public FloatIndexable frozen()
    {
      return this;
    }

    @Override
    public void initByIndex(@NotNull IntToFloatFunction1 setter)
    {
    }

    @Override
    public void forEachFloat(@NotNull FloatConsumer action)
    {
    }

    @Override
    public <E extends Exception> void forEachFloatFragile(@NotNull FragileFloatConsumer<E> action) throws E
    {
    }

    @Override
    public boolean containsFloat(float value)
    {
      return false;
    }

    @Override
    public boolean containsFloat(float value, float eps)
    {
      return false;
    }

    @Override
    public void forEach(Consumer<? super Float> action)
    {
    }

    @NotNull
    @Override
    public MutableFloatIndexable getCopy()
    {
      return this;
    }

    @Override
    public void order(@NotNull FloatOrdering ordering)
    {
    }

    @Override
    public void order()
    {
    }

    @Override
    @NotNull
    public String toString()
    {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }
  };
}
