// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.*;

import java.util.*;
import java.util.function.*;

/**
 * Primitive boolean indexable with fixed length but mutable content.
 * This may be used as a substitute for {@code boolean[]} as it eg allows
 * read-only views or transparent views of subsets of the array.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public interface MutableBooleanIndexable
        extends BooleanIndexable,
                Copyable<MutableBooleanIndexable>
{
  /**
   * Set the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @param value value to put to the given index
   */
  void set(int index, boolean value);

  /**
   * Pythonesque set.
   * This allows to access elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @param value value to put to the given index
   */
  default void syt(int index, boolean value)
  {
    if (index >= 0) {
      set(index, value);
    }
    final int realIndex = index + size();
    if (realIndex < 0) {
      throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                         index, size()));
    }
    set(realIndex, value);
  }

  /**
   * Set the values of this mutable boolean indexable one after
   * the other from the given Number iterable until either the
   * iterable is emptied or {@link #size()} elements are set.
   * @param iterable iterable from which this indexable is filled
   * @return number of elements set
   */
  default int setFrom(@NotNull Iterable<? extends Boolean> iterable)
  {
    return setFrom(iterable, 0, size());
  }

  /**
   * Set the values of this mutable boolean indexable one after
   * the other from the given iterable until either the
   * iterable is emptied or {@code numElements} elements are set.
   * @param iterable     iterable from which this indexable is filled
   * @param startIndex   start index where the setting begins
   * @param numElements  number of elements to set
   * @return number of elements set
   */
  default int setFrom(@NotNull Iterable<? extends Boolean> iterable,
                      int startIndex, int numElements)
  {
    if (startIndex > size() - numElements) {
      numElements = size() - startIndex;
    }

    int n = 0;

    for (Boolean elem : iterable) {
      set(startIndex + n++, elem);
      if (n == numElements) {
        break;
      }
    }
    return n;
  }

  /**
   * Set multiple elements to the same value.
   * @param from  first index to be set (Pythonesque)
   * @param len   number of elements to be set (non-negative)
   * @param value value to be set
   */
  default void setMulti(int from, int len, boolean value)
  {
    from = Pythonesque.mapX(from, this);
    if (len < 0) {
      throw new IllegalArgumentException("len has to be non-negative: "+len);
    }
    final int end = from + len;
    if (end > size()) {
      throw new IndexOutOfBoundsException(String.format("from + len exceed size(): %d + %d > %d", from, len, size()));
    }
    for (int i = from;  i < end;  ++i) {
      set(i, value);
    }
  }

  /**
   * Swap the values at two indices.
   * @param idx1 first index
   * @param idx2 second index
   */
  default void swap(int idx1, int idx2)
  {
    if (idx1 == idx2) {
      return;
    }
    final boolean tmp = get(idx1);
    set(idx1, get(idx2));
    set(idx2, tmp);
  }

  /**
   * Swap the values at two indices using Pythonesque indices.
   * @param idx1 first index ({@link Pythonesque})
   * @param idx2 second index ({@link Pythonesque})
   */
  default void swyp(int idx1, int idx2)
  {
    if (idx1 == idx2) {
      return;
    }
    swap(Pythonesque.mapX(idx1, this),
         Pythonesque.mapX(idx2, this));
  }

  /**
   * Randomize the content of this mutable indexable.
   * This will reorder the elements randomly.
   * @param random random number generator
   */
  default void shuffle(@NotNull Random random)
  {
    final int size = size();
    for (int i = size - 1;  i >= 0;  --i) {
      swap(i, random.nextInt(i + 1));
    }
  }

  @NotNull
  @Override
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0) {
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    }
    if (toIndex > size()) {
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    }
    if (fromIndex > toIndex) {
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    }
    final int length = toIndex - fromIndex;

    return new Base()
    {
      @Override
      public void set(int index, boolean elem)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("index = "+index);
        }
        MutableBooleanIndexable.this.set(index + fromIndex,
                                         elem);
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("index = "+index);
        }
        return MutableBooleanIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public MutableBooleanIndexable.Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > size()) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return MutableBooleanIndexable.this.subSet(fromIndex + fromIdx,
                                                  fromIndex + toIdx - fromIdx);
      }
    };
  }

  @NotNull
  @Override
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(fromIndex < 0
                          ? size() + fromIndex
                          : fromIndex,
                  toIndex < 0
                          ? size() + toIndex
                          : toIndex);
  }

  @NotNull
  @Override
  default Base tailSet(int fromIndex)
  {
    return subSet(fromIndex < 0
                          ? size() + fromIndex
                          : fromIndex,
                  size());
  }

  @NotNull
  @Override
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  toIndex < 0
                          ? size() + toIndex
                          : toIndex);
  }

  @NotNull
  @Override
  default Base reverse()
  {
    return new MutableBooleanIndexable.Base()
    {
      @Override
      public void set(int index, boolean elem)
      {
        MutableBooleanIndexable.this.set(size() - index - 1, elem);
      }

      @Override
      public int size()
      {
        return MutableBooleanIndexable.this.size();
      }

      @Override
      public boolean get(int index)
      {
        return MutableBooleanIndexable.this.get(size() - index - 1);
      }

      @NotNull
      @Override
      public MutableBooleanIndexable.Base reverse()
      {
        return based(MutableBooleanIndexable.this);
      }
    };
  }

  /**
   * Set all values of this boolean indexable by index.
   * @param setter setter which provides the value to be set for a given index
   */
  default void initByIndex(@NotNull IntPredicate1 setter)
  {
    final int size = size();
    for (int i = 0; i < size; ++i) {
      set(i, setter.testInt(i));
    }
  }

  /**
   * View this indexable as a standard list.
   * Overwritten to allow for modifications.
   * The returned list will allow usage of the {@link List#set(int, Object)} method,
   * but neither any adding nor deleting methods. Standard sorting algorithms will work
   * on the returned list as a sorting algorithm is expected to neither move nor add
   * elements.
   * @return list allowing setting elements
   */
  @NotNull
  @Override
  default List<Boolean> asList()
  {
    return new AbstractList<Boolean>()
    {
      @Override
      public Boolean get(int index)
      {
        return MutableBooleanIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return MutableBooleanIndexable.this.size();
      }

      @Override
      public Boolean set(int index, Boolean element)
      {
        MutableBooleanIndexable.this.set(index, element);
        return element;
      }

      @Override
      public Iterator<Boolean> iterator()
      {
        return MutableBooleanIndexable.this.iterator();
      }
    };
  }

  @NotNull
  @Override
  default MutableBooleanIndexable getCopy()
  {
    return viewArray(toArray());
  }

  /**
   * View any mutable boolean indexable as a base boolean indexable.
   * The latter provides various standard methods for comparison, output and hanshing.
   * @param indexable mutqable boolean indexable to view as a {@link Base} mutable boolean indexable
   * @return base view of given indexable, or the indexable itself if it is already a Base
   */
  @NotNull
  static Base based(@NotNull MutableBooleanIndexable indexable)
  {
    if (indexable instanceof Base) {
      return (Base)indexable;
    }
    return new Base()
    {
      @Override
      public void set(int index, boolean value)
      {
        indexable.set(index, value);
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean get(int index)
      {
        return indexable.get(index);
      }
    };
  }

  /**
   * Initialize a mutable indexable to a given size.
   * @param size     size of indexable
   * @param creator  creator for the initial elements
   * @return mutable indexable with the given size
   * @see #init(int, boolean)
   * @see #init(int, BooleanSupplier)
   */
  @NotNull
  static Base init(int size, @NotNull BooleanSupplier creator)
  {
    final boolean[] array = new boolean[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = creator.getAsBoolean();
    }
    return viewArray(array);
  }

  /**
   * Create a mutable indexable which is initialized from a
   * given collection.
   * @param collection collection
   * @return mutable indexable
   */
  @NotNull
  static Base copyOf(@NotNull Collection<? extends Boolean> collection)
  {
    final boolean[] array = new boolean[collection.size()];
    int i = 0;
    for (Boolean b : collection) {
      array[i++] = b;
    }
    return viewArray(array);
  }

  /**
   * Create a mutable indexable which is initialized from copied elements of
   * a given collection.
   * @param collection collection
   * @param copier     element copier
   * @param <IN>       incoming element type
   * @return mutable indexable
   */
  @NotNull
  static <IN> Base copy(@NotNull Collection<IN> collection,
                        @NotNull Predicate<IN> copier)
  {
    final boolean[] array = new boolean[collection.size()];
    int i = 0;
    for (IN v : collection) {
      array[i++] = copier.test(v);
    }
    return viewArray(array);
  }

  @NotNull
  static Base empty()
  {
    return EMPTY;
  }

  /**
   * Create a mutable indexable which is initialized from elements
   * of the given array.
   * @param elements  elements
   * @return mutable indexable independent of {@code elements}
   */
  @NotNull
  static Base fromArray(@NotNull boolean... elements)
  {
    return fromArray(elements, 0, elements.length);
  }

  /**
   * Create a mutable indexable which is initialized from elements
   * of the given array.
   * @param elements    array of elements
   * @param startIndex  index of first element used in the returned indxable
   * @param length      length number of elements used in the returned indexable
   * @return mutable indexable independnet of {@code eleements}
   */
  @NotNull
  static Base fromArray(@NotNull boolean[] elements,
                        int startIndex, int length)
  {
    if (startIndex < 0  ||  startIndex + length > elements.length)  {
      throw new IllegalArgumentException("Indexes out of bounds!");
    }
    return viewArray(Arrays.copyOfRange(elements, startIndex, startIndex + length));
  }

  /**
   * Mutable indexable view which operates on the given array.
   * Any changes done by the returned indexable will be forwarded to the array.
   * @param array array
   * @return mutable indexable view of the given array
   */
  @NotNull
  static Base viewArray(@NotNull boolean[] array)
  {
    return new Base()
    {
      @Override
      public void set(int index, boolean elem)
      {
        array[index] = elem;
      }

      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public boolean get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * Mutable indexable view which operates on the given array.
   * Any changes done by the returned indexable will be forwarded to the list.
   * @param list list
   * @return mutable indexable view of the given list
   */
  @NotNull
  static Base viewList(@NotNull List<Boolean> list)
  {
    return new Base()
    {
      @Override
      public void set(int index, boolean elem)
      {
        list.set(index, elem);
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean get(int index)
      {
        return list.get(index);
      }
    };
  }

  /**
   * Create a mutable indexable which is the copy of a standard indexable.
   * @param indexable standard indexable
   * @return mutable indexable initialize from the standard indexable
   */
  @NotNull
  static Base fromIndexable(@NotNull Indexable<? extends Boolean> indexable)
  {
    return copyOf(indexable.asCollection());
  }

  /**
   * Create a mutable indexable which is the copy of a standard indexable.
   * This especieally allows to copy each element during the initialization.
   * @param indexable  base indexable
   * @param converter  converter from base indexable type to result element type,
   * @param <B> element type of incoming idexable
   * @return mutable indexable
   */
  @NotNull
  static <B> Base fromIndexable(@NotNull Indexable<B> indexable,
                                @NotNull Predicate<? super B> converter)
  {
    final boolean[] array = new boolean[indexable.size()];
    int i = 0;
    for (B elem : indexable) {
      array[i++] = converter.test(elem);
    }
    return viewArray(array);
  }
  
  /**
   * View a generic indexable as a mutable boolean indexable by accessing
   * a boolean property of the elements of the underlying generic indexable.
   * <p>
   * This is useful if you have complex items, but are interested
   * into only one (boolean) property of each item.
   * @param indexable underlying generic indexable
   * @param getter    function used to extract the property of interest
   * @param setter    procedure used to set the property of interest
   * @param <T>       element type of the underlying indexable
   * @return mutable boolean indexable view of the underlying indexable
   */
  @NotNull
  static <T> MutableBooleanIndexable.Base viewIndexable(@NotNull Indexable<T> indexable,
                                                        @NotNull Predicate<? super T> getter,
                                                        @NotNull BooleanSetter<? super T> setter)
  {
    return new MutableBooleanIndexable.Base()
    {
      @Override
      public void set(int index, boolean value)
      {
        setter.set(indexable.get(index), value);
      }

      @Override
      public boolean get(int index)
      {
        return getter.test(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }
    };
  }

  /**
   * Return a mutable boolean indexable initialized with the content of a
   * standard boolean indexable.
   * @param indexable boolean indexable
   * @return mutable indexable initialized with the values of the given boolean indexable
   */
  @NotNull
  static Base fromBooleanIndexable(@NotNull BooleanIndexable indexable)
  {
    return initByIndex(indexable.size(), indexable::get);
  }

  /**
   * Create a mutable indexable from an iterable or a part of it.
   * @param size maximum size of returned indexable
   * @param iter iterable used for initializing the indexable
   * @return mutable indexable
   */
  @NotNull
  static Base fromIterable(int size, @NotNull Iterable<? extends Boolean> iter)
  {
    final boolean[] array = new boolean[size];
    int i = 0;
    for (Boolean elem : iter) {
      array[i++] = elem;
      if (i == size) {
        break;
      }
    }
    return viewArray(array);
  }

  /**
   * Create a mutable 
   * @param size  required size of returned mutable boolean indexable
   * @param value initial value of all elements
   * @return mutable boolean indexable of the given size initialized with
   *         {@code value}
   * @see #init(int, BooleanSupplier)
   */
  @NotNull
  static MutableBooleanIndexable.Base init(int size, boolean value)
  {
    final boolean[] array = new boolean[size];
    if (value) {
      Arrays.fill(array, true);
    }
    return viewArray(array);
  }

  /**
   * Get a mutable boolean indexable of a given size which contains elements created by index.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value              
   * @return a mutable boolean indexable with the given size with elements initialized by calling the {@code producer}
   */
  @NotNull
  static MutableBooleanIndexable.Base initByIndex(int size, @NotNull IntPredicate producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    final boolean[] array = new boolean[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = producer.test(i);
    }
    return viewArray(array);
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   * @see Indexable#asBase()
   */
  abstract class Base extends BooleanIndexable.Base implements MutableBooleanIndexable
  {
  }

  /**
   * Empty mutable indexable.
   * Use {@link #empty()} instead.
   */
  Base EMPTY = new MutableBooleanIndexable.Base()
  {
    @Override
    public void set(int index, boolean value)
    {
      throw new IndexOutOfBoundsException("Cannot set in empty indexable!");
    }

    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public boolean get(int index)
    {
      throw new IndexOutOfBoundsException("Cannot get from empty indexable!");
    }

    @NotNull
    @Override
    public ListIterator<Boolean> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public MutableBooleanIndexable.Base reverse()
    {
      return this;
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public Iterator<Boolean> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public PrimitiveBooleanIterator booleanIterator()
    {
      return Types.EMPTY_BOOLEAN_ITERATOR;
    }

    @Override
    public void initByIndex(@NotNull IntPredicate1 setter)
    {
    }

    @NotNull
    @Override
    public Collection<Boolean> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Boolean> asList()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return Types.emptyIterable();
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @Override
    public boolean foldLeft(boolean initialValue, @NotNull BooleanOperator2 foldOperation)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public boolean[] toArray()
    {
      return Empty.BOOLEAN_ARRAY;
    }

    @Override
    public int addToArray(@NotNull boolean[] array, int pos)
    {
      return pos;
    }

    @NotNull
    @Override
    public BooleanIndexable frozen()
    {
      return this;
    }

    @NotNull
    @Override
    public MutableBooleanIndexable getCopy()
    {
      return this;
    }

    @Override
    public void forEachBool(@NotNull BooleanConsumer action)
    {
    }

    @Override
    public boolean containsBool(boolean value)
    {
      return false;
    }

    @Override
    public boolean containsTrue()
    {
      return false;
    }

    @Override
    public boolean containsFalse()
    {
      return false;
    }

    @Override
    public boolean or()
    {
      return false;
    }

    @Override
    public boolean and()
    {
      return false;
    }

    @Override
    public boolean xor()
    {
      return false;
    }

    @Override
    public void forEach(Consumer<? super Boolean> action)
    {
    }

    @Override
    @NotNull
    public String toString()
    {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }
  };
}
