// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Predicate1;

import java.util.function.BiPredicate;

/*
 * A matcher knows how to check for equality.
 *
 * Deciding whether two objects are equal is no simple task, because it
 * may depend on the circumstances. This interface allows factoring
 * these problems out.
 *
 * @author <a href="mailto>rammi@caff.de">Rammi</a>
 * @param <T1> type of the one kind of objects to compare
 * @param <T2> type of the other kind of objects to compare
 * @see de.caff.generics.UniformMatcher
 * @see de.caff.generics.HashCodeCalculator
 */
@FunctionalInterface
public interface Matcher<T1, T2>
        extends BiPredicate<T1, T2>
{
  /**
   * Are the given two objects considered equal?
   * @param object1 first object
   * @param object2 second object
   * @return {@code true} if this matcher considers the two objects equal<br>
   *         {@code false} if this matcher considers them different
   */
  boolean areEqual(T1 object1, T2 object2);

  @Override
  default boolean test(T1 t1, T2 t2)
  {
    return areEqual(t1, t2);
  }

  /**
   * Get a predicate with a fix argument for the first object
   * of the equality check.
   * @param object1 fix first object
   * @return predicate checking an object for being the same as
   *         {@code object1}
   */
  @NotNull
  default Predicate1<T2> foldLeft(T1 object1)
  {
    return obj -> areEqual(object1, obj);
  }

  /**
   * Get a predicate with a fix argument for the second object
   * of the equality check.
   * @param object2 fix second object argument
   * @return predicate checking an object for being the same as
   *         {@code object2}
   */
  @NotNull
  default Predicate1<T1> foldRight(T2 object2)
  {
    return obj -> areEqual(obj, object2);
  }
}
