// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.matcher.Match;

import java.util.*;

/**
 * A set which is using an {@link UniformMatcher} for checker for equality.
 * <p>
 * Equality checking will always use the element already in the list as
 * first argument of the matcher.
 * <p>
 * Because a matcher can only check for equality, each insertion operation
 * needs to compare all objects contained in the set with the new one.
 * Therefore this is quite a slow set, all operations are either O(N)
 * or O(N^2).
 * <p>
 * Another difference is that the order of insertions is kept, duplicate additions
 * just add nothing.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <E> set type
 */
public class MatchSet<E>
        implements Set<E>
{
  private final UniformMatcher<? super E> matcher;
  private final List<E> list = new LinkedList<>();

  /**
   * Constructor.
   * This uses a {@link de.caff.generics.matcher.Match#DEFAULT_MATCHER} as matcher,
   * i.e. uses the elements equals() method.
   */
  public MatchSet()
  {
    this(Match.DEFAULT_MATCHER);
  }

  /**
   * Constructor.
   * @param matcher matcher to use for equality checking
   */
  public MatchSet(UniformMatcher<? super E> matcher)
  {
    this.matcher = matcher;
  }

  /**
   * Constructor.
   * @param matcher matcher to use for equality checking
   * @param iterable elements to add
   */
  public MatchSet(UniformMatcher<? super E> matcher, Iterable<? extends E> iterable)
  {
    this.matcher = matcher;
    for (E elem: iterable) {
      add(elem);
    }
  }

  /**
   * Returns the number of elements in this set (its cardinality).  If this
   * set contains more than <tt>Integer.MAX_VALUE</tt> elements, returns
   * <tt>Integer.MAX_VALUE</tt>.
   *
   * @return the number of elements in this set (its cardinality).
   */
  @Override
  public int size()
  {
    return list.size();
  }

  /**
   * Returns <tt>true</tt> if this set contains no elements.
   *
   * @return <tt>true</tt> if this set contains no elements.
   */
  @Override
  public boolean isEmpty()
  {
    return list.isEmpty();
  }

  /**
   * Returns <tt>true</tt> if this set contains the specified element.  More
   * formally, returns <tt>true</tt> if and only if this set contains an
   * element {@code e} such that
   * {@code (o==null ? e==null : o.equals(e))}.
   *
   * @param o element whose presence in this set is to be tested.
   * @return <tt>true</tt> if this set contains the specified element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this set (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              set does not support null elements (optional).
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean contains(Object o)
  {
    for (E elem : list) {
      if (matcher.areEqual(elem, (E)o)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Returns an iterator over the elements in this set.  The elements are
   * returned in no particular order (unless this set is an instance of some
   * class that provides a guarantee).
   *
   * @return an iterator over the elements in this set.
   */
  @Override
  @NotNull
  public Iterator<E> iterator()
  {
    return list.iterator();
  }

  /**
   * Returns an array containing all of the elements in this set.
   * Obeys the general contract of the <tt>Collection.toArray</tt> method.
   *
   * @return an array containing all of the elements in this set.
   */
  @Override
  @NotNull
  public Object[] toArray()
  {
    return list.toArray();
  }

  /**
   * Returns an array containing all of the elements in this set; the
   * runtime type of the returned array is that of the specified array.
   * Obeys the general contract of the
   * <tt>Collection.toArray(Object[])</tt> method.
   *
   * @param a the array into which the elements of this set are to
   *          be stored, if it is big enough; otherwise, a new array of the
   *          same runtime type is allocated for this purpose.
   * @return an array containing the elements of this set.
   * @throws ArrayStoreException  the runtime type of a is not a supertype
   *                              of the runtime type of every element in this set.
   * @throws NullPointerException if the specified array is <tt>null</tt>.
   */
  @Override
  @NotNull
  @SuppressWarnings("unchecked")
  public <U> U[] toArray(@NotNull U[] a)
  {
    return list.toArray(a);
  }

  /**
   * Adds the specified element to this set if it is not already present
   * (optional operation).  More formally, adds the specified element,
   * {@code o}, to this set if this set contains no element
   * {@code e} such that {@code (o==null ? e==null : o.equals(e))}.
   * If this set already contains the specified
   * element, the call leaves this set unchanged and returns <tt>false</tt>.
   * In combination with the restriction on constructors, this ensures that
   * sets never contain duplicate elements.<p>
   *
   * The stipulation above does not imply that sets must accept all
   * elements; sets may refuse to add any particular element, including
   * <tt>null</tt>, and throwing an exception, as described in the
   * specification for <tt>Collection.add</tt>.  Individual set
   * implementations should clearly document any restrictions on the
   * elements that they may contain.
   *
   * @param o element to be added to this set.
   * @return <tt>true</tt> if this set did not already contain the specified
   *         element.
   * @throws UnsupportedOperationException if the <tt>add</tt> method is not
   *                                       supported by this set.
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this set.
   * @throws NullPointerException          if the specified element is null and this
   *                                       set does not support null elements.
   * @throws IllegalArgumentException      if some aspect of the specified element
   *                                       prevents it from being added to this set.
   */
  @Override
  public boolean add(E o)
  {
    if (!contains(o)) {
      list.add(o);
      return true;
    }
    return false;
  }

  /**
   * Removes the specified element from this set if it is present (optional
   * operation).  More formally, removes an element {@code e} such that
   * {@code (o==null ?  e==null : o.equals(e))}, if the set contains
   * such an element.  Returns <tt>true</tt> if the set contained the
   * specified element (or equivalently, if the set changed as a result of
   * the call).  (The set will not contain the specified element once the
   * call returns.)
   *
   * @param o object to be removed from this set, if present.
   * @return true if the set contained the specified element.
   * @throws ClassCastException            if the type of the specified element
   *                                       is incompatible with this set (optional).
   * @throws NullPointerException          if the specified element is null and this
   *                                       set does not support null elements (optional).
   * @throws UnsupportedOperationException if the <tt>remove</tt> method is
   *                                       not supported by this set.
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean remove(Object o)
  {
    for (ListIterator<E> it = list.listIterator();  it.hasNext();  ) {
      if (matcher.areEqual(it.next(), (E)o)) {
        it.remove();
        return true;
      }
    }
    return false;
  }

  /**
   * Returns <tt>true</tt> if this set contains all of the elements of the
   * specified collection.  If the specified collection is also a set, this
   * method returns <tt>true</tt> if it is a <i>subset</i> of this set.
   *
   * @param c collection to be checked for containment in this set.
   * @return <tt>true</tt> if this set contains all of the elements of the
   *         specified collection.
   * @throws ClassCastException   if the types of one or more elements
   *                              in the specified collection are incompatible with this
   *                              set (optional).
   * @throws NullPointerException if the specified collection contains one
   *                              or more null elements and this set does not support null
   *                              elements (optional);
   *                              or if the specified collection is
   *                              <tt>null</tt>.
   * @see #contains(Object)
   */
  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    for (Object o : c) {
      if (!contains(o)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Adds all of the elements in the specified collection to this set if
   * they're not already present (optional operation).  If the specified
   * collection is also a set, the <tt>addAll</tt> operation effectively
   * modifies this set so that its value is the <i>union</i> of the two
   * sets.  The behavior of this operation is unspecified if the specified
   * collection is modified while the operation is in progress.
   *
   * @param c collection whose elements are to be added to this set.
   * @return <tt>true</tt> if this set changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>addAll</tt> method is
   *                                       not supported by this set.
   * @throws ClassCastException            if the class of some element of the
   *                                       specified collection prevents it from being added to this
   *                                       set.
   * @throws NullPointerException          if the specified collection contains one
   *                                       or more null elements and this set does not support null
   *                                       elements, or if the specified collection is <tt>null</tt>.
   * @throws IllegalArgumentException      if some aspect of some element of the
   *                                       specified collection prevents it from being added to this
   *                                       set.
   * @see #add(Object)
   */
  @Override
  public boolean addAll(@NotNull Collection<? extends E> c)
  {
    boolean result = false;
    for (E o : c) {
      if (add(o)) {
        result = true;
      }
    }
    return result;
  }

  /**
   * Retains only the elements in this set that are contained in the
   * specified collection (optional operation).  In other words, removes
   * from this set all of its elements that are not contained in the
   * specified collection.  If the specified collection is also a set, this
   * operation effectively modifies this set so that its value is the
   * <i>intersection</i> of the two sets.
   *
   * @param c collection that defines which elements this set will retain.
   * @return <tt>true</tt> if this collection changed as a result of the
   *         call.
   * @throws UnsupportedOperationException if the <tt>retainAll</tt> method
   *                                       is not supported by this Collection.
   * @throws ClassCastException            if the types of one or more elements in this
   *                                       set are incompatible with the specified collection
   *                                       (optional).
   * @throws NullPointerException          if this set contains a null element and
   *                                       the specified collection does not support null elements
   *                                       (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean retainAll(@NotNull Collection<?> c)
  {
    boolean changed = false;
  outer:
    for (ListIterator<E> it = list.listIterator();  it.hasNext();  ) {
      E elem = it.next();
      for (Object o : c) {
        if (matcher.areEqual(elem, (E)o)) {
          continue outer;
        }
      }
      it.remove();
      changed = true;
    }
    return changed;
  }

  /**
   * Removes from this set all of its elements that are contained in the
   * specified collection (optional operation).  If the specified
   * collection is also a set, this operation effectively modifies this
   * set so that its value is the <i>asymmetric set difference</i> of
   * the two sets.
   *
   * @param c collection that defines which elements will be removed from
   *          this set.
   * @return <tt>true</tt> if this set changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>removeAll</tt>
   *                                       method is not supported by this Collection.
   * @throws ClassCastException            if the types of one or more elements in this
   *                                       set are incompatible with the specified collection
   *                                       (optional).
   * @throws NullPointerException          if this set contains a null element and
   *                                       the specified collection does not support null elements
   *                                       (optional;
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   */
  @Override
  public boolean removeAll(@NotNull Collection<?> c)
  {
    boolean result = false;
    for (Object o : c) {
      if (remove(o)) {
        result = true;
      }
    }
    return result;
  }

  /**
   * Removes all of the elements from this set (optional operation).
   * This set will be empty after this call returns (unless it throws an
   * exception).
   *
   * @throws UnsupportedOperationException if the <tt>clear</tt> method
   *                                       is not supported by this set.
   */
  @Override
  public void clear()
  {
    list.clear();
  }
}
