// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.Function3;
import de.caff.generics.function.Function4;

import java.util.NoSuchElementException;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * Generic version of a loop where loop items can contain any value.
 * <p>
 * This is useful in cases when the loop is manipulated from the outside.
 * It is also restricted to non-{@code null} values.
 * <p>
 * As the underlying {@link BasicLoop} this class is not thread-safe.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 02, 2022
 */
public class Loop<T>
        extends BasicLoop<Loop.Item<T>>
{
  @NotNull
  private final Function<T, Function3<Item<T>, BasicLoop<Item<T>>, Item<T>, Item<T>>> valueToItemCreator;
  {
    final Function4<Item<T>, T, BasicLoop<Item<T>>, Item<T>, Item<T>> newLoop = Item::new;
    valueToItemCreator = newLoop::partial1;
  }

  /**
   * Generic loop item.
   * @param <T> value type
   */
  public static class Item<T>
          extends BasicLoop.Item<Item<T>>
  {
    @NotNull
    private T value;

    /**
     * Create an item.
     * @param value    value contained in this item
     * @param loop     loop to which this item will belong
     * @param previous previous item (or {@code null} if this is the first item)
     * @param next     next item (or {@code null} if this is the first item)
     */
    protected Item(@NotNull T value,
                   @NotNull BasicLoop<Item<T>> loop,
                   @Nullable Item<T> previous,
                   @Nullable Item<T> next)
    {
      super(loop, previous, next);

      this.value = value;
    }

    /**
     * Get the value.
     * @return value
     */
    @NotNull
    public T getValue()
    {
      return value;
    }

    /**
     * Exchange the value
     * @param newValue new value to set for this item
     * @return old value of this item
     */
    @NotNull
    public T setValue(@NotNull T newValue)
    {
      final T oldValue = value;
      value = newValue;
      return oldValue;
    }

    /**
     * Find the next item for which its value fulfills a given condition.
     * @param condition value condition
     * @return next matching item, or {@code null} if no item matches
     */
    @Nullable
    public Item<T> findNextValue(@NotNull Predicate<? super T> condition)
    {
      return findNext(item -> condition.test(item.value));
    }

    /**
     * Find the previous item for which its value fulfills a given condition.
     * @param condition value condition
     * @return next matching item, or {@code null} if no item matches
     */
    @Nullable
    public Item<T> findPreviousValue(@NotNull Predicate<? super T> condition)
    {
      return findPrevious(item -> condition.test(item.value));
    }

    @Override
    public String toString()
    {
      return value.toString();
    }
  }

  /**
   * Empty loop constructor.
   */
  public Loop()
  {}

  /**
   * Constructor.
   * @param elements initial loop elements
   */
  public Loop(@NotNull Iterable<T> elements)
  {
    addAll(elements);
  }

  /**
   * Get a countable view of the values of this loop.
   * @return countable value view
   * @see #closedValueView()
   */
  @NotNull
  public final Countable<T> values()
  {
    return view(Item::getValue);
  }

  /**
   * View this loop's values as a countable which is "closed".
   * The returned countable will virtually close the view by adding the first
   * value of the loop again at the end. This simplifies some usages of the loop.
   * @return closed value view of this loop
   * @see #values()
   */
  @NotNull
  public Countable<T> closedValueView()
  {
    return closedItemView().view(Item::getValue);
  }

  /**
   * Add a value to the end of the loop.
   * @param value value to add
   * @return added loop item
   */
  @NotNull
  public Item<T> add(@NotNull T value)
  {
    return add(valueToItemCreator.apply(value));
  }

  /**
   * Add all values.
   * @param values values to add to this loop
   */
  public void addAll(@NotNull Iterable<? extends T> values)
  {
    addAll(values, valueToItemCreator);
  }

  /**
   * Get the value of the first item in this loop.
   * @return first item's value
   * @throws NoSuchElementException if this countable is empty
   */
  public T firstValue()
  {
    return first().getValue();
  }

  /**
   * Get the value of the last item in this loop.
   * @return last item's value
   * @throws NoSuchElementException if this countable is empty
   */
  public T lastValue()
  {
    return last().getValue();
  }

  /**
   * Find the first item in this loop for which the value fulfills a given condition.
   * @param condition value condition
   * @return first matching item, or {@code null} if none matches
   */
  @Nullable
  public Item<T> findFirstValue(@NotNull Predicate<? super T> condition)
  {
    return super.findFirst(item -> condition.test(item.value));
  }

  /**
   * Find the last item in this loop for which the value fulfills a given condition.
   * @param condition value condition
   * @return last matching item, or {@code null} if none matches
   */
  @Nullable
  public Item<T> findLastValue(@NotNull Predicate<? super T> condition)
  {
    return super.findLast(item -> condition.test(item.value));
  }

  /**
   * Rotate this loop forward until the value of the first item fulfills a given condition.
   * @param condition value condition
   * @return {@code true} if the condition is now fulfilled for the {@link #getFirstItem() first} value,
   *         {@code false} if not or if this loop is empty
   */
  public boolean rotateForwardUntilValue(@NotNull Predicate<? super T> condition)
  {
    return rotateForwardUntil(item -> condition.test(item.value));
  }

  /**
   * Rotate this loop backward until the value of the first item fulfills a given condition.
   * @param condition value condition
   * @return {@code true} if the condition is now fulfilled for the {@link #getFirstItem() first} value,
   *         {@code false} if not or if this loop is empty
   */
  public boolean rotateBackwardUntilValue(@NotNull Predicate<? super T> condition)
  {
    return rotateBackwardUntil(item -> condition.test(item.value));
  }

  /**
   * Perform an action for each value in this loop.
   * @param valueAction action to be performed for each value
   */
  public void forEachValue(@NotNull Consumer<? super T> valueAction)
  {
    values().forEach(valueAction);
  }

  /**
   * Clear this loop.
   */
  public void clear()
  {
    forEach(BasicLoop.Item::remove);
  }
}
