// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.LongFunction2;
import de.caff.generics.function.LongOperator2;
import de.caff.generics.function.LongPredicate2;
import de.caff.generics.tuple.Tuple;
import de.caff.generics.tuple.Tuple2;

import java.io.Serializable;
import java.util.Locale;
import java.util.Objects;

/**
 * A pair of primitive long values.
 * <p>
 * This is similar to {@link Pair}, but for raw {@code long} values.
 * As these values are defined final, this class is also immutable.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 04, 2020
 */
public final class LongPair
        implements Serializable,
                   LongIndexable
{
  private static final long serialVersionUID = -4239662296731228131L;
  /** The first value of this pair. */
  public final long first;
  /** The second value of this pair. */
  public final long second;

  /**
   * Constructor.
   * @param first    first value of pair
   * @param second   second value of pair
   */
  public LongPair(long first, long second)
  {
    this.first = first;
    this.second = second;
  }

  /**
   * Get this pair with a different first value.
   * @param first new first value
   * @return pair with the given {@code first} value,
   *         and the {@link #second} value of this pair
   */
  @NotNull
  public LongPair withFirst(long first)
  {
    if (first == this.first) {
      return this;
    }
    return new LongPair(first, this.second);
  }

  /**
   * Get this pair with a different first value.
   * @param second new second value
   * @return pair with the {@link #first} value of this pair,
   *         and the given {@code second} value
   */
  @NotNull
  public LongPair withSecond(long second)
  {
    if (second == this.second) {
      return this;
    }
    return new LongPair(this.first, second);
  }

  /**
   * Get this pair with swapped values.
   * @return pair with swapped {@link #first} and {@link #second} values
   */
  @NotNull
  public LongPair swapped()
  {
    if (first == second) {
      return this;
    }
    return new LongPair(second, first);
  }

  @Override
  public int size()
  {
    return 2;
  }

  @Override
  public long get(int index)
  {
    switch (index) {
    case 0:
      return first;
    case 1:
      return second;
    default:
      throw new IndexOutOfBoundsException(String.format("Pair has only 2 elements, so %d is out of bounds!", index));
    }
  }

  /**
   * Apply a 2-long function on this pair.
   * @param function function to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @param <T> output type
   * @return result of applying the function
   */
  public <T> T applyAsLongs(@NotNull LongFunction2<T> function)
  {
    return function.applyAsLong(first, second);
  }

  /**
   * Call an 2-long operator on this pair.
   * @param operator operator to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @return result of applying the operator
   */
  public long operate(@NotNull LongOperator2 operator)
  {
    return operator.applyAsLong(first, second);
  }

  /**
   * Call a 2-long predicate on this pair. 
   * @param predicate predicate to apply, {@link #first} becomes its first, {@link #second} its second argument 
   * @return result of testing this pair
   */
  public boolean test(@NotNull LongPredicate2 predicate)
  {
    return predicate.testLongs(first, second);
  }

  /**
   * Convert this into a tuple.
   * @return tuple created from this pair
   */
  @NotNull
  public Tuple2<Long, Long> toTuple()
  {
    return Tuple.of(first, second);
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final LongPair that = (LongPair)o;
    return that.first == first && that.second == second;
  }

  @Override
  public int hashCode()
  {
    return Objects.hash(first, second);
  }

  @Override
  @NotNull
  public String toString()
  {
    return String.format(Locale.US, "LongPair(%s, %s)", first, second);
  }
}
