// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.LongOrdering;
import de.caff.generics.range.Range;

import java.util.*;
import java.util.function.*;

import static de.caff.generics.Order.Ascending;
import static de.caff.generics.Order.Descending;

/**
 * Class which allows readonly access of raw long integers by index.
 * <p>
 * The {@link #EMPTY} constant or {@link #emptyIndexable()} method provide the same
 * useful return value to indicate emptiness.
 * <p>
 * Simple implementations should extend {@link Base} because
 * that provides useful implementations for standard Object methods and implements
 * {@link Comparable}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 7, 2020
 * @see de.caff.generics.algorithm.FastLongSequenceSearch
 */
public interface LongIndexable
        extends PrimitiveLongIterable,
                Sizeable
{
  /**
   * Get the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @return element at the given index
   */
  long get(int index);

  /**
   * Pythonesque get.
   * This allows accessing elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @return element at the given index
   */
  default long gyt(int index)
  {
    return get(Pythonesque.mapX(index, this));
  }

  /**
   * Get an element modulo size.
   * <p>
   * This maps the given {@code index} into the range of this indexable
   * by applying a modulo {@link #size()} operation. For empty indexable
   * this will throw an {@link IndexOutOfBoundsException} as there is no
   * possible index to get.
   * @param index index, possibly out of range, possibly even negative
   *              for Pythonesque access
   * @return element at the given, possibly modulated, index
   */
  default long getMod(int index)
  {
    final int size = size();
    if (size == 0) {
      throw new IndexOutOfBoundsException("No element for empty indexable!");
    }
    return gyt(index % size);
  }

  /**
   * Get this indexable but with inverted order.
   * @return a view to this indexable which accesses the elements in reverse order
   */
  @NotNull
  default Base reverse()
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }

      @Override
      public long get(int index)
      {
        return LongIndexable.this.get(LongIndexable.this.size() - index - 1);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return LongIndexable.this.asBase();
      }
    };
  }

  /**
   * Is this indexable empty?
   * @return {@code true}: there are no elements in this indexable<br>
   *         {@code false}: this indexable has elements
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * Get an primitive long iterator.
   * @return long iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveIterator.OfLong longIterator()
  {
    return longIterator(0, size());
  }

  /**
   * Get a long iterator from the given sub set.
   * @param from first index of iterator
   * @param to   one past last index of iterator
   * @return int iterator which iterates over the given part of this indexable
   */
  @NotNull
  default PrimitiveIterator.OfLong longIterator(final int from, final int to)
  {
    return new PrimitiveIterator.OfLong()
    {
      private int index = from;

      @Override
      public long nextLong()
      {
        if (index >= to) {
          throw new NoSuchElementException(String.format("Index %s out of allowed range [%d, %d[!",
                                                         index, from, to));
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < to;
      }
    };
  }

  /**
   * Returns an iterator over elements of type {@code T}.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  default Iterator<Long> iterator()
  {
    return listIterator();
  }

  /**
   * Returns a list iterator over elements of type {@code T}.
   *
   * @return a list iterator.
   */
  @NotNull
  default ListIterator<Long> listIterator()
  {
    return new ListIterator<Long>() {
      private int index = 0;

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public Long next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public Long previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(--index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(Long t)
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void add(Long t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  /**
   * Get an indexable subset.
   * The subset includes indices {@code fromIndex} to {@code toIndex - 1}.
   * @param fromIndex start index of sub set
   * @param toIndex   index after last index
   * @return indxable subset view of this indexable
   */
  @NotNull
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0)
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    if (toIndex > size())
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    if (fromIndex > toIndex)
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    final int length = toIndex - fromIndex;
    return length == 0
            ? EMPTY
            : new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public long get(int index)
      {
        if (index >= length) {
          throw new IndexOutOfBoundsException("No such element: "+index);
        }
        return LongIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return LongIndexable.this.subSet(fromIndex + fromIdx,
                                         fromIndex + toIdx);
      }
    };
  }

  /**
   * Get an indexable subset.
   * This is the Pythonesque version which allows negative indexes.
   * @param fromIndex start index of sub set
   * @param toIndex   end index of sub set
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param fromIndex index to start with, negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param toIndex index one after the end (equal to the length of the returned set),
   *                negative counts from the back
   * @return indexable subset view of this indexable set
   */
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Get a view on this indexable as an unmodifiable collection.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   */
  @NotNull
  default Collection<Long> asCollection()
  {
    return new AbstractCollection<Long>()
    {
      @NotNull
      @Override
      public Iterator<Long> iterator()
      {
        return LongIndexable.this.iterator();
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable list.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   * @see #toList()
   */
  @NotNull
  default List<Long> asList()
  {
    return new AbstractList<Long>()
    {
      @Override
      public Long get(int index)
      {
        return LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }

      @NotNull
      @Override
      public Iterator<Long> iterator()
      {
        return LongIndexable.this.iterator();
      }
    };
  }

  /**
   * Create a list from this long indexable.
   * This creates an independent list to which the elements of this
   * indexable are copied.
   * @return list copy of this indexable
   * @see #asList()
   */
  @NotNull
  default ArrayList<Long> toList()
  {
    final ArrayList<Long> list = new ArrayList<>(size());
    addAllTo(list);
    return list;
  }

  /**
   * Add all elements of this indexable to the given collection.
   * @param collection collection to add to
   */
  default void addAllTo(@NotNull Collection<? super Long> collection)
  {
    for (Long entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Get a view of this indexable as a standard object-based indexable.
   * @return indexable with non-null {@code Long} values
   */
  @NotNull
  default Indexable<Long> asIndexable()
  {
    return new Indexable.Base<Long>()
    {
      @Override
      public Long get(int index)
      {
        return LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * Convert this indexable into a Java array.
   * @return array copy containing all values of this indexable
   */
  @NotNull
  default long[] toArray()
  {
    if (isEmpty()) {
      return Empty.LONG_ARRAY;
    }
    final long[] array = new long[size()];
    addToArray(array, 0);
    return array;
  }


  /**
   * Add the content of this indexable to the given array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull long[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull long[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveIterator.OfLong it = longIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextLong();
    }
    return arrayPos;
  }

  /**
   * Get the indexes of this indexable as an iterable range.
   * @return iterable indexes
   * @see #intIndexes()
   * @see Range#indexes(int)
   */
  @NotNull
  default Iterable<Integer> indexes()
  {
    return Range.indexes(size());
  }

  /**
   * Get the indexes of this indexable as an integer indexable.
   * @return indexes as integer indexable
   * @see #indexes()
   * @see IntIndexable#rangeFromSize(int)
   */
  @NotNull
  default IntIndexable intIndexes()
  {
    return IntIndexable.rangeFromSize(size());
  }

  /**
   * Go over all values and do a cumulative calculation.
   * <p>
   * Eg calculate the maximum of this indexable use
   * <pre>{@code
   *   long max = indexable.foldLeft(Long.MIN_VALUE, Math::max);
   * }</pre>
   *
   * @param initialValue  start value for the operation, will be returned if this iterable is empty
   * @param foldOperation operation applied to each value, will get the accumulated value as its
   *                      first and the current element value as its second argument
   * @return accumulated value, result of applying fold operation to all values of this indexable
   *         iteratively
   */
  default long foldLeft(long initialValue,
                        @NotNull LongBinaryOperator foldOperation)
  {
    long result = initialValue;
    for (PrimitiveIterator.OfLong iterator = longIterator();
         iterator.hasNext();  ) {
      result = foldOperation.applyAsLong(result, iterator.nextLong());
    }
    return result;
  }

  /**
   * Get the next index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int nextMatch(int startIndex, @NotNull LongPredicate check)
  {
    final int size = size();
    for (int i = Pythonesque.mapX(startIndex, this);  i < size;  ++i) {
      if (check.test(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the first index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return first index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int firstMatch(@NotNull LongPredicate check)
  {
    return nextMatch(0, check);
  }

  /**
   * Get the previous index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int previousMatch(int startIndex, @NotNull LongPredicate check)
  {
    for (int i = Pythonesque.mapX(startIndex, this);  i >= 0;  --i) {
      if (check.test(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the last index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return last index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int lastMatch(@NotNull LongPredicate check)
  {
    return previousMatch(-1, check);
  }

  /**
   * Get a frozen version of this indexable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying colelction might
   * nevertheless change. This copies the current state of this indexable
   * into an unmodifiable state, and returns an Indexable which is
   * stable in size and will return always the same element for a given index.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, so you can safely call this method more than once.
   *
   * @return frozen version of this Indexable
   */
  @NotNull
  default LongIndexable frozen()
  {
    return IndexableHelper.frozenFromArray(toArray());
  }

  /**
   * Get a spliterator on this indexable.
   * The returned spliterator will not have its {@link Spliterator#IMMUTABLE} flag set,
   * see {@link #frozenLongSpliterator()} for an alternative.
   * @return a spliterator on this indexable
   */
  @NotNull
  default Spliterator.OfLong longSpliterator()
  {
    return new LongIndexable.LongIndexableSpliterator(this);
  }

  /**
   * Get an immutable spliterator on a frozen copy of this indexable.
   * If this indexable is already immutable (i.e. if {@link #frozen()} returns {@code this})
   * then this is the same as {@link #spliterator()} with the difference that the spliterator
   * returned here will have its {@link Spliterator#IMMUTABLE} flag set. If this is not immutable
   * a frozen copy will be created, and the spliterator will be operating on that. Please refer to
   * {@link #frozen()} to understand what level of immutabiliy it will provide.
   * @return an "immutable" spliterator with the possible cost of copying this indexable
   */
  @NotNull
  default Spliterator.OfLong frozenLongSpliterator()
  {
    final LongIndexable frozen = frozen();
    return new LongIndexable.LongIndexableSpliterator(frozen,
                                                    0, frozen.size(),
                                                    true);
  }
  /**
   * Spliterator for long indexables.
   */
  class LongIndexableSpliterator implements Spliterator.OfLong
  {
    @NotNull
    private final LongIndexable indexable;
    private int index;
    private final int fence;
    private final int character;

    /**
     * Instantiate a mutable spliterator for an long indexable.
     * @param indexable indexable of this spliterator
     */
    public LongIndexableSpliterator(@NotNull LongIndexable indexable)
    {
      this(indexable, 0, indexable.size(), false);
    }

    /**
     * Instantiate a spliterator for part of an long indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param immutable is the underlying indexable immutable? Use with care!
     */
    public LongIndexableSpliterator(@NotNull LongIndexable indexable,
                                   int start,
                                   int fence,
                                   boolean immutable)
    {
      this(indexable, start, fence,
           immutable
                   ? IMMUTABLE | ORDERED | SIZED | SUBSIZED
                   : ORDERED | SIZED | SUBSIZED);
    }

    /**
     * Instantiate a spliterator for an long indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param character characteristics of this spliterator
     */
    private LongIndexableSpliterator(@NotNull LongIndexable indexable, int start, int fence, int character)
    {
      this.indexable = indexable;
      this.index = start;
      this.fence = fence;
      this.character = character;
    }

    @Override
    public OfLong trySplit()
    {
      final int here = index;
      final int mid = (here + fence) / 2;
      if (here < mid) {
        index = mid;
        return new LongIndexable.LongIndexableSpliterator(indexable,
                                                        here, mid,
                                                        character);
      }
      return null;
    }

    @Override
    public boolean tryAdvance(@NotNull LongConsumer action)
    {
      if (index < fence) {
        action.accept(indexable.get(index++));
        return true;
      }
      return false;
    }

    @Override
    public long estimateSize()
    {
      return fence - index;
    }

    @Override
    public int characteristics()
    {
      return character;
    }
  }

  /**
   * Create a view of this long indexable with an inserted value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are inserted.
   * Use {@link ExpandableLongIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable might 
   * result in a disaster when using the returned indexable!
   *
   * @param index index where the value is inserted before the current index
   *              (<b>not Pythonesque</b>, because it is possible to add an value at {@code index == size()},
   *              and it is expected that most insertion will happen at 0 or at the end)
   * @param value inserted value at that index
   * @return a view of this indexable with 1 more value at the given index,
   *         all values beyond are moved one index forward
   * @see #withAppendedValue(long)
   */
  @NotNull
  default LongIndexable withInsertedValueAt(int index, long value)
  {
    final int insertIndex = index;
    final int newSize = size() + 1;
    if (insertIndex == 0) {
      return new LongIndexable.Base() {
        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public long get(int index)
        {
          return index == 0
                  ? value
                  : LongIndexable.this.get(index - 1);
        }
      };
    }
    if (insertIndex == newSize - 1) {
      return new LongIndexable.Base() {

        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public long get(int index)
        {
          return index == newSize - 1
                  ? value
                  : LongIndexable.this.get(index);
        }
      };
    }
    return new LongIndexable.Base()
    {
      @Override
      public int size()
      {
        return newSize;
      }

      @Override
      public long get(int index)
      {
        if (index == insertIndex) {
          return value;
        }
        return LongIndexable.this.get(index < insertIndex
                                                ? index
                                                : index - 1);
      }
    };
  }

  /**
   * Create a view with of this indexable with another value added at the end.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are added.
   * @param value value to add
   * @return indexable view of this indexable, with the given value added at the end,
   *         so size is larger by 1
   * @see #withInsertedValueAt(int, long)
   */
  @NotNull
  default LongIndexable withAppendedValue(long value)
  {
    return withInsertedValueAt(size(), value);
  }

  /**
   * Create a view of this long indexable with an exchanged value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are exchanged.
   * Use {@link ExpandableLongIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable could 
   * lead to unexpected results when using the returned indexable!
   *
   * @param index index where the element is exchanged, hiding the current element.
   *              (Pythonesque)
   * @param value exchanged value at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   */
  @NotNull
  default LongIndexable withExchangedValueAt(int index, long value)
  {
    final int insertIndex = Pythonesque.mapX(index, this);
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return index == insertIndex
                ? value
                : LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this indexable with one value removed.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few values are removed.
   *
   * @param index index of the removed value (Pythonesque)
   * @return a view of this indexable where the value at the given index
   *         is removed
   */
  @NotNull
  default LongIndexable withRemovedValueAt(int index)
  {
    final int removeIndex = Pythonesque.mapX(index, this);
    if (removeIndex == 0) {
      // remove first
      return tailSet(1);
    }
    final int sz = size() - 1;
    if (removeIndex == sz) {
      // remove last
      return headSet(-1);
    }
    return new Base()
    {
      @Override
      public long get(int index)
      {
        return index < removeIndex
                ? LongIndexable.this.get(index)
                : LongIndexable.this.get(index + 1);
      }

      @Override
      public int size()
      {
        return sz;
      }
    };
  }

  /**
   * Create a view of this long indexable with two values swapped.
   * Note that this is not efficient, but handy in certain situations
   * where only a few items are swapped.
   *
   * @param index1 index of the first value (Pythonesque)
   * @param index2 index of the second value (Pythonesque)
   * @return a view of this indexable where the first and second value
   *         have exchanged places
   */
  @NotNull
  default LongIndexable withSwappedValuesAt(int index1, int index2)
  {
    final int swap1 = Pythonesque.mapX(index1, this);
    final int swap2 = Pythonesque.mapX(index2, this);
    if (swap1 == swap2) {
      return this;
    }
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        if (index == swap1) {
          return LongIndexable.this.get(swap2);
        }
        if (index == swap2) {
          return LongIndexable.this.get(swap1);
        }
        return LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * Get a rotated view of this long indexable.
   * The returned indexable will show this indexable with rotated indexes.
   * A positive {@code steps} will rotate left, i.e. any index will be accessed
   * as if the number of steps is added before extraction (modulo length).
   * E.g. a rotation of {@code -1} will return the last element when index {@code 0} is
   * requested, and the first element for index {@code 1}.
   *
   * @param steps steps to rotate
   * @return rotated view of this indexable, use  {@link #frozen()}
   *         to create an indexable which no longer depends on this one
   */
  @NotNull
  default LongIndexable rotated(int steps)
  {
    steps = steps % size();
    if (steps == 0) {
      return this;
    }
    if (steps < 0) {
      steps += size();
    }
    final int rotate = steps;
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return LongIndexable.this.get((index + rotate) % size());
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }

      @NotNull
      @Override
      public LongIndexable rotated(int steps)
      {
        return LongIndexable.this.rotated(rotate + steps);
      }
    };
  }

  /**
   * View this long indexable transformed by an operator.
   * @param op transforming operator
   * @return view of this indexable with transformed values
   */
  @NotNull
  default LongIndexable viewOp(@NotNull LongUnaryOperator op)
  {
    return LongIndexable.viewByIndex(size(),
                                       idx -> op.applyAsLong(LongIndexable.this.get(idx)));
  }

  /**
   * View this long indexable as a double indexable.
   * Note that not all long values have an exact double representation.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a double indexable view of this indexable
   */
  @NotNull
  default DoubleIndexable viewAsDouble()
  {
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as a double indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return double indexable view of this indexable
   */
  @NotNull
  default DoubleIndexable viewAsDouble(@NotNull LongToDoubleFunction convert)
  {
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return convert.applyAsDouble(LongIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View this long indexable as a float indexable.
   * Note that not all long values have an exact float representation.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a float indexable view of this indexable
   */
  @NotNull
  default FloatIndexable viewAsFloat()
  {
    return new FloatIndexable.Base()
    {
      @Override
      public float get(int index)
      {
        return (float)LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View this long indexable as a short integer indexable.
   * Note that this may result in loss of numeric range of the values.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return an integer indexable view of this indexable
   */
  @NotNull
  default IntIndexable viewAsInt()
  {
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return (int)LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as an integer indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return integer indexable view of this indexable
   */
  @NotNull
  default IntIndexable viewAsInt(@NotNull LongToIntFunction convert)
  {
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return convert.applyAsInt(LongIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View this long indexable as a short integer indexable.
   * Note that this may result in severe loss of numeric range of the values.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a short integer indexable view of this indexable
   */
  @NotNull
  default ShortIndexable viewAsShort()
  {
    return new ShortIndexable.Base()
    {
      @Override
      public short get(int index)
      {
        return (short)LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View this long indexable as a byte integer indexable.
   * Note that this may result in severe loss of numeric range of the values.
   * This will cast the float values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a byte integer indexable view of this indexable
   */
  @NotNull
  default ByteIndexable viewAsByte()
  {
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return (byte)LongIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as a boolean indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return boolean indexable view of this indexable
   */
  @NotNull
  default BooleanIndexable viewAsBoolean(@NotNull LongPredicate convert)
  {
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return convert.test(LongIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This requires a list of non-null numbers as a base.
   * @param list list used as a base, required to have only non-null elements
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public long get(int index)
      {
        return list.get(index).longValue();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This allows to provide a fallback for null elements in the list.
   * @param list list used as a base, required to have only non-null elements
   * @param nullValue value returned for {@code null} elements in the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list, int nullValue)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public long get(int index)
      {
        final Number number = list.get(index);
        return number != null
                ? number.longValue()
                : nullValue;
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic list as a long indexable.
   * @param list generic list
   * @param extractor extractor function which extracts a long from the elements of {@code list}
   * @param <T> element type of list
   * @return a long indexable view of {@code list}
   */
  @NotNull
  static <T> Base viewList(@NotNull List<T> list,
                           @NotNull ToLongFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public long get(int index)
      {
        return extractor.applyAsLong(list.get(index));
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic Number indexable as long indexable.
   * @param indexable generic indexable with Number elements, required to have only non-null elements
   * @return long indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable)
  {
    if (indexable.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public long get(int index)
      {
        return indexable.get(index).longValue();
      }
    };
  }

  /**
   * View a generic Number indexable as long indexable.
   * @param indexable generic indexable with Number elements
   * @param nullFallback fallback for {@code null} elements
   * @return long indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable,
                            long nullFallback)
  {
    if (indexable.isEmpty()) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public long get(int index)
      {
        final Number number = indexable.get(index);
        return number != null
                ? number.longValue()
                : nullFallback;
      }
    };
  }

  /**
   * View a generic indexable as a long indexable.
   * @param indexable generic indexable
   * @param extractor extractor function which extracts a long from the elements of {@code indexable}
   * @param <T> element type of indexable
   * @return a long indexable view of {@code indexable}
   */
  @NotNull
  static <T> Base viewIndexable(@NotNull Indexable<T> indexable,
                                @NotNull ToLongFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public long get(int index)
      {
        return extractor.applyAsLong(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean isEmpty()
      {
        return indexable.isEmpty();
      }
    };
  }

  /**
   * View a Number array as an Indexable.
   * To view only part of the array use {@link #subSet(int, int)}.
   * @param array array used as base, required to hold only non-null values
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewNumberArray(@NotNull Number ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public long get(int index)
      {
        return array[index].longValue();
      }
    };
  }

  /**
   * View an int array as an Indexable.
   * To view only part of the array use {@link #viewArray(long[], int, int)}.
   * @param array array used as base
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull long ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public long get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * View part of an integer array as an Indexable.
   * @param array array used as base
   * @param start index of first byte to use from the array
   * @param length number of elements to use from the array
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull long[] array,
                        int start,
                        int length)
  {
    if (start < 0  ||  length < 0  ||  start + length > array.length) {
      throw new IndexOutOfBoundsException("Start or end outside of range!");
    }
    if (length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public long get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("Index %d outside range [0, %d[!",
                                                            index, length));
        }
        return array[index + start];
      }
    };
  }

  /**
   * View an object array as a long indexable.
   * @param array         viewed array 
   * @param valueExtract  converter from array elements to the long values of this indexable 
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable 
   * @param <T> array element type
   */
  @NotNull
  static <T> LongIndexable.Base viewArray(@NotNull T[] array,
                                          @NotNull ToLongFunction<? super T> valueExtract)
  {
    if (array.length == 0) {
      return EMPTY;
    }
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return valueExtract.applyAsLong(array[index]);
      }

      @Override
      public int size()
      {
        return array.length;
      }
    };
  }

  /**
   * View a single long value as a long indexable of size 1.
   * @param value single value
   * @return indexable with one value
   */
  @NotNull
  static LongIndexable.Base singleton(long value)
  {
    return new LongIndexable.Base() {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public long get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException(String.format("Index %d for indexable of size 1!", index));
        }
        return value;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return this;
      }

      @NotNull
      @Override
      public Collection<Long> asCollection()
      {
        return Collections.singleton(value);
      }

      @NotNull
      @Override
      public List<Long> asList()
      {
        return Collections.singletonList(value);
      }

      @NotNull
      @Override
      public long[] toArray()
      {
        return new long[] { value };
      }

      @Override
      public int addToArray(@NotNull long[] array, int pos)
      {
        array[pos] = value;
        return pos + 1;
      }

      @NotNull
      @Override
      public DoubleIndexable viewAsDouble()
      {
        return super.viewAsDouble();
      }

      @NotNull
      @Override
      public Iterable<Integer> indexes()
      {
        return IntIndexable.singleton(0);
      }

      @Override
      public long sum()
      {
        return value;
      }

      @Override
      public long sumX()
      {
        return value;
      }

      @NotNull
      @Override
      public OptionalDouble average()
      {
        return OptionalDouble.of(value);
      }
    };
  }

  /**
   * Get a long indexable of a given size which always returns the same value.
   * @param size  size of the returned indexable
   * @param value value returned for each element
   * @return indexable of size {@code size} with always the same element
   */
  @NotNull
  static Base init(int size, long value)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new Base()
    {
      @Override
      public long get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return value;
      }

      @Override
      public int size()
      {
        return 0;
      }

      @NotNull
      @Override
      public LongIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a long indexable of a given size which returns elements created by index.
   * This will always call the {@code producer} when an element is requested.
   * Use {@link #frozen()} on the result to create an indexable which avoids this.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable view which calls the {@code producer} with the given index when an element is requested
   * @see #initByIndex(int, IntToLongFunction)
   */
  @NotNull
  static Base viewByIndex(int size, @NotNull IntToLongFunction producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new Base()
    {
      @Override
      public long get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return producer.applyAsLong(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get a long indexable of a given size which returns elements created by index.
   * This will call the producer only once for each element during the call of this method,
   * and return the results on later requests.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable with elements initialized by the {@code producer}
   * @see #viewByIndex(int, IntToLongFunction)
   */
  @NotNull
  static LongIndexable initByIndex(int size, @NotNull IntToLongFunction producer)
  {
    return viewByIndex(size, producer).frozen();
  }

  /**
   * An empty indexable.
   */
  @NotNull
  Base EMPTY = new Base()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public long get(int index)
    {
      throw new IndexOutOfBoundsException("Empty indexable has no elements!");
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public long[] toArray()
    {
      return Empty.LONG_ARRAY;
    }

    @Override
    public int addToArray(@NotNull long[] array, int pos)
    {
      return pos;
    }

    @Override
    public int addToArray(@NotNull long[] array, int arrayPos, int index, int length)
    {
      return arrayPos;
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return IntIndexable.EMPTY;
    }

    @Override
    public long foldLeft(long initialValue, @NotNull LongBinaryOperator foldOperation)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public LongIndexable.Base reverse()
    {
      return this;
    }

    @NotNull
    @Override
    public LongIndexable rotated(int steps)
    {
      return this;
    }

    @NotNull
    @Override
    public Iterator<Long> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public ListIterator<Long> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public PrimitiveIterator.OfLong longIterator()
    {
      return Types.EMPTY_LONG_ITERATOR;
    }

    @Override
    public int compareTo(@NotNull LongIndexable o)
    {
      return o.isEmpty() ? 0 : -1;
    }

    @Override
    public String toString()
    {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }

    @Override
    public boolean equals(Object obj)
    {
      return (obj instanceof LongIndexable  &&  ((LongIndexable)obj).isEmpty());
    }

    @NotNull
    @Override
    public Collection<Long> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Long> asList()
    {
      return Collections.emptyList();
    }

    @Override
    public void forEachLong(@NotNull LongConsumer consumer)
    {
    }

    @Override
    public boolean containsLong(long value)
    {
      return false;
    }

    @NotNull
    @Override
    public Base frozen()
    {
      return this;
    }

    @Override
    public <E> Indexable<E> view(@NotNull LongFunction<E> convert)
    {
      return Indexable.emptyIndexable();
    }

    @Override
    public void forEach(Consumer<? super Long> action)
    {
    }

    @Override
    public long sum()
    {
      return 0;
    }

    @Override
    public long sumX()
    {
      return 0;
    }

    @NotNull
    @Override
    public OptionalDouble average()
    {
      return OptionalDouble.empty();
    }

    @NotNull
    @Override
    public Spliterator.OfLong longSpliterator()
    {
      return Spliterators.emptyLongSpliterator();
    }
  };

  /**
   * Get an empty indexable set.
   * @return empty indexable
   */
  @NotNull
  static Base emptyIndexable()
  {
    return EMPTY;
  }

  /**
   * Create a string representation of the given indexable.
   * @param indexable indexable
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull LongIndexable indexable)
  {
    if (indexable.isEmpty()) {
      return "[]";
    }
    final StringBuilder sb = new StringBuilder();
    sb.append('[').append(indexable.get(0));
    for (PrimitiveIterator.OfLong it = indexable.tailSet(1).longIterator();  it.hasNext();  ) {
      sb.append(',').append(it.nextLong());
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Are two long indexables equal?
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equal(@NotNull LongIndexable indexable1,
                       @NotNull LongIndexable indexable2)
  {
    if (indexable1 == indexable2) {
      return true;
    }
    if (indexable1.size() != indexable2.size()) {
      return false;
    }
    final PrimitiveIterator.OfLong it1 = indexable1.longIterator();
    final PrimitiveIterator.OfLong it2 = indexable2.longIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      if (it1.nextLong() != it2.nextLong()) {
        return false;
      }
    }
    return !(it1.hasNext()  ||  it2.hasNext());
  }

  /**
   * Are two long indexables equal?
   * This is a convenience method which may be called from
   * implementations for their {@link Object#equals(Object)}
   * method.
   * @param indexable1  first indexable
   * @param indexable2  object expected to be an indexable itself
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if the second object is not an indexable, of if sizes or values differ
   */
  static boolean equal(@NotNull LongIndexable indexable1,
                       @Nullable Object indexable2)
  {
    if (indexable2 instanceof LongIndexable) {
      return equal(indexable1, (LongIndexable)indexable2);
    }
    return false;
  }

  /**
   * Compare two long indexables.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compare(@NotNull LongIndexable indexable1,
                     @NotNull LongIndexable indexable2)
  {
    final PrimitiveIterator.OfLong it1 = indexable1.longIterator();
    final PrimitiveIterator.OfLong it2 = indexable2.longIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Long.compare(it1.nextLong(),
                                        it2.nextLong());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Compare two long indexables as if they contain unsigned long values.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compareUnsigned(@NotNull LongIndexable indexable1,
                             @NotNull LongIndexable indexable2)
  {
    final PrimitiveIterator.OfLong it1 = indexable1.longIterator();
    final PrimitiveIterator.OfLong it2 = indexable2.longIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Long.compareUnsigned(it1.nextLong(),
                                               it2.nextLong());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Calculate a hashcode for a long indexable.
   * @param indexable indexable for which the hash code is required
   * @return hash code for the given indexable
   */
  static int hash(@NotNull LongIndexable indexable)
  {
    // this follows Arrays.hashCode()
    int result = 1;

    for (PrimitiveIterator.OfLong it = indexable.longIterator();  it.hasNext();  ) {
      result = 31 * result + Long.hashCode(it.next());
    }

    return result;
  }

  /**
   * Wrap an indexable with one which caches the hash value.
   * This is useful if indexables are used as hash keys because hash value calculation
   * is expensive.
   * <p>
   * The wrapped indexable must not change after it is wrapped, otherwise strange things are expected to happen.
   * @param indexable wrapped indexable, must not change
   * @return indexable which forwards most methods to {@code indexable}, but also provides
   *         useful implementations for {@link Object#hashCode()}, {@link Object#equals(Object)},
   *         and {@link Object#toString()}
   */
  @NotNull
  static LongIndexable.Base withCachedHash(@NotNull LongIndexable indexable)
  {
    final int hashCode = hash(indexable);
    return new LongIndexable.Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public long get(int index)
      {
        return indexable.get(index);
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfLong longIterator()
      {
        return indexable.longIterator();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfLong longIterator(int from, int to)
      {
        return indexable.longIterator(from, to);
      }

      @NotNull
      @Override
      public Iterator<Long> iterator()
      {
        return indexable.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Long> listIterator()
      {
        return indexable.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return indexable.subSet(fromIndex, toIndex);
      }

      @Override
      public int addToArray(@NotNull long[] array, int pos)
      {
        return indexable.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull long[] array, int arrayPos, int index, int length)
      {
        return indexable.addToArray(array, arrayPos, index, length);
      }

      @Override
      public int hashCode()
      {
        return hashCode;
      }

      @Override
      public boolean equals(Object obj)
      {
        return equal(indexable, obj);
      }

      @Override
      public String toString()
      {
        return LongIndexable.toString(indexable);
      }
    };
  }

  /**
   * View this raw long indexable as if it has another element type.
   * @param convert converter between long and the requested element type
   * @param <E> element type of return view
   * @return view of this raw long indexable
   */
  default <E> Indexable<E> view(@NotNull LongFunction<E> convert)
  {
    return new Indexable.Base<E>()
    {
      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }

      @Override
      public E get(int index)
      {
        return convert.apply(LongIndexable.this.get(index));
      }
    };
  }

  /**
   * Is this indexable sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * See {@linkplain #isStrictlyOrdered(LongOrdering)} for a stricter alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isOrdered(@NotNull LongOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    long last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final long value = get(i);
      if (order.checkLong(value, last) == Descending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered(LongOrdering)} for a more relaxed alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered(@NotNull LongOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    long last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final long value = get(i);
      if (order.checkLong(value, last) != Ascending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Do a binary search in an indexable ordered in natural ascending order.
   * This requires this indexable to be ordered in
   * {@link LongOrdering#ASCENDING natural ascending order},
   * i.e. {@linkplain #isOrdered(LongOrdering)} has to return {@code true}
   * for this ordering.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(LongOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @return a positive integer defining the index where the given value was found,
   *         and a negative integer if the value is not contained. The latter defines
   *         the insertion point where the looked up value would match into this
   *         sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(long value)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final long dp = get(mid);

      if (dp < value) {
        low = mid + 1;
      }
      else if (dp > value) {
        hi = mid - 1;
      }
      else {
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Do a binary search in an ordered indexable.
   * This requires this indexable to be ordered in non-descending order
   * as defined by the given {@code order},
   * i.e. {@linkplain #isOrdered(LongOrdering)} has to return {@code true}
   * for this order.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(LongOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @param order sort order
   * @return a positive integer defining the index where the given value was found,
   * and a negative integer if the value is not contained. The latter defines
   * the insertion point where the looked up value would match into this
   * sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(long value, @NotNull LongOrdering order)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final long dp = get(mid);

      switch (order.checkLong(dp, value)) {
      case Ascending:
        low = mid + 1;
        break;
      case Descending:
        hi = mid - 1;
        break;
      default:
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Create a (mutable) long indexable from the values of this indexable
   * which is ordered as defined by the given ordering.
   * <p>
   * If natural ordering is required prefer calling {@link #ordered()}
   * because that typically runs 1.5 times faster.
   * @param order sort order
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableLongIndexable ordered(@NotNull LongOrdering order)
  {
    final MutableLongIndexable result = MutableLongIndexable.fromLongIndexable(this);
    result.order(order);
    return result;
  }

  /**
   * Create a (mutable) long indexable from the values of this indexable
   * which is ordered in natural order.
   * <p>
   * This can be some 1.5x faster than using {@link #ordered(LongOrdering)}
   * with {@link LongOrdering#ASCENDING natural order}.
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableLongIndexable ordered()
  {
    final MutableLongIndexable result = MutableLongIndexable.fromLongIndexable(this);
    result.order();
    return result;
  }

  /**
   * Make this long integer indexable usable as a class with defined standard Object methods.
   * @return based version of this indexable
   */
  @NotNull
  default Base asBase()
  {
    return new LongIndexable.Base() {
      @Override
      public int size()
      {
        return LongIndexable.this.size();
      }

      @Override
      public long get(int index)
      {
        return LongIndexable.this.get(index);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return LongIndexable.this.reverse();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfLong longIterator()
      {
        return LongIndexable.this.longIterator();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfLong longIterator(int from, int to)
      {
        return LongIndexable.this.longIterator(from ,to);
      }

      @NotNull
      @Override
      public Iterator<Long> iterator()
      {
        return LongIndexable.this.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Long> listIterator()
      {
        return LongIndexable.this.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return LongIndexable.this.subSet(fromIndex, toIndex);
      }

      @NotNull
      @Override
      public long[] toArray()
      {
        return LongIndexable.this.toArray();
      }

      @Override
      public int addToArray(@NotNull long[] array, int pos)
      {
        return LongIndexable.this.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull long[] array, int arrayPos, int index, int length)
      {
        return LongIndexable.this.addToArray(array, arrayPos, index, length);
      }
    };
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}. It also provides
   * {@link Comparable#compareTo(Object)}.
   * @see LongIndexable#asBase()
   */
  abstract class Base implements LongIndexable,
                                 Comparable<LongIndexable>
  {
    @Override
    public int compareTo(@NotNull LongIndexable o)
    {
      Objects.requireNonNull(o);
      return compare(this, o);
    }

    @Override
    public int hashCode()
    {
      return hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      return equal(this, obj);
    }

    @Override
    public String toString()
    {
      return LongIndexable.toString(this);
    }

    @NotNull
    @Override
    public Base asBase()
    {
      return this;
    }
  }
}
