// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.ListIterator;
import java.util.function.Function;

/**
 * A list iterator which maps the returned value.
 * <p>
 * It does not support setting or adding.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Mai.25, 2016
 * @param <T> target type
 * @param <S> source type
 */
public class ListIteratorConverter<T, S>
        implements ListIterator<T>
{
  @NotNull
  private final ListIterator<S> sourceIterator;
  @NotNull
  private final Function<? super S, T> mapper;

  /**
   * Constructor.
   * @param sourceIterator  source iterator
   * @param mapper          mapper from source to target type
   */
  public ListIteratorConverter(@NotNull ListIterator<S> sourceIterator, @NotNull Function<? super S, T> mapper)
  {
    this.sourceIterator = sourceIterator;
    this.mapper = mapper;
  }

  /**
   * Returns {@code true} if this list iterator has more elements when
   * traversing the list in the forward direction. (In other words,
   * returns {@code true} if {@link #next} would return an element rather
   * than throwing an exception.)
   *
   * @return {@code true} if the list iterator has more elements when
   * traversing the list in the forward direction
   */
  @Override
  public boolean hasNext()
  {
    return sourceIterator.hasNext();
  }

  /**
   * Returns the next element in the list and advances the cursor position.
   * This method may be called repeatedly to iterate through the list,
   * or intermixed with calls to {@link #previous} to go back and forth.
   * (Note that alternating calls to {@code next} and {@code previous}
   * will return the same element repeatedly.)
   *
   * @return the next element in the list
   * @throws java.util.NoSuchElementException if the iteration has no next element
   */
  @Override
  public T next()
  {
    return mapper.apply(sourceIterator.next());
  }

  /**
   * Returns {@code true} if this list iterator has more elements when
   * traversing the list in the reverse direction.  (In other words,
   * returns {@code true} if {@link #previous} would return an element
   * rather than throwing an exception.)
   *
   * @return {@code true} if the list iterator has more elements when
   * traversing the list in the reverse direction
   */
  @Override
  public boolean hasPrevious()
  {
    return sourceIterator.hasPrevious();
  }

  /**
   * Returns the previous element in the list and moves the cursor
   * position backwards.  This method may be called repeatedly to
   * iterate through the list backwards, or intermixed with calls to
   * {@link #next} to go back and forth.  (Note that alternating calls
   * to {@code next} and {@code previous} will return the same
   * element repeatedly.)
   *
   * @return the previous element in the list
   * @throws java.util.NoSuchElementException if the iteration has no previous
   *                                element
   */
  @Override
  public T previous()
  {
    return mapper.apply(sourceIterator.previous());
  }

  /**
   * Returns the index of the element that would be returned by a
   * subsequent call to {@link #next}. (Returns list size if the list
   * iterator is at the end of the list.)
   *
   * @return the index of the element that would be returned by a
   * subsequent call to {@code next}, or list size if the list
   * iterator is at the end of the list
   */
  @Override
  public int nextIndex()
  {
    return sourceIterator.nextIndex();
  }

  /**
   * Returns the index of the element that would be returned by a
   * subsequent call to {@link #previous}. (Returns -1 if the list
   * iterator is at the beginning of the list.)
   *
   * @return the index of the element that would be returned by a
   * subsequent call to {@code previous}, or -1 if the list
   * iterator is at the beginning of the list
   */
  @Override
  public int previousIndex()
  {
    return sourceIterator.previousIndex();
  }

  /**
   * Removes from the list the last element that was returned by {@link
   * #next} or {@link #previous} (optional operation).  This call can
   * only be made once per call to {@code next} or {@code previous}.
   * It can be made only if {@link #add} has not been
   * called after the last call to {@code next} or {@code previous}.
   *
   * @throws UnsupportedOperationException if the {@code remove}
   *                                       operation is not supported by this list iterator
   * @throws IllegalStateException         if neither {@code next} nor
   *                                       {@code previous} have been called, or {@code remove} or
   *                                       {@code add} have been called after the last call to
   *                                       {@code next} or {@code previous}
   */
  @Override
  public void remove()
  {
    sourceIterator.remove();
  }

  /**
   * Replaces the last element returned by {@link #next} or
   * {@link #previous} with the specified element (optional operation).
   * This call can be made only if neither {@link #remove} nor {@link
   * #add} have been called after the last call to {@code next} or
   * {@code previous}.
   *
   * @param t the element with which to replace the last element returned by
   *          {@code next} or {@code previous}
   * @throws UnsupportedOperationException if the {@code set} operation
   *                                       is not supported by this list iterator
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list
   * @throws IllegalArgumentException      if some aspect of the specified
   *                                       element prevents it from being added to this list
   * @throws IllegalStateException         if neither {@code next} nor
   *                                       {@code previous} have been called, or {@code remove} or
   *                                       {@code add} have been called after the last call to
   *                                       {@code next} or {@code previous}
   */
  @Override
  public void set(T t)
  {
    throw new UnsupportedOperationException("No set() for wrapping list iterator!");
  }

  /**
   * Inserts the specified element into the list (optional operation).
   * The element is inserted immediately before the element that
   * would be returned by {@link #next}, if any, and after the element
   * that would be returned by {@link #previous}, if any.  (If the
   * list contains no elements, the new element becomes the sole element
   * on the list.)  The new element is inserted before the implicit
   * cursor: a subsequent call to {@code next} would be unaffected, and a
   * subsequent call to {@code previous} would return the new element.
   * (This call increases by one the value that would be returned by a
   * call to {@code nextIndex} or {@code previousIndex}.)
   *
   * @param t the element to insert
   * @throws UnsupportedOperationException if the {@code add} method is
   *                                       not supported by this list iterator
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list
   * @throws IllegalArgumentException      if some aspect of this element
   *                                       prevents it from being added to this list
   */
  @Override
  public void add(T t)
  {
    throw new UnsupportedOperationException("No add() for wrapping list iterator!");
  }
}
