// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.util.Counter;
import de.caff.generics.util.ThreadSafeCounter;

import java.lang.ref.SoftReference;
import java.util.*;
import java.util.function.Function;

/**
 * A map which implements a least recently used cache.
 * <p>
 * Usage here means that the latest value which is either add by {@link #put(Object, Object)}
 * or retrieved by {@link #get(Object)} is the least recently used one.
 * <p>
 * The implemented map basically keeps only soft references to its values, but not its keys.
 * Therefore keys and values must be of different types, otherwise store values will not
 * be freed, ever.
 * <p>
 * This does deliberately not implement {@link java.util.Map} as that has too many methods
 * mot making much sense here.
 * <p>
 * Attention: {@code null} values are not allowed for this map, as {@code null} is used to
 *             mark valeus which are removed by the garbage collector
 * <p>
 * See {@link RecreatingLeastRecentlyUsedCache} for a way to include an automatic recreation
 * of garbage-collected values.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since May 10, 2021
 * @param <K> key type of map, as always keys are expected to not mutate during the lifetime of this object
 * @param <V> value type of map
 */
public class LeastRecentlyUsedCache<K, V>
{
  private final int leastRecentLimit;
  @NotNull
  protected final Map<K, SoftReference<V>> map;
  @NotNull
  private final LinkedList<V> leastRecentCache = new LinkedList<>();
  @NotNull
  private final Counter numHits = new ThreadSafeCounter();
  @NotNull
  private final Counter numGarbageCollected = new ThreadSafeCounter();
  @NotNull
  private final Counter numMisses = new ThreadSafeCounter();

  /**
   * Constructor.
   * This just keeps up to {@code leastRecentLimit} values cached.
   * It does not define a recreator, therefore values removed by
   * garbage collection are gone, and have to be recreated and
   * readded to this maps elsewhere.
   * @param leastRecentLimit maximal number of cached values, non-negative
   */
  public LeastRecentlyUsedCache(int leastRecentLimit)
  {
    if (leastRecentLimit < 0) {
      throw new IllegalArgumentException("leastRecentLimit has to be non-negative, but is "+leastRecentLimit);
    }
    this.leastRecentLimit = leastRecentLimit;
    this.map = new HashMap<>();
  }

  /**
   * Update the least recent cache.
   * @param value least recent value
   */
  private void updateLeastRecent(@NotNull V value)
  {
    if (leastRecentLimit == 0) {
      return;
    }
    synchronized (leastRecentCache) {
      leastRecentCache.removeIf(v -> Objects.deepEquals(v, value));
      leastRecentCache.add(value);
      if (leastRecentCache.size() > leastRecentLimit) {
        leastRecentCache.removeFirst();
      }
    }
  }

  /**
   * Remove a value from the least recent cache.
   * @param value value to remove
   */
  private void removeFromLeastRecent(@Nullable V value)
  {
    if (value == null  ||  leastRecentLimit == 0) {
      return;
    }
    synchronized (leastRecentCache) {
      leastRecentCache.removeIf(v -> Objects.deepEquals(v, value));
    }
  }

  /**
   * Remove all key-value pairs which are no longer available.
   */
  public void cleanup()
  {
    synchronized (map) {
      for (Map.Entry<K, SoftReference<V>> entry : new HashSet<>(map.entrySet())) {
        if (entry.getValue().get() == null) {
          numGarbageCollected.add1();
          map.remove(entry.getKey());
        }
      }
    }
  }

  /**
   * Get the size of this mapping.
   * This implicitly calls {@link #cleanup()} before returning the size,
   * but as garbage collection can happen any time users should not depend
   * too much on the returned value.
   * @return number of currently cached values
   */
  public int size()
  {
    synchronized (map) {
      cleanup();
      return map.size();
    }
  }

  /**
   * Get the maximum number of values kept in the LRU cache.
   * This cache may keep a greater number of values as long as there is enough memory.
   * @return cache limit
   */
  public int getLeastRecentCacheDepth()
  {
    return leastRecentLimit;
  }

  /**
   * Is this mapping empty?
   * This implicitly calls {@link #cleanup()} before returning the answer
   * but as garbage collection can happen any time users should not depend
   * too much on the returned value.
   * @return number of currently cached values
   */
  public boolean isEmpty()
  {
    synchronized (map) {
      cleanup();
      return map.isEmpty();
    }
  }

  /**
   * Get the value for the given key.
   *
   * @param key key associated with the requested value
   * @return {@code null} when the required value was never added to this cache
   *          or when it was garbage collected in the meantime, otherwise the cached object
   *          associated with the given key
   */
  @Nullable
  public V get(@NotNull K key)
  {
    final V value;
    synchronized (map) {
      final SoftReference<V> ref = map.get(key);
      if (ref == null) {
        numMisses.add1();
        return null;
      }
      value = ref.get();
      if (value == null) {
        assert !leastRecentCache.contains(value);
        numGarbageCollected.add1();
        map.remove(key);
      }
    }
    if (value != null) {
      numHits.add1();
      updateLeastRecent(value);
    }
    return value;
  }

  /**
   * Put a new key-value pair into this cache.
   * @param key    key for retrieving the cached value
   * @param value  value to cache
   */
  public void put(K key, @NotNull V value)
  {
    Objects.requireNonNull(value);
    final SoftReference<V> oldRef;
    synchronized(map) {
      oldRef = map.put(key, new SoftReference<>(value));
    }
    if (oldRef != null) {
      removeFromLeastRecent(oldRef.get());
    }
    updateLeastRecent(value);
  }

  /**
   * Get the value for the given key, or compute a new value and add it for
   * the given key.
   * @param key     key of requested value
   * @param creator creator for new value if key is (no longer) present
   * @return stored or newly created vale, {@code null} only if creator returns {@code null}
   */
  public V computeIfAbsent(K key,
                           @NotNull Function<? super K, ? extends V> creator)
  {
    synchronized(map) {
      final V value = get(key);
      if (value != null) {
        return value;
      }
      final V newValue = creator.apply(key);
      if (newValue != null) {
        put(key, newValue);
      }
      return newValue;
    }
  }

  /**
   * Get the keys currently valid in this cache.
   * This calls {@link #cleanup()} before the keys are returned.
   * This is just a snapshot, don't depend too much on the returned keys.
   * @return set of currently valid keys
   */
  @NotNull
  public Set<K> keys()
  {
    synchronized (map) {
      cleanup();
      return map.keySet();
    }
  }

  /**
   * Remove a key-value pair from this cache.
   * @param key key of the key-value pair
   * @return removed object if it was still cached,
   *         or {@code null} if the object was either never cached at all
   *         or has been garbage collected
   */
  @Nullable
  public V remove(K key)
  {
    final SoftReference<V> ref;
    synchronized (map) {
      ref = map.remove(key);
    }
    if (ref != null) {
      final V value = ref.get();
      removeFromLeastRecent(value);
      return value;
    }
    return null;
  }

  /**
   * Remove all entries from this cache.
   */
  public void clear()
  {
    synchronized (map) {
      map.clear();
    }
    synchronized (leastRecentCache) {
      leastRecentCache.clear();
    }
  }

  /**
   * Get the values in the LRU cache in order.
   * @return values in the LRU cache, most least recently used item first
   */
  @NotNull
  public List<V> getLeastRecentlyUsed()
  {
    if (leastRecentLimit == 0) {
      return Collections.emptyList();
    }
    final List<V> result;
    synchronized (leastRecentCache) {
      result = new ArrayList<>(leastRecentCache);
    }
    Collections.reverse(result);
    return result;
  }

  /**
   * Get the number of times a cached value was successfully retrieved.
   * @return number of cache hits
   */
  public int getNumHits()
  {
    return numHits.getValue();
  }

  /**
   * Get the number of times an entry was garbage collected.
   * @return number of garbage collected entries
   */
  public int getNumGarbageCollected()
  {
    return numGarbageCollected.getValue();
  }

  /**
   * Get the number of times an entry was not found in the cache.
   * @return number of cache misses
   */
  public int getNumMisses()
  {
    return numMisses.getValue();
  }
}
