// ============================================================================
// File:               Lazy
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2025  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            4/5/25 12:45 PM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FragileFunction0;
import de.caff.generics.util.IReference;
import de.caff.generics.util.ReferenceType;

import java.util.function.Supplier;

/**
 * Support for lazy evaluation.
 * <p>
 * The underlying value is only created if the {@link #get()} method is called the first time.
 * Later calls will return the value created on the first call.
 *
 * @param <T> result type of the evaluation
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since April 05, 2025
 */
public interface Lazy<T>
        extends Supplier<T>
{
  /**
   * Create a lazy evaluation from a supplier.
   * Using this method, the value is stored after its first
   * creation, and just returned on further calls of {@link #get()}.
   *
   * @param supplier The supplier which creates the value.
   *                 Will only be called at most once.
   * @param <V> value type the lazy evaluation will provide
   * @return lazy evaluation object
   */
  @NotNull
  static <V> Lazy<V> from(@NotNull Supplier<? extends V> supplier)
  {
    return new Lazy<V>()
    {
      /** Is the value initialized? */
      private boolean initialized = false;
      /** The value. */
      private V value;

      public synchronized V get()
      {
        if (!initialized) {
          value = supplier.get();
          initialized = true;
        }
        return value;
      }
    };
  }

  /**
   * Create a lazy evaluation from a fragile supplier.
   * Note that in case the supplier throws the {@link #get()} method of the returned
   * {@code Lazy} will throw a {@link LazyEvalutionFailure}.
   *
   * @param fragileSupplier supplier which might throw a checked exception
   * @param <V>             value type of the returned lazy evaluator
   * @return lazy evaluator
   */
  @NotNull
  static <V> Lazy<V> fromFragile(@NotNull FragileFunction0<? extends V, ?> fragileSupplier)
  {
    return from(wrapFragile(fragileSupplier));
  }

  /**
   * Helper method allowing to define the type of the reference
   * which holds the value after it is provided by the supplier.
   * @param referenceType reference type used for keeping the value
   * @param supplier      the value supplier, which might be called more than once depending on {@code referenceType}
   * @param <V>           value type of the returned lazy evaluator
   * @return lazy evaluator
   */
  @NotNull
  static <V> Lazy<V> from(@NotNull ReferenceType referenceType,
                          @NotNull Supplier<? extends V> supplier)
  {
    if (referenceType == ReferenceType.Hard) {
      return from(supplier);
    }

    return new Lazy<V>()
    {
      private IReference<V> reference;

      @Override
      public synchronized V get()
      {
        if (reference != null) {
          final V v = reference.get();
          if (v != null) {
            return v;
          }
          else if (reference.isNull()) {
            return null;
          }
        }
        // value was either not yet created or GC'ed
        final V value = supplier.get();
        reference = value == null
                ? IReference.nullRef()
                : referenceType.makeReference(value);
        return value;
      }
    };
  }

  /**
   * Create a lazy evaluation from a fragile supplier,
   * but reference the value internally only via a soft reference.
   * <p>
   * This allows the value to be garbage-collected when it is no longer
   * used elsewhere, with the cost of creating it again if it is requested
   * afterward.
   *
   * @param supplier supplier which is possibly called more than once,
   *                 so it should be idempotent
   * @return a lazy evaluator which keeps its value via a soft reference
   * @param <V>      value type of the returned lazy evaluator
   * @see java.lang.ref.SoftReference
   */
  @NotNull
  static <V> Lazy<V> softFrom(@NotNull Supplier<? extends V> supplier)
  {
    return from(ReferenceType.Soft, supplier);
  }

  /**
   * Create a lazy evaluation from a fragile supplier,
   * but reference the value internally only via a soft reference.
   * <p>
   * This allows the value to be garbage-collected when it is no longer
   * used elsewhere, with the cost of creating it again if it is requested
   * afterward.
   *
   * @param fragileSupplier supplier which is possibly called more than once,
   *                        so it should be idempotent
   * @return a lazy evaluator which keeps its value via a soft reference
   * @param <V>      value type of the returned lazy evaluator
   * @see java.lang.ref.SoftReference
   */
  @NotNull
  static <V> Lazy<V> softFromFragile(@NotNull FragileFunction0<? extends V, ?> fragileSupplier)
  {
    return from(ReferenceType.Soft, wrapFragile(fragileSupplier));
  }

  /**
   * Create a lazy evaluation from a fragile supplier,
   * but reference the value internally only via a weak reference.
   * <p>
   * This allows the value to be garbage-collected when it is no longer
   * used elsewhere, with the cost of creating it again if it is requested
   * afterward.
   *
   * @param supplier supplier which is possibly called more than once,
   *                 so it should be idempotent
   * @return a lazy evaluator which keeps its value via a weak reference
   * @param <V>      value type of the returned lazy evaluator
   * @see java.lang.ref.WeakReference
   */
  @NotNull
  static <V> Lazy<V> weakFrom(@NotNull Supplier<? extends V> supplier)
  {
    return from(ReferenceType.Weak, supplier);
  }

  /**
   * Create a lazy evaluation from a fragile supplier,
   * but reference the value internally only via a weak reference.
   * <p>
   * This allows the value to be garbage-collected when it is no longer
   * used elsewhere, with the cost of creating it again if it is requested
   * afterward.
   *
   * @param fragileSupplier supplier which is possibly called more than once,
   *                        so it should be idempotent
   * @return a lazy evaluator which keeps its value via a weak reference
   * @param <V>      value type of the returned lazy evaluator
   * @see java.lang.ref.WeakReference
   */
  @NotNull
  static <V> Lazy<V> weakFromFragile(@NotNull FragileFunction0<? extends V, ?> fragileSupplier)
  {
    return from(ReferenceType.Weak, wrapFragile(fragileSupplier));
  }

  /**
   * Wrap a fragile supplier to be fragile no longer.
   * This wraps a supplier which might throw a checked exception
   * with one which will throw an unchecked exception instead. 
   * @param fragileSupplier fragile supplier
   * @return supplier which will throw a {@link LazyEvalutionFailure} if
   *         the original {@code fragileSupplier} throws an exception
   * @param <V> type of supplied value
   */
  @NotNull
  static <V> Supplier<V> wrapFragile(@NotNull FragileFunction0<? extends V, ?> fragileSupplier)
  {
    return () -> {
      try {
        return fragileSupplier.apply();
      } catch (Exception e) {
        throw new LazyEvalutionFailure(e);
      }
    };
  }
}
