// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.FragileIntConsumer;
import de.caff.generics.function.IntOrdering;
import de.caff.generics.function.IntToCharFunction1;
import de.caff.generics.range.Range;

import java.util.*;
import java.util.function.*;

import static de.caff.generics.Order.Ascending;
import static de.caff.generics.Order.Descending;

/**
 * Class which allows readonly access of raw integers by index.
 * <p>
 * The {@link #EMPTY} constant or {@link #emptyIndexable()} method provide the same
 * useful return value to indicate emptiness.
 * <p>
 * Simple implementations should extend {@link Base} because
 * that provides useful implementations for standard Object methods and implements
 * {@link Comparable}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 7, 2020
 * @see de.caff.generics.algorithm.FastIntSequenceSearch
 */
public interface IntIndexable
        extends IntCountable
{
  /**
   * Get the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @return element at the given index
   */
  int get(int index);

  /**
   * Pythonesque get.
   * This allows accessing elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @return element at the given index
   */
  default int gyt(int index)
  {
    return get(Pythonesque.mapX(index, this));
  }

  /**
   * Get an element modulo size.
   * <p>
   * This maps the given {@code index} into the range of this indexable
   * by applying a modulo {@link #size()} operation. For empty indexable
   * this will throw an {@link IndexOutOfBoundsException} as there is no
   * possible index to get.
   * @param index index, possibly out of range, possibly even negative
   *              for Pythonesque access
   * @return element at the given, possibly modulated, index
   */
  default int getMod(int index)
  {
    final int size = size();
    if (size == 0) {
      throw new IndexOutOfBoundsException("No element for empty indexable!");
    }
    return gyt(index % size);
  }

  /**
   * Return a new integer indexable where the given index is changed.
   * This default implementation will copy all values, so this should be used
   * for small arrays preferably.
   * @param index index between {@code 0} and {@code size() - 1}
   * @param value value to be set
   * @return new integer indexable with changed value
   */
  @NotNull
  default Base with(int index, int value)
  {
    final int[] array = toArray();
    array[index] = value;
    return viewArray(array);
  }

  /**
   * Return a new integer indexable where the given index is changed
   * (Pythonesque version)
   * This allows to index elements from the end by using negative indexes.
   * This default implementation will forward everything to
   * {@link #with(int, int)} which by default copies all values, so this
   * should be used for small arrays preferably.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @param value value to be set
   * @return new integer indexable with changed value
   */
  @NotNull
  default Base wyth(int index, int value)
  {
    if (index > 0) {
      return with(index, value);
    }
    final int realIndex = index + size();
    if (realIndex < 0) {
      throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                         index, size()));
    }
    return with(realIndex, value);
  }

  /**
   * Get this indexable but with inverted order.
   * @return a view to this indexable which accesses the elements in reverse order
   */
  @NotNull
  default Base reverse()
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }

      @Override
      public int get(int index)
      {
        return IntIndexable.this.get(IntIndexable.this.size() - index - 1);
      }

      @NotNull
      @Override
      public IntIndexable.Base reverse()
      {
        return IntIndexable.this.asBase();
      }
    };
  }

  /**
   * Is this indexable empty?
   * @return {@code true}: there are no elements in this indexable<br>
   *         {@code false}: this indexable has elements
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * Get a primitive int iterator.
   * @return int iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveIterator.OfInt intIterator()
  {
    return intIterator(0, size());
  }

  /**
   * Get an int iterator from the given sub set.
   * @param from first index of iterator
   * @param to   one past last index of iterator
   * @return int iterator which iterates over the given part of this indexable
   */
  @NotNull
  default PrimitiveIterator.OfInt intIterator(final int from, final int to)
  {
    return new PrimitiveIterator.OfInt()
    {
      private int index = from;

      @Override
      public int nextInt()
      {
        if (index >= to) {
          throw new NoSuchElementException(String.format("Index %s out of allowed range [%d, %d[!",
                                                         index, from, to));
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < to;
      }
    };
  }

  /**
   * Call an entry consumer foreach entry in this indexable.
   * @param consumer consumer to be called
   */
  default void forEachIntEntry(@NotNull EntryConsumer consumer)
  {
    for (int idx = 0;  idx < size();  ++idx) {
      consumer.accept(idx, get(idx));
    }
  }

  /**
   * Returns an iterator over elements of type {@code T}.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  default Iterator<Integer> iterator()
  {
    return listIterator();
  }

  /**
   * Returns a list iterator over elements of type {@code T}.
   *
   * @return a list iterator.
   */
  @NotNull
  default ListIterator<Integer> listIterator()
  {
    return new ListIterator<Integer>() {
      private int index = 0;

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public Integer next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public Integer previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(--index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(Integer t)
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void add(Integer t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  /**
   * Get an indexable subset.
   * The subset includes indices {@code fromIndex} to {@code toIndex - 1}.
   * @param fromIndex start index of sub set
   * @param toIndex   index after last index
   * @return indxable subset view of this indexable
   */
  @NotNull
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0)
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    if (toIndex > size())
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    if (fromIndex > toIndex)
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    final int length = toIndex - fromIndex;
    return length == 0
            ? EMPTY
            : new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public int get(int index)
      {
        if (index >= length) {
          throw new IndexOutOfBoundsException("No such element: "+index);
        }
        return IntIndexable.this.get(index + fromIndex);
      }

      @Override
      public int addToArray(@NotNull int[] array, int arrayPos, int index, int length)
      {
        return IntIndexable.this.addToArray(array, arrayPos, index + fromIndex, length);
      }

      @NotNull
      @Override
      public IntIndexable.Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return IntIndexable.this.subSet(fromIndex + fromIdx,
                                        fromIndex + toIdx);
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator(int from, int to)
      {
        return IntIndexable.this.intIterator(from + fromIndex,
                                             to + fromIndex);
      }
    };
  }

  /**
   * Get an indexable subset.
   * This is the Pythonesque version which allows negative indexes.
   * @param fromIndex start index of sub set
   * @param toIndex   end index of sub set
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param fromIndex index to start with, negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param toIndex index one after the end (equal to the length of the returned set),
   *                negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Convert this into an integer array.
   * @return integer array
   */
  @NotNull
  default int[] toArray()
  {
    if (isEmpty()) {
      return Empty.INT_ARRAY;
    }
    final int[] result = new int[size()];
    addToArray(result, 0);
    return result;
  }

  /**
   * Add the content of this indexable to the given array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull int[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull int[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveIterator.OfInt it = intIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextInt();
    }
    return arrayPos;
  }

  /**
   * Get a view on this indexable as an unmodifiable collection.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   */
  @NotNull
  default Collection<Integer> asCollection()
  {
    return new AbstractCollection<Integer>()
    {
      @NotNull
      @Override
      public Iterator<Integer> iterator()
      {
        return IntIndexable.this.iterator();
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable list.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   * @see #toList()
   */
  @NotNull
  default List<Integer> asList()
  {
    return new AbstractList<Integer>()
    {
      @Override
      public Integer get(int index)
      {
        return IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }

      @NotNull
      @Override
      public Iterator<Integer> iterator()
      {
        return IntIndexable.this.iterator();
      }
    };
  }

  /**
   * Create a list from this integer indexable.
   * This creates an independent list to which the elements of this
   * indexable are copied.
   * @return list copy of this indexable
   * @see #asList()
   */
  @NotNull
  default ArrayList<Integer> toList()
  {
    final ArrayList<Integer> list = new ArrayList<>(size());
    addAllTo(list);
    return list;
  }

  /**
   * Add all elements of this indexable to the given collection.
   * @param collection collection to add to
   */
  default void addAllTo(@NotNull Collection<? super Integer> collection)
  {
    for (Integer entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Get a view of this indexable as a standard object-based indexable.
   * @return indexable with non-null {@code Integer} values
   */
  @NotNull
  default Indexable<Integer> asIndexable()
  {
    return new Indexable.Base<Integer>()
    {
      @Override
      public Integer get(int index)
      {
        return IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * Get the indexes of this indexable as an iterable range.
   * @return iterable indexes
   * @see #intIndexes()
   * @see Range#indexes(int) 
   */
  @NotNull
  default Iterable<Integer> indexes()
  {
    return Range.indexes(size());
  }

  /**
   * Get the indexes of this indexable as an integer indexable.
   * @return ordered indexes
   * @see #indexes()
   * @see IntIndexable#rangeFromSize(int)
   */
  @NotNull
  default IntIndexable intIndexes()
  {
    return IntIndexable.rangeFromSize(size());
  }

  /**
   * View this integer indexable as a long indexable.
   * @return long indexable view
   */
  @NotNull
  default LongIndexable.Base asLongIndexable()
  {
    return new LongIndexable.Base()
    {
      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }

      @Override
      public long get(int index)
      {
        return IntIndexable.this.get(index);
      }
    };
  }

  /**
   * Return a mapped view on this integer indexable.
   * @param mapper mapper
   * @param <T> result type pf mapper
   * @return mapped indexable
   */
  @NotNull
  default <T> Indexable<T> view(@NotNull IntFunction<? extends T> mapper)
  {
    return new Indexable.Base<T>()
    {
      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }

      @Override
      public T get(int index)
      {
        return mapper.apply(IntIndexable.this.get(index));
      }
    };
  }

  /**
   * View this integer indexable as a long indexable using unsigned values.
   * @return long indexable view
   */
  @NotNull
  default LongIndexable.Base asUnsignedIndexable()
  {
    return new LongIndexable.Base()
    {
      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }

      @Override
      public long get(int index)
      {
        return IntIndexable.this.get(index) & 0xFFFFFFFFL;
      }
    };
  }

  /**
   * Go over all values and do a cumulative calculation.
   * <p>
   * Eg calculate the maximum of this indexable use
   * <pre>{@code
   *   int max = indexable.foldLeft(Pythonesque.UNMAPPABLE, Math::max);
   * }</pre>
   *
   * @param initialValue  start value for the operation, will be returned if this iterable is empty
   * @param foldOperation operation applied to each value, will get the accumulated value as its
   *                      first and the current element value as its second argument
   * @return accumulated value, result of applying fold operation to all values of this indexable
   *         iteratively
   */
  default int foldLeft(int initialValue,
                       @NotNull IntBinaryOperator foldOperation)
  {
    int result = initialValue;
    for (PrimitiveIterator.OfInt iterator = intIterator();
         iterator.hasNext();  ) {
      result = foldOperation.applyAsInt(result, iterator.nextInt());
    }
    return result;
  }

  /**
   * Get the next index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int nextMatch(int startIndex, @NotNull IntPredicate check)
  {
    final int size = size();
    for (int i = Pythonesque.mapX(startIndex, this);  i < size;  ++i) {
      if (check.test(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the first index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return first index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int firstMatch(@NotNull IntPredicate check)
  {
    return nextMatch(0, check);
  }

  /**
   * Get the previous index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int previousMatch(int startIndex, @NotNull IntPredicate check)
  {
    for (int i = Pythonesque.mapX(startIndex, this);  i >= 0;  --i) {
      if (check.test(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the last index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return last index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int lastMatch(@NotNull IntPredicate check)
  {
    return previousMatch(-1, check);
  }

  /**
   * Get a frozen version of this indexable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying colelction might
   * nevertheless change. This copies the current state of this indexable
   * into an unmodifiable state, and returns an Indexable which is
   * stable in size and will return always the same element for a given index.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, so you can safely call this method more than once.
   *
   * @return frozen version of this Indexable
   */
  @NotNull
  default IntIndexable frozen()
  {
    return IndexableHelper.frozenFromArray(toArray());
  }

  /**
   * Get a spliterator on this indexable.
   * The returned spliterator will not have its {@link Spliterator#IMMUTABLE} flag set,
   * see {@link #frozenIntSpliterator()} for an alternative.
   * @return a spliterator on this indexable
   */
  @NotNull
  default Spliterator.OfInt intSpliterator()
  {
    return new IntIndexableSpliterator(this);
  }

  /**
   * Get an immutable spliterator on a frozen copy of this indexable.
   * If this indexable is already immutable (i.e. if {@link #frozen()} returns {@code this})
   * then this is the same as {@link #spliterator()} with the difference that the spliterator
   * returned here will have its {@link Spliterator#IMMUTABLE} flag set. If this is not immutable
   * a frozen copy will be created, and the spliterator will be operating on that. Please refer to
   * {@link #frozen()} to understand what level of immutabiliy it will provide.
   * @return an "immutable" spliterator with the possible cost of copying this indexable
   */
  @NotNull
  default Spliterator.OfInt frozenIntSpliterator()
  {
    final IntIndexable frozen = frozen();
    return new IntIndexableSpliterator(frozen,
                                       0, frozen.size(),
                                       true);
  }

  /**
   * Spliterator for int indexables.
   */
  class IntIndexableSpliterator implements Spliterator.OfInt
  {
    @NotNull
    private final IntIndexable indexable;
    private int index;
    private final int fence;
    private final int character;

    /**
     * Instantiate a mutable spliterator for an int indexable.
     * @param indexable indexable of this spliterator
     */
    public IntIndexableSpliterator(@NotNull IntIndexable indexable)
    {
      this(indexable, 0, indexable.size(), false);
    }

    /**
     * Instantiate a spliterator for part of an int indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param immutable is the underlying indexable immutable? Use with care!
     */
    public IntIndexableSpliterator(@NotNull IntIndexable indexable,
                                   int start,
                                   int fence,
                                   boolean immutable)
    {
      this(indexable, start, fence,
           immutable
                   ? IMMUTABLE | ORDERED | SIZED | SUBSIZED
                   : ORDERED | SIZED | SUBSIZED);
    }

    /**
     * Instantiate a spliterator for an int indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param character characteristics of this spliterator
     */
    private IntIndexableSpliterator(@NotNull IntIndexable indexable, int start, int fence, int character)
    {
      this.indexable = indexable;
      this.index = start;
      this.fence = fence;
      this.character = character;
    }

    @Override
    public OfInt trySplit()
    {
      final int here = index;
      final int mid = (here + fence) / 2;
      if (here < mid) {
        index = mid;
        return new IntIndexableSpliterator(indexable,
                                           here, mid,
                                           character);
      }
      return null;
    }

    @Override
    public boolean tryAdvance(IntConsumer action)
    {
      if (index < fence) {
        action.accept(indexable.get(index++));
        return true;
      }
      return false;
    }

    @Override
    public long estimateSize()
    {
      return fence - index;
    }

    @Override
    public int characteristics()
    {
      return character;
    }
  }

  /**
   * Create a view of this int indexable with an inserted value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are inserted.
   * Use {@link ExpandableIntIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable might 
   * result in a disaster when using the returned indexable!
   *
   * @param index index where the value is inserted before the current index
   *              (<b>not Pythonesque</b>, because it is possible to add an value at {@code index == size()},
   *              and it is expected that most insertion will happen at 0 or at the end)
   * @param value inserted value at that index
   * @return a view of this indexable with 1 more value at the given index,
   *         all values beyond are moved one index forward
   * @see #withAppendedValue(int)
   */
  @NotNull
  default IntIndexable withInsertedValueAt(int index, int value)
  {
    final int insertIndex = index;
    final int newSize = size() + 1;
    if (insertIndex == 0) {
      return new IntIndexable.Base() {
        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public int get(int index)
        {
          return index == 0
                  ? value
                  : IntIndexable.this.get(index - 1);
        }
      };
    }
    if (insertIndex == newSize - 1) {
      return new IntIndexable.Base() {

        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public int get(int index)
        {
          return index == newSize - 1
                  ? value
                  : IntIndexable.this.get(index);
        }
      };
    }
    return new IntIndexable.Base()
    {
      @Override
      public int size()
      {
        return newSize;
      }

      @Override
      public int get(int index)
      {
        if (index == insertIndex) {
          return value;
        }
        return IntIndexable.this.get(index < insertIndex
                                                ? index
                                                : index - 1);
      }
    };
  }

  /**
   * Create a view with of this indexable with another value added at the end.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are added.
   * @param value value to add
   * @return indexable view of this indexable, with the given value added at the end,
   *         so size is larger by 1
   * @see #withInsertedValueAt(int, int)
   */
  @NotNull
  default IntIndexable withAppendedValue(int value)
  {
    return withInsertedValueAt(size(), value);
  }

  /**
   * Create a view of this int indexable with an exchanged value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are exchanged.
   * Use {@link ExpandableIntIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable could 
   * lead to unexpected results when using the returned indexable!
   *
   * @param index index where the element is exchanged, hiding the current element.
   *              (Pythonesque)
   * @param value exchanged value at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   */
  @NotNull
  default IntIndexable withExchangedValueAt(int index, int value)
  {
    final int insertIndex = Pythonesque.mapX(index, this);
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return index == insertIndex
                ? value
                : IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this indexable with one value removed.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few values are removed.
   *
   * @param index index of the removed value (Pythonesque)
   * @return a view of this indexable where the value at the given index
   *         is removed
   */
  @NotNull
  default IntIndexable withRemovedValueAt(int index)
  {
    final int removeIndex = Pythonesque.mapX(index, this);
    if (removeIndex == 0) {
      // remove first
      return tailSet(1);
    }
    final int sz = size() - 1;
    if (removeIndex == sz) {
      // remove last
      return headSet(-1);
    }
    return new Base()
    {
      @Override
      public int get(int index)
      {
        return index < removeIndex
                ? IntIndexable.this.get(index)
                : IntIndexable.this.get(index + 1);
      }

      @Override
      public int size()
      {
        return sz;
      }
    };
  }

  /**
   * Create a view of this int indexable with two values swapped.
   * Note that this is not efficient, but handy in certain situations
   * where only a few items are swapped.
   *
   * @param index1 index of the first value (Pythonesque)
   * @param index2 index of the second value (Pythonesque)
   * @return a view of this indexable where the first and second value
   *         have exchanged places
   */
  @NotNull
  default IntIndexable withSwappedValuesAt(int index1, int index2)
  {
    final int swap1 = Pythonesque.mapX(index1, this);
    final int swap2 = Pythonesque.mapX(index2, this);
    if (swap1 == swap2) {
      return this;
    }
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        if (index == swap1) {
          return IntIndexable.this.get(swap2);
        }
        if (index == swap2) {
          return IntIndexable.this.get(swap1);
        }
        return IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * Get a rotated view of this int indexable.
   * The returned indexable will show this indexable with rotated indexes.
   * A positive {@code steps} will rotate left, i.e. any index will be accessed
   * as if the number of steps is added before extraction (modulo length).
   * E.g. a rotation of {@code -1} will return the last element when index {@code 0} is
   * requested, and the first element for index {@code 1}.
   *
   * @param steps steps to rotate
   * @return rotated view of this indexable, use  {@link #frozen()}
   *         to create an indexable which no longer depends on this one
   */
  @NotNull
  default IntIndexable rotated(int steps)
  {
    steps = steps % size();
    if (steps == 0) {
      return this;
    }
    if (steps < 0) {
      steps += size();
    }
    final int rotate = steps;
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return IntIndexable.this.get((index + rotate) % size());
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }

      @NotNull
      @Override
      public IntIndexable rotated(int steps)
      {
        return IntIndexable.this.rotated(rotate + steps);
      }
    };
  }

  /**
   * View this int indexable transformed by an operator.
   * @param op transforming operator
   * @return view of this indexable with transformed values
   */
  @NotNull
  default IntIndexable viewOp(@NotNull IntUnaryOperator op)
  {
    return IntIndexable.viewByIndex(size(),
                                    idx -> op.applyAsInt(IntIndexable.this.get(idx)));
  }

  /**
   * View this integer indexable as a double indexable.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a double indexable view of this indexable
   */
  @NotNull
  default DoubleIndexable viewAsDouble()
  {
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as a double indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return double indexable view of this indexable
   */
  @NotNull
  default DoubleIndexable viewAsDouble(@NotNull IntToDoubleFunction convert)
  {
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return convert.applyAsDouble(IntIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this integer indexable as a float indexable.
   * Note that not all integer values have an exact float representation.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a float indexable view of this indexable
   */
  @NotNull
  default FloatIndexable viewAsFloat()
  {
    return new FloatIndexable.Base()
    {
      @Override
      public float get(int index)
      {
        return (float)IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this integer indexable as a long integer indexable.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a long integer indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsLong()
  {
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this integer indexable as an unsigned long integer indexable.
   * This will assume that the values of this indexable don't have a sign bit
   * and range from 0 to 4,294,967,295.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return an unsigned long integer indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsUnsignedLong()
  {
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return IntIndexable.this.get(index) & 0xffff_ffffL;
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as a long integer indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return long indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsLong(@NotNull IntToLongFunction convert)
  {
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return convert.applyAsLong(IntIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this integer indexable as a short integer indexable.
   * Note that this may result in loss of numeric range of the values.
   * This will cast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a short integer indexable view of this indexable
   */
  @NotNull
  default ShortIndexable viewAsShort()
  {
    return new ShortIndexable.Base()
    {
      @Override
      public short get(int index)
      {
        return (short)IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this integer indexable as a byte integer indexable.
   * Note that this may result in severe loss of numeric range of the values.
   * This will cast the float values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a byte integer indexable view of this indexable
   */
  @NotNull
  default ByteIndexable viewAsByte()
  {
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return (byte)IntIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as a boolean indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return boolean indexable view of this indexable
   */
  @NotNull
  default BooleanIndexable viewAsBoolean(@NotNull IntPredicate convert)
  {
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return convert.test(IntIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View this int indexable as a char indexable.
   * @param convert conversion function which turns the values of this indexable into characters
   * @return char indexable view of this indexable
   */
  @NotNull
  default CharIndexable viewAsChar(@NotNull IntToCharFunction1 convert)
  {
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        return convert.applyAsChar(IntIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This requires a list of non-null numbers as a base.
   * @param list list used as a base, required to have only non-null elements
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public int get(int index)
      {
        return list.get(index).intValue();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This allows to provide a fallback for null elements in the list.
   * @param list list used as a base, required to have only non-null elements
   * @param nullFallback value returned for {@code null} elements in the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list, int nullFallback)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public int get(int index)
      {
        final Number number = list.get(index);
        return number != null
                ? number.intValue()
                : nullFallback;
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic list as a int indexable.
   * @param list generic list
   * @param extractor extractor function which extracts a int from the elements of {@code list}
   * @param <T> element type of list
   * @return a int indexable view of {@code list}
   */
  @NotNull
  static <T> Base viewList(@NotNull List<T> list,
                           @NotNull ToIntFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public int get(int index)
      {
        return extractor.applyAsInt(list.get(index));
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic Number indexable as integer indexable.
   * @param indexable generic indexable with Number elements, required to have only non-null elements
   * @return integer indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public int get(int index)
      {
        return indexable.get(index).intValue();
      }
    };
  }

  /**
   * View a generic Number indexable as integer indexable.
   * @param indexable generic indexable with Number elements
   * @param nullFallback fallback for {@code null} elements
   * @return integer indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable,
                            int nullFallback)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public int get(int index)
      {
        final Number number = indexable.get(index);
        return number != null
                ? number.intValue()
                : nullFallback;
      }
    };
  }

  /**
   * View a generic indexable as an int indexable.
   * @param indexable generic indexable
   * @param extractor extractor function which extracts an int from the elements of {@code indexable}
   * @param <T> element type of indexable
   * @return an int indexable view of {@code indexable}
   */
  @NotNull
  static <T> Base viewIndexable(@NotNull Indexable<T> indexable,
                                @NotNull ToIntFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public int get(int index)
      {
        return extractor.applyAsInt(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean isEmpty()
      {
        return indexable.isEmpty();
      }
    };
  }

  /**
   * View a Number array as an Indexable.
   * To view only part of the array use {@link #subSet(int, int)}.
   * @param array array used as base, required to hold only non-null values
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewNumberArray(@NotNull Number ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public int get(int index)
      {
        return array[index].intValue();
      }
    };
  }

  /**
   * View an int array as an Indexable.
   * To view only part of the array use {@link #viewArray(int[], int, int)}.
   * @param array array used as base
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull int ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public int get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * View part of an integer array as an Indexable.
   * @param array array used as base
   * @param start index of first byte to use from the array
   * @param length number of elements to use from the array
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull int[] array,
                        int start,
                        int length)
  {
    if (start < 0  ||  length < 0  ||  start + length > array.length) {
      throw new IndexOutOfBoundsException("Start or end outside of range!");
    }
    if (length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public int get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("Index %d outside range [0, %d[!",
                                                            index, length));
        }
        return array[index + start];
      }
    };
  }

  /**
   * View an object array as an integer indexable.
   * @param array         viewed array
   * @param valueExtract  converter from array elements to the integer values of this indexable
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   * @param <T> array element type
   */
  @NotNull
  static <T> Base viewArray(@NotNull T[] array,
                            @NotNull ToIntFunction<? super T> valueExtract)
  {
    if (array.length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int get(int index)
      {
        return valueExtract.applyAsInt(array[index]);
      }

      @Override
      public int size()
      {
        return array.length;
      }
    };
  }

  /**
   * Get an integer indexable of a given size which always returns the same value.
   * @param size  size of the returned indexable
   * @param value value returned for each element
   * @return indexable of size {@code size} with always the same element
   */
  @NotNull
  static Base init(int size, int value)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new Base()
    {
      @Override
      public int get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return value;
      }

      @Override
      public int size()
      {
        return 0;
      }

      @NotNull
      @Override
      public IntIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get an integer indexable of a given size which returns elements created by index.
   * This will always call the {@code producer} when an element is requested.
   * Use {@link #frozen()} on the result to create an indexable which avoids this.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable view which calls the {@code producer} with the given index when an element is requested
   * @see #initByIndex(int, IntUnaryOperator)
   */
  @NotNull
  static Base viewByIndex(int size, @NotNull IntUnaryOperator producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new Base()
    {
      @Override
      public int get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return producer.applyAsInt(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get an integer indexable of a given size which returns elements created by index.
   * This will call the producer only once for each element during the call of this method,
   * and return the results on later requests.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable with elements initialized by the {@code producer}
   * @see #viewByIndex(int, IntUnaryOperator)
   */
  @NotNull
  static IntIndexable initByIndex(int size, @NotNull IntUnaryOperator producer)
  {
    return viewByIndex(size, producer).frozen();
  }

  /**
   * View a single int value as a int indexable of size 1.
   * @param value single value
   * @return indexable with one value
   */
  @NotNull
  static IntIndexable.Base singleton(int value)
  {
    return new IntIndexable.Base() {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public int get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException(String.format("Index %d for indexable of size 1!", index));
        }
        return value;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public IntIndexable.Base reverse()
      {
        return this;
      }

      @NotNull
      @Override
      public IntIndexable rotated(int steps)
      {
        return this;
      }

      @NotNull
      @Override
      public Collection<Integer> asCollection()
      {
        return Collections.singleton(value);
      }

      @NotNull
      @Override
      public List<Integer> asList()
      {
        return Collections.singletonList(value);
      }

      @NotNull
      @Override
      public int[] toArray()
      {
        return new int[] { value };
      }

      @Override
      public int addToArray(@NotNull int[] array, int pos)
      {
        array[pos] = value;
        return pos + 1;
      }

      @NotNull
      @Override
      public Iterable<Integer> indexes()
      {
        return SINGLE_0;
      }

      @NotNull
      @Override
      public IntIndexable intIndexes()
      {
        return SINGLE_0;
      }

      @Override
      public int sum()
      {
        return value;
      }

      @Override
      public int sumX()
      {
        return value;
      }

      @Override
      public long longSum()
      {
        return value;
      }

      @Override
      public long longSumX()
      {
        return value;
      }

      @NotNull
      @Override
      public OptionalDouble average()
      {
        return OptionalDouble.of(value);
      }
    };
  }

  /**
   * An empty indexable.
   */
  @NotNull
  Base EMPTY = new Base()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public int get(int index)
    {
      throw new IndexOutOfBoundsException("Empty indexable has no elements!");
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public int[] toArray()
    {
      return Empty.INT_ARRAY;
    }

    @Override
    public int addToArray(@NotNull int[] array, int pos)
    {
      return pos;
    }

    @Override
    public int addToArray(@NotNull int[] array, int arrayPos, int index, int length)
    {
      return arrayPos;
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable.Base reverse()
    {
      return this;
    }

    @NotNull
    @Override
    public IntIndexable rotated(int steps)
    {
      return this;
    }

    @NotNull
    @Override
    public Iterator<Integer> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public ListIterator<Integer> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public PrimitiveIterator.OfInt intIterator()
    {
      return Types.EMPTY_INT_ITERATOR;
    }

    @Override
    public int foldLeft(int initialValue, @NotNull IntBinaryOperator foldOperation)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public Collection<Integer> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Integer> asList()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public LongIndexable.Base asLongIndexable()
    {
      return LongIndexable.EMPTY;
    }

    @NotNull
    @Override
    public LongIndexable.Base asUnsignedIndexable()
    {
      return LongIndexable.EMPTY;
    }

    @Override
    public void forEachInt(@NotNull IntConsumer consumer)
    {
    }

    @Override
    public <E extends Exception> void forEachIntFragile(@NotNull FragileIntConsumer<E> action) throws E
    {
    }

    @Override
    public boolean containsInt(int value)
    {
      return false;
    }

    @Override
    public void forEach(Consumer<? super Integer> action)
    {
    }

    @NotNull
    @Override
    public IntIndexable.Base frozen()
    {
      return this;
    }

    @Override
    public int sum()
    {
      return 0;
    }

    @Override
    public int sumX()
    {
      return 0;
    }

    @Override
    public long longSum()
    {
      return 0L;
    }

    @Override
    public long longSumX()
    {
      return 0;
    }

    @NotNull
    @Override
    public OptionalDouble average()
    {
      return OptionalDouble.empty();
    }

    @NotNull
    @Override
    public Spliterator.OfInt intSpliterator()
    {
      return Spliterators.emptyIntSpliterator();
    }

    @Override
    public int compareTo(@NotNull IntIndexable o)
    {
      return o.isEmpty() ? 0 : -1;
    }

    @Override
    public String toString()
    {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }

    @Override
    public boolean equals(Object obj)
    {
      return (obj instanceof IntIndexable  &&  ((IntIndexable)obj).isEmpty());
    }
  };

  /** Singleton integer indexable which has only {@code 0} as its single element. */
  @NotNull
  Base SINGLE_0 = singleton(0);

  /**
   * Get an empty indexable set.
   * @return empty indexable
   */
  @NotNull
  static Base emptyIndexable()
  {
    return EMPTY;
  }

  /**
   * Create a string representation of the given indexable.
   * @param indexable indexable
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull IntIndexable indexable)
  {
    if (indexable.isEmpty()) {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }
    final StringBuilder sb = new StringBuilder();
    sb.append('[').append(indexable.get(0));
    for (PrimitiveIterator.OfInt it = indexable.tailSet(1).intIterator();  it.hasNext();  ) {
      sb.append(',').append(it.nextInt());
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Are two int indexables equal?
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equal(@NotNull IntIndexable indexable1,
                       @NotNull IntIndexable indexable2)
  {
    if (indexable1 == indexable2) {
      return true;
    }
    if (indexable1.size() != indexable2.size()) {
      return false;
    }
    final PrimitiveIterator.OfInt it1 = indexable1.intIterator();
    final PrimitiveIterator.OfInt it2 = indexable2.intIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      if (it1.nextInt() != it2.nextInt()) {
        return false;
      }
    }
    return !(it1.hasNext()  ||  it2.hasNext());
  }

  /**
   * Are two int indexables equal?
   * This is a convenience method which may be called from
   * implementations for their {@link Object#equals(Object)}
   * method.
   * @param indexable1  first indexable
   * @param indexable2  object expected to be an indexable itself
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if the second object is not an indexable, of if sizes or values differ
   */
  static boolean equal(@NotNull IntIndexable indexable1,
                       @Nullable Object indexable2)
  {
    if (indexable2 instanceof IntIndexable) {
      return equal(indexable1, (IntIndexable)indexable2);
    }
    return false;
  }

  /**
   * Compare two int indexables.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compare(@NotNull IntIndexable indexable1,
                     @NotNull IntIndexable indexable2)
  {
    final PrimitiveIterator.OfInt it1 = indexable1.intIterator();
    final PrimitiveIterator.OfInt it2 = indexable2.intIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Integer.compare(it1.nextInt(),
                                          it2.nextInt());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Compare two int indexables as if they contain unsigned int values.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compareUnsigned(@NotNull IntIndexable indexable1,
                             @NotNull IntIndexable indexable2)
  {
    final PrimitiveIterator.OfInt it1 = indexable1.intIterator();
    final PrimitiveIterator.OfInt it2 = indexable2.intIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Integer.compareUnsigned(it1.nextInt(),
                                                  it2.nextInt());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Calculate a hashcode for a int indexable.
   * @param indexable indexable for which the hash code is required
   * @return hash code for the given indexable
   */
  static int hash(@NotNull IntIndexable indexable)
  {
    // this follows Arrays.hashCode()
    int result = 1;

    for (PrimitiveIterator.OfInt it = indexable.intIterator();  it.hasNext();  ) {
      result = 31 * result + Integer.hashCode(it.next());
    }

    return result;
  }

  /**
   * Wrap an indexable with one which caches the hash value.
   * This is useful if indexables are used as hash keys because hash value calculation
   * is expensive.
   * <p>
   * The wrapped indexable must not change after it is wrapped, otherwise strange things are expected to happen.
   * @param indexable wrapped indexable, must not change
   * @return indexable which forwards most methods to {@code indexable}, but also provides
   *         useful implementations for {@link Object#hashCode()}, {@link Object#equals(Object)},
   *         and {@link Object#toString()}
   */
  @NotNull
  static Base withCachedHash(@NotNull IntIndexable indexable)
  {
    final int hashCode = hash(indexable);
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public int get(int index)
      {
        return indexable.get(index);
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator()
      {
        return indexable.intIterator();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator(int from, int to)
      {
        return indexable.intIterator(from, to);
      }

      @NotNull
      @Override
      public Iterator<Integer> iterator()
      {
        return indexable.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Integer> listIterator()
      {
        return indexable.listIterator();
      }

      @NotNull
      @Override
      public IntIndexable.Base subSet(int fromIndex, int toIndex)
      {
        return withCachedHash(indexable.subSet(fromIndex, toIndex));
      }

      @Override
      public int addToArray(@NotNull int[] array, int pos)
      {
        return indexable.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull int[] array, int arrayPos, int index, int length)
      {
        return indexable.addToArray(array, arrayPos, index, length);
      }

      @Override
      public int hashCode()
      {
        return hashCode;
      }

      @Override
      public boolean equals(Object obj)
      {
        return equal(indexable, obj);
      }

      @Override
      public String toString()
      {
        return IntIndexable.toString(indexable);
      }
    };
  }

  /**
   * Create an integer indexable which is defined by a given range.
   * This returns an indexable which returns successive numbers in a given range.
   * As indexables have an integer size this method will throw an
   * {@link IllegalArgumentException} if the parameters require a range
   * of a larger size. If such larger size is required, but indexing is not,
   * see {@link Range#of(int, int, int)}.
   * @param startValue  start value of range (included)
   * @param maxValue  end value of range (included if stepped on)
   * @param stepSize  step size, not {@code 0}, may be negative if {@code startValue < endValue}
   * @return indexable returning the given range of integers
   */
  @NotNull
  static IntIndexable range(int startValue, int maxValue, int stepSize)
  {
    if (stepSize == 0) {
      throw new IllegalArgumentException("stepSize must not be 0!");
    }
    final long lsize = ((long)maxValue - startValue + stepSize) / stepSize;
    if (lsize > Integer.MAX_VALUE) {
      throw new IllegalArgumentException(String.format("Cannot create indexable with too large size %s", lsize));
    }
    if (lsize <= 0) {
      return EMPTY;
    }
    final int size = (int)lsize;
    return new IntIndexable.Base() {
      @Override
      public int size()
      {
        return size;
      }

      @Override
      public int get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException("["+index+"]");
        }
        return startValue + index * stepSize;
      }
    };
  }

  /**
   * Create an integer indexable which represents a range with step size 1.
   * Depending on the arguments a range with a size larger than the maximal
   * allowed value for integers may be required, which will raise an
   * {@link IllegalArgumentException}. If in need of such large ranges
   * but not of indexing see {@link Range#of(int, int)}.
   * @param startValue start value of range (included)
   * @param endValue   end value of range (included)
   * @return range from {@code startValue} to {@code endValue} with step size
   *         {@code 1} or {@code -1} (if {@code startValue > endValue} )
   */
  @NotNull
  static IntIndexable range(int startValue, int endValue)
  {
    return range(startValue, endValue, endValue < startValue ? -1 : 1);
  }

  /**
   * Create an integer range from {@code 0} with the given size.
   * As zero is included last index of the range is {@code size - 1}
   * @param size range size
   * @return zero based range
   */
  @NotNull
  static IntIndexable rangeFromSize(int size)
  {
    if (size < 0) {
      throw new IllegalArgumentException("size must not be negative!");
    }
    switch (size) {
    case 0:
      return EMPTY;
    case 1:
      return SINGLE_0;
    default:
      return new Base()
      {
        @Override
        public int get(int index)
        {
          if (index < 0  ||  index >= size) {
            throw new IndexOutOfBoundsException("["+index+"]");
          }
          return index;
        }

        @Override
        public int size()
        {
          return size;
        }
      };
    }
  }

  /**
   * Is this indexable sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * See {@linkplain #isStrictlyOrdered(IntOrdering)} for a stricter alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isOrdered(@NotNull IntOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    int last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final int value = get(i);
      if (order.checkInt(value, last) == Descending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered(IntOrdering)} for a more relaxed alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered(@NotNull IntOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    int last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final int value = get(i);
      if (order.checkInt(value, last) != Ascending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Do a binary search in an indexable ordered in natural ascending order.
   * This requires this indexable to be ordered in
   * {@link IntOrdering#ASCENDING natural ascending order},
   * i.e. {@linkplain #isOrdered(IntOrdering)} has to return {@code true}
   * to make this method work.
   * <p>
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(IntOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @return a positive integer defining the index where the given value was found,
   *         and a negative integer if the value is not contained. The latter defines
   *         the insertion point where the looked up value would match into this
   *         sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(int value)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final int dp = get(mid);

      if (dp < value) {
        low = mid + 1;
      }
      else if (dp > value) {
        hi = mid - 1;
      }
      else {
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Do a binary search in an ordered indexable.
   * This requires this indexable to be ordered in non-descending order
   * as defined by the given {@code order},
   * i.e. {@linkplain #isOrdered(IntOrdering)} has to return {@code true}
   * for this order.
   * <p>
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(IntOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @param order sort order
   * @return a positive integer defining the index where the given value was found,
   * and a negative integer if the value is not contained. The latter defines
   * the insertion point where the looked up value would match into this
   * sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(int value, @NotNull IntOrdering order)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final int dp = get(mid);

      switch (order.checkInt(dp, value)) {
      case Ascending:
        low = mid + 1;
        break;
      case Descending:
        hi = mid - 1;
        break;
      default:
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Create a (mutable) int indexable from the values of this indexable
   * which is ordered as defined by the given ordering.
   * <p>
   * If natural ordering is required prefer calling {@link #ordered()}
   * because that typically runs 1.5 times faster.
   * @param order sort order
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableIntIndexable ordered(@NotNull IntOrdering order)
  {
    final MutableIntIndexable result = MutableIntIndexable.fromIntIndexable(this);
    result.order(order);
    return result;
  }

  /**
   * Create a (mutable) int indexable from the values of this indexable
   * which is ordered in natural order.
   * <p>
   * This can be some 1.5x faster than using {@link #ordered(IntOrdering)}
   * with {@link IntOrdering#ASCENDING natural order}.
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableIntIndexable ordered()
  {
    final MutableIntIndexable result = MutableIntIndexable.fromIntIndexable(this);
    result.order();
    return result;
  }

  /**
   * Make this integer indexable usable as a class with defined standard Object methods.
   * @return based version of this indexable
   */
  @NotNull
  default IntIndexable.Base asBase()
  {
    return new IntIndexable.Base() {
      @Override
      public int size()
      {
        return IntIndexable.this.size();
      }

      @Override
      public int get(int index)
      {
        return IntIndexable.this.get(index);
      }

      @NotNull
      @Override
      public IntIndexable.Base reverse()
      {
        return IntIndexable.this.reverse();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator()
      {
        return IntIndexable.this.intIterator();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator(int from, int to)
      {
        return IntIndexable.this.intIterator(from ,to);
      }

      @NotNull
      @Override
      public Iterator<Integer> iterator()
      {
        return IntIndexable.this.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Integer> listIterator()
      {
        return IntIndexable.this.listIterator();
      }

      @NotNull
      @Override
      public IntIndexable.Base subSet(int fromIndex, int toIndex)
      {
        return IntIndexable.this.subSet(fromIndex, toIndex);
      }

      @NotNull
      @Override
      public int[] toArray()
      {
        return IntIndexable.this.toArray();
      }

      @Override
      public int addToArray(@NotNull int[] array, int pos)
      {
        return IntIndexable.this.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull int[] array, int arrayPos, int index, int length)
      {
        return IntIndexable.this.addToArray(array, arrayPos, index, length);
      }

      @NotNull
      @Override
      public LongIndexable.Base asLongIndexable()
      {
        return IntIndexable.this.asLongIndexable();
      }

      @NotNull
      @Override
      public LongIndexable.Base asUnsignedIndexable()
      {
        return IntIndexable.this.asUnsignedIndexable();
      }
    };
    
  }
  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}. It also provides
   * {@link Comparable#compareTo(Object)}.
   * @see IntIndexable#asBase()
   */
  abstract class Base extends IntCountable.Base
          implements IntIndexable,
                     Comparable<IntIndexable>
  {
    @Override
    public int compareTo(@NotNull IntIndexable o)
    {
      Objects.requireNonNull(o);
      return compare(this, o);
    }

    @Override
    public int hashCode()
    {
      return hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      return equal(this, obj);
    }

    @Override
    public String toString()
    {
      return IntIndexable.toString(this);
    }

    @NotNull
    @Override
    public IntIndexable.Base asBase()
    {
      return this;
    }
  }

  /**
   * Consumer for index-value pairs with {@code int} values.
   */
  @FunctionalInterface
  interface EntryConsumer
  {
    /**
     * Do whatever is necessary for the given index-value combination.
     * @param index index of the given {@code value}
     * @param value value at the given {@code index}
     */
    void accept(int index, int value);
  }
}
