// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.FragileIntConsumer;
import de.caff.generics.function.IntOrdering;
import de.caff.generics.util.Counter;

import java.util.*;
import java.util.function.*;

import static de.caff.generics.Types.EMPTY_HASH;

/**
 * Something containing a defined number of int values which can be iterated over.
 * <p>
 * Having a known size usually helps in optimization, e.g. when copying.
 * </p>
 * <p>
 *   This is meant to be used as a read-only interface, although {@link Iterable}
 *   basically exports mutability via {@link Iterator#remove()}.
 *   This interface extends {@code Iterable} nevertheless so
 *   implementations can be used in enhanced {@code for} loops.
 *   In an ideal world Java collections should implement this interface to provide
 *   a better compile-time read-only access compared to the
 *   {@code Collections#unmodifiableCollection()} which will result
 *   in runtime errors when mutating.
 * </p>
 * <p>
 *   To allow living in a standard Java environment this interface provides
 *   methods {@link IntCountable#asCollection()} to view it as an unmodifiable
 *   collection and {@link IntCountable#viewCollection(Collection)} allowing
 *   to view a collection as {@link IntCountable}.
 * </p>
 * <p>
 *   A countable can also view an array or part of it ({@link IntIndexable#viewArray(int[])}
 *   and {@link IntIndexable#viewArray(int[], int, int)}), and can be converted to
 *   an array.
 * </p>
 * <p>
 *   Last not least an {@link IntCountable#empty() empty countable} is useful sometimes.
 * </p>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 24, 2020
 */
public interface IntCountable
        extends PrimitiveIntIterable,
                Sizeable
{
  /**
   * Basic implementation of an empty countable
   * Don't use this, use {@link #empty()}.
   */
  IntCountable EMPTY = new IntCountable.Base()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @Override
    @NotNull
    public PrimitiveIterator.OfInt intIterator()
    {
      return new PrimitiveIterator.OfInt()
      {
        @Override
        public int nextInt()
        {
          throw new NoSuchElementException();
        }

        @Override
        public boolean hasNext()
        {
          return false;
        }
      };
    }

    @NotNull
    @Override
    public Collection<Integer> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public int[] toArray()
    {
      return Empty.INT_ARRAY;
    }

    @Override
    public int addToArray(@NotNull int[] array, int arrayPos)
    {
      return arrayPos;
    }


    @NotNull
    @Override
    public ArrayList<Integer> toList()
    {
      return new ArrayList<>();
    }

    @Override
    public void addAllTo(@NotNull Collection<? super Integer> collection)
    {
    }

    @NotNull
    @Override
    public PrimitiveIntIterable filtered(@NotNull IntPredicate filter)
    {
      return this;
    }

    @Override
    public <E extends Exception> void forEachIntFragile(@NotNull FragileIntConsumer<E> consumer) throws E
    {
    }

    @Override
    public void forEachInt(@NotNull IntConsumer action)
    {
    }

    @NotNull
    @Override
    public IntIndexable sorted(@NotNull IntOrdering order)
    {
      return IntIndexable.emptyIndexable();
    }

    @Override
    public boolean isSorted(@NotNull IntOrdering order)
    {
      return true;
    }

    @Override
    public boolean isStrictlySorted(@NotNull IntOrdering order)
    {
      return true;
    }

    @Override
    public int first()
    {
      throw new NoSuchElementException("No first element in an empty countable!");
    }

    @Override
    public int last()
    {
      throw new NoSuchElementException("No last element in an empty countable!");
    }

    @NotNull
    @Override
    public IntIndexable frozen()
    {
      return IntIndexable.EMPTY;
    }
  };

  /**
   * Is this countable empty?
   * An empty countable does not contain any elements which means that {@link #size()}
   * will return {@code 0}.
   * <p>
   * This default implementation just checks for {@link #size()} returning {@code 0}
   * @return {@code true} if this countable is empty, {@code false} otherwise.
   * Because calculating the size might be expensive sometimes in these cases
   * implementations should override this method if possible.
   * </p>
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * View this countable as an unmodifiable standard Java collection.
   * @return standard Java {@link Collection} view of this countable
   */
  @NotNull
  default Collection<Integer> asCollection()
  {
    return new AbstractCollection<Integer>()
    {
      @NotNull
      @Override
      public Iterator<Integer> iterator()
      {
        return IntCountable.this.intIterator();
      }

      @Override
      public int size()
      {
        return IntCountable.this.size();
      }
    };
  }

  /**
   * Get a list which is the copy of this countable.
   * @return array list with the elements of this countable
   */
  @NotNull
  default ArrayList<Integer> toList()
  {
    final ArrayList<Integer> result = new ArrayList<>(size());
    addAllTo(result);
    return result;
  }

  /**
   * Add all entries of this countable to the given collection.
   * @param collection collection where all entries are added
   */
  default void addAllTo(@NotNull Collection<? super Integer> collection)
  {
    for (Integer entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Copy the content of this countable to an array.
   * @return array with the copied content of this countabel
   */
  @NotNull
  @SuppressWarnings("unchecked")
  default int[] toArray()
  {
    final int len = size();
    if (len == 0) {
      return Empty.INT_ARRAY;
    }
    final int[] array = new int[len];
    addToArray(array, 0);
    return array;
  }

  /**
   * Add the complete content of this countable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  start position in hte array where the content is added
   * @return array position after the added content
   */
  default int addToArray(@NotNull int[] array, int arrayPos)
  {
    final Counter pos = Counter.simple(arrayPos);
    forEachInt(v -> {
      array[pos.getValue()] = v;
      pos.add1();
    });
    return pos.getValue();
  }

  /**
   * View this int countable as if it has another type.
   * This is a view which just converts each element when accessed.
   * @param mapper converter from elements of this countable to elements of the returned countable
   * @param <R> resulting element type
   * @return countable view of this countable with mapped elements
   */
  @NotNull
  default <R> Countable<R> view(@NotNull IntFunction<? extends R> mapper)
  {
    return new Countable.Base<R>()
    {
      @Override
      public int size()
      {
        return IntCountable.this.size();
      }

      @NotNull
      @Override
      public Iterator<R> iterator()
      {
        final PrimitiveIterator.OfInt iterator = intIterator();
        return new Iterator<R>()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public R next()
          {
            return mapper.apply(iterator.nextInt());
          }
        };
      }

      @Override
      public boolean isEmpty()
      {
        return IntCountable.this.isEmpty();
      }
    };
  }

  /**
   * Get a filtered view of this countable.
   * @param filter check which defines the elements which are included in the returned {@code Iterable}.
   * @return iterable only providing the elements which {@code filter} accepts
   */
  @NotNull
  default PrimitiveIntIterable filtered(@NotNull IntPredicate filter)
  {
    return () -> {
      final PrimitiveIterator.OfInt baseIterator = IntCountable.this.intIterator();
      return new PrimitiveIterator.OfInt()
      {
        boolean hasNext;
        int nextValue;

        private void forward()
        {
          while (baseIterator.hasNext()) {
            nextValue = baseIterator.next();
            if (filter.test(nextValue)) {
              hasNext = true;
            }
          }
          hasNext = false;

        }

        @Override
        public int nextInt()
        {
          if (!hasNext) {
            throw new NoSuchElementException();
          }
          try {
            return nextValue;
          } finally {
            forward();
          }
        }

        @Override
        public boolean hasNext()
        {
          return hasNext;
        }
      };
    };
  }

  /**
   * Get a sorted frozen version of this countable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying collection might
   * nevertheless change. This method copies the current state of this countable
   * into an unmodifiable state, and returns an Indexable which is
   * sorted, stable in size and will always return the same elements
   * in the given order.
   * @param order ordering condition for the returned indexable
   * @return sorted frozen indexable version of this Countable
   */
  @NotNull
  default IntIndexable sorted(@NotNull IntOrdering order)
  {
    final MutableIntIndexable array = MutableIntIndexable.viewArray(toArray());
    array.order(order);
    return array;
  }

  /**
   * Is this countable ordered according to the given sort order?
   * This method accepts equal elements and assumes them ordered.
   * See {@link #isStrictlySorted(IntOrdering)} for a check which does not accept that.
   * Empty or 1-element countables will always return {@code true}.
   * @param order order which is checked
   * @return {@code true} if the elements in this countable are ordered considering the given {@code order}<br>
   *         {@code false} if not
   */
  default boolean isSorted(@NotNull IntOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    final PrimitiveIterator.OfInt it = intIterator();
    int lastValue = it.next();
    while (it.hasNext()) {
      final int value = it.nextInt();
      if (order.descending(lastValue, value)) {
        return false;
      }
      lastValue = value;
    }
    return true;
  }

  /**
   * Is this countable strictly ordered according to the given sort order?
   * This method does consider equal elements unordered.
   * See {@link #isSorted(IntOrdering)} for a check which is more relaxed.
   * Empty or 1-element countables will always return {@code true}.
   * @param order order which is checked
   * @return {@code true} if the elements in this countable are ordered considering the given {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlySorted(@NotNull IntOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    final PrimitiveIterator.OfInt it = intIterator();
    int lastValue = it.next();
    while (it.hasNext()) {
      final int value = it.nextInt();
      if (order.descendingOrSame(lastValue, value)) {
        return false;
      }
      lastValue = value;
    }
    return true;
  }

  /**
   * Get a frozen version of this countable.
   * <p>
   * Often Countables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying collection might
   * nevertheless change. This copies the current state of this countable
   * into an unmodifiable state, and returns a Countable which is
   * stable in size and will returns always the same elements.
   * This method also takes care of possibly mutable elements, which are
   * copied when freezing and possibly also on getting.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, but calling this method again will always create a
   * new object.
   *
   * @return frozen indexable version of this Countable
   */
  @NotNull
  default IntIndexable frozen()
  {
    return IndexableHelper.frozenFromArray(toArray());
  }

  /**
   * Get the first value.
   * @return first value in this countable
   * @throws NoSuchElementException if this countable is empty
   */
  default int first()
  {
    return intIterator().nextInt();
  }

  /**
   * Get the last value.
   * In the base implementation iterates over all elements in this countable
   * to reach the last element.
   * @return last element in this countable
   */
  default int last()
  {
    if (isEmpty()) {
      throw new NoSuchElementException("No last element because there are no elements!");
    }
    int latest = 0;
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext(); ) {
      latest = it.nextInt();
    }
    return latest;
  }

  /**
   * Convert this into a base countable.
   * Base countables have useful default implementation for
   * {@link Object#equals(Object)}, {@link Object#hashCode()},
   * and {@link Object#toString()}.
   * @return base indexable
   */
  @NotNull
  default IntCountable.Base asBase()
  {
    return new IntCountable.Base() {
      @Override
      public PrimitiveIterator.OfInt intIterator()
      {
        return IntCountable.this.intIterator();
      }

      @Override
      public int size()
      {
        return IntCountable.this.size();
      }
    };
  }

  /**
   * Does this countable equal another one?
   * <p>
   * This method is more flexible than standard {@link Object#equals(Object)}
   * because it allows to define the check for element equality.
   *
   * @param other          other countable
   * @return {@code true} if this and the other countable are considered equal regarding the given element check<br>
   *         {@code false} if not
   * @param <E> element type of other countable
   */
  default <E> boolean isEqual(@NotNull IntCountable other)
  {
    if (size() != other.size()) {
      return false;
    }
    final PrimitiveIterator.OfInt thisIt = intIterator();
    final PrimitiveIterator.OfInt thatIt = other.intIterator();
    while (thisIt.hasNext()  &&  thatIt.hasNext()) {
      if (thisIt.nextInt() != thatIt.nextInt()) {
        return false;
      }
    }
    return !thisIt.hasNext() && !thatIt.hasNext();
  }

  /**
   * View a collection as if it were a countable.
   * This creates read-only access to the underlying collection.
   * <p>
   * As this is a view any changes in the underlying collection
   * will be reflected from the returned {@code Countable}.
   * If this may pose a problem copy the collection first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(new ArrayList<>(collection));
   * }</pre></blockquote>
   * @param collection collection to view
   * @return countable view of the given {@code collection}
   */
  @NotNull
  static IntCountable viewCollection(@NotNull Collection<? extends Number> collection)
  {
    return new IntCountable.Base()
    {
      @Override
      public int size()
      {
        return collection.size();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator()
      {
        final Iterator<? extends Number> iterator = collection.iterator();
        return new PrimitiveIterator.OfInt()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public int nextInt()
          {
            return iterator.next().intValue();
          }
        };
      }
    };
  }

  /**
   * View a collection which is potentially {@code null} as if it were a countable.
   * This creates read-only access to the underlying collection.
   * <p>
   * As this is a view any changes in the underlying collection
   * will be reflected from the returned {@code Countable}.
   * If this may pose a problem copy the collection first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(new ArrayList<>(collection));
   * }</pre></blockquote>
   * or call {@linkplain #frozen()} on the returned countable
   * @param collection collection to view
   * @return countable view of the given {@code collection}, {@link #empty() empty}
   *         if {@code collection} is {@code null}
   */
  @NotNull
  static  IntCountable viewCollectionN(@Nullable Collection<? extends Number> collection)
  {
    return collection == null
            ? empty()
            : viewCollection(collection);
  }

  /**
   * View a collection as if it were a countable.
   * This creates read-only access to the underlying collection.
   * <p>
   * As this is a view any changes in the underlying collection
   * will be reflected from the returned {@code Countable}.
   * If this may pose a problem copy the collection first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(new ArrayList<>(collection));
   * }</pre></blockquote>
   * @param collection collection to view
   * @param intMapper  mapper from the collection's value type to {@code int}
   * @param <E> valeu type of the collection
   * @return countable view of the given {@code collection}
   */
  @NotNull
  static <E> IntCountable viewCollection(@NotNull Collection<E> collection,
                                         @NotNull ToIntFunction<? super E> intMapper)
  {
    return new IntCountable.Base()
    {
      @Override
      public int size()
      {
        return collection.size();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator()
      {
        final Iterator<E> iterator = collection.iterator();
        return new PrimitiveIterator.OfInt()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public int nextInt()
          {
            return intMapper.applyAsInt(iterator.next());
          }
        };
      }
    };
  }

  /**
   * View a collection which is potentially {qcode null}as if it were a countable.
   * This creates read-only access to the underlying collection.
   * <p>
   * As this is a view any changes in the underlying collection
   * will be reflected from the returned {@code Countable}.
   * If this may pose a problem copy the collection first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(new ArrayList<>(collection));
   * }</pre></blockquote>
   * @param collection collection to view
   * @param intMapper  mapper from the collection's value type to {@code int}
   * @param <E> valeu type of the collection
   * @return countable view of the given {@code collection}
   */
  @NotNull
  static <E> IntCountable viewCollectionN(@Nullable Collection<E> collection,
                                          @NotNull ToIntFunction<? super E> intMapper)
  {
    return collection == null
            ? empty()
            : viewCollection(collection, intMapper);
  }

  /**
   * Get an empty countable.
   * This is often preferable as a return result to indicate <i>nothing</i>
   * instead of returning {@code null}.
   * @return empty countable
   */
  @NotNull
  static IntCountable empty()
  {
    return EMPTY;
  }

  /**
   * Get an int countable with a single value.
   * @param singleValue single value of this countable
   * @return countable of size 1 with the given element
   */
  @NotNull
  static IntCountable singleton(int singleValue)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      @NotNull
      public PrimitiveIterator.OfInt intIterator()
      {
        return new PrimitiveIterator.OfInt() {
          boolean finished;

          @Override
          public int nextInt()
          {
            if (finished) {
              throw new NoSuchElementException();
            }
            finished = true;
            return singleValue;
          }

          @Override
          public boolean hasNext()
          {
            return !finished;
          }
        };
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public IntIndexable frozen()
      {
        return IntIndexable.singleton(singleValue);
      }

      @NotNull
      @Override
      public IntIndexable sorted(@NotNull IntOrdering order)
      {
        return IntIndexable.singleton(singleValue);
      }

      @Override
      public boolean isSorted(@NotNull IntOrdering order)
      {
        return true;
      }

      @Override
      public boolean isStrictlySorted(@NotNull IntOrdering order)
      {
        return true;
      }

      @Override
      public int first()
      {
        return singleValue;
      }

      @Override
      public int last()
      {
        return singleValue;
      }
    };
  }

  /**
   * Get a countable which is returning the same value multiple times.
   * This can be useful under special circumstances.
   * @param value value returned from this countable on each {@code next()} call of its iterator
   * @param count   number of times the same value is returned from the iterator, must not be negative
   * @return countable which returns {@code value} {@code count} times during iteration
   */
  @NotNull
  static IntCountable uniform(int value, int count)
  {
    if (count < 0) {
      throw new IllegalArgumentException("Count has to be positive or 0, but is "+count);
    }
    if (count == 0) {
      return empty();
    }
    if (count == 1) {
      return singleton(value);
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return count;
      }

      @Override
      @NotNull
      public PrimitiveIterator.OfInt intIterator()
      {
        return new PrimitiveIterator.OfInt()
        {
          private int index = 0;
          
          @Override
          public boolean hasNext()
          {
            return index < count;
          }

          @Override
          public int nextInt()
          {
            if (index >= count) {
              throw new NoSuchElementException("No value for "+index+" in this Countable!");
            }
            ++index;
            return value;
          }
        };
      }
    };
  }

  /**
   * Create a string representation of the given countable.
   * @param countable countable
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull IntCountable countable)
  {
    if (countable.isEmpty()) {
      return "[]";
    }
    final StringBuilder sb = new StringBuilder("[");
    boolean first = true;
    for (PrimitiveIterator.OfInt it = countable.intIterator();  it.hasNext();  ) {
      if (first) {
        first = false;
      }
      else {
        sb.append(',');
      }
      sb.append(it.nextInt());
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   * @see IntCountable#asBase()
   */
  abstract class Base implements IntCountable
  {
    @Override
    public int hashCode()
    {
      int result = EMPTY_HASH;

      for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext(); ) {
        result = 31 * result + it.nextInt();
      }

      return result;
    }

    @Override
    public boolean equals(Object obj)
    {
      if (!(obj instanceof IntCountable)) {
        return false;
      }
      return isEqual((IntCountable)obj);
    }

    @Override
    public String toString()
    {
      return IntCountable.toString(this);
    }

    @Override
    @NotNull
    public Base asBase()
    {
      return this;
    }
  }
}
