// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;
import java.util.function.Function;

/**
 * Helper class for indexables.
 * This class provides methods used for indexables.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Dezember 10, 2021
 */
class IndexableHelper
{
  /** Don't create. */
  private IndexableHelper()
  {}

  /**
   * Create a frozen indexable from a list.
   * This assumes that the list will never be changed.
   * @param list list to be wrapped by an indexable
   * @param <T> element type of indexable/list
   * @return indexable based on list, which will return itself on frozen and will include
   *         further optimizations for an unmodified list
   */
  @NotNull
  static <T> Indexable.Base<T> frozenFromList(@NotNull final List<T> list)
  {
    // this is nearly the same as Indexable.viewList(), but takes care of overriding frozen(), ordered(), and some more
    return new Indexable.Base<T>()
    {
      /**
       * Access to this base for inner classes.
       *
       * @return the outer base
       */
      @NotNull
      Indexable.Base<T> getOuter()
      {
        return this;
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public T get(int index)
      {
        return list.get(index);
      }

      @NotNull
      @Override
      public Collection<T> asCollection()
      {
        return Collections.unmodifiableCollection(list);
      }

      @NotNull
      @Override
      public List<T> asList()
      {
        return Collections.unmodifiableList(list);
      }

      @NotNull
      @Override
      public Indexable<T> subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > size()) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        final int length = toIndex - fromIndex;
        return length == 0
                ? Indexable.emptyIndexable()
                : new Indexable.Base<T>()
        {
          @Override
          public int size()
          {
            return length;
          }

          @Override
          public T get(int index)
          {
            if (index < 0 || index >= length) {
              throw new IndexOutOfBoundsException("No such element: " + index);
            }
            return list.get(index + fromIndex);
          }

          @NotNull
          @Override
          public Indexable<T> subSet(int fromIdx, int toIdx)
          {
            if (fromIdx < 0) {
              throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
            }
            if (toIdx > length) {
              throw new IndexOutOfBoundsException("toIndex = " + toIdx);
            }
            if (fromIdx > toIdx) {
              throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                                 ") > toIndex(" + toIdx + ")");
            }
            return getOuter().subSet(fromIndex + fromIdx,
                                     fromIndex + toIdx);
          }

          @NotNull
          @Override
          public Indexable<T> frozen()
          {
            return this;
          }

          @NotNull
          @Override
          public Spliterator<T> spliterator()
          {
            return frozenSpliterator();
          }

          @NotNull
          @Override
          public Spliterator<T> frozenSpliterator()
          {
            return new IndexableSpliterator<>(getOuter(),
                                              fromIndex, toIndex,
                                              true);
          }
        };
      }

      @NotNull
      @Override
      public Indexable<T> frozen()
      {
        return this;  // already frozen
      }

      @NotNull
      @Override
      public Spliterator<T> spliterator()
      {
        return frozenSpliterator();
      }

      @NotNull
      @Override
      public Spliterator<T> frozenSpliterator()
      {
        return new IndexableSpliterator<>(this, 0, size(), true);
      }
    };
  }

  /**
   * Create a frozen indexable from a list while keeping possibly mutable elements safe.
   * This assumes that the list will never be changed.
   * It also takes care of mutable elements by copying them on each request, so the elements
   * in the returned indexable will never change.
   * @param list list to be wrapped by an indexable
   * @param elementCloner function called for each requested element which returns a copy it
   * @param <T> element type of indexable/list
   * @return indexable based on list, which will return itself on frozen and will include
   *         further optimizations for an unmodified list
   */
  @NotNull
  static <T> Indexable.Base<T> frozenFromList(@NotNull final List<T> list,
                                         @NotNull final Function<? super T, ? extends T> elementCloner)
  {
    // this is nearly the same as Indexable.viewList(), but takes care of overriding frozen(), ordered(), and some more
    return new Indexable.Base<T>()
    {
      /**
       * Access to this base for inner classes.
       *
       * @return the outer base
       */
      @NotNull
      Indexable.Base<T> getOuter()
      {
        return this;
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public T get(int index)
      {
        return elementCloner.apply(list.get(index));
      }

      @NotNull
      @Override
      public Collection<T> asCollection()
      {
        return this.<T>view(elementCloner).asCollection();
      }

      @NotNull
      @Override
      public List<T> asList()
      {
        return this.<T>view( elementCloner).asList();
      }

      @NotNull
      @Override
      public Indexable<T> subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > size()) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        final int length = toIndex - fromIndex;
        return length == 0
                ? Indexable.emptyIndexable()
                : new Indexable.Base<T>()
        {
          @Override
          public int size()
          {
            return length;
          }

          @Override
          public T get(int index)
          {
            if (index < 0 || index >= length) {
              throw new IndexOutOfBoundsException("No such element: " + index);
            }
            return elementCloner.apply(list.get(index + fromIndex));
          }

          @NotNull
          @Override
          public Indexable<T> subSet(int fromIdx, int toIdx)
          {
            if (fromIdx < 0) {
              throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
            }
            if (toIdx > length) {
              throw new IndexOutOfBoundsException("toIndex = " + toIdx);
            }
            if (fromIdx > toIdx) {
              throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                                 ") > toIndex(" + toIdx + ")");
            }
            return getOuter().subSet(fromIndex + fromIdx,
                                     fromIndex + toIdx);
          }

          @NotNull
          @Override
          public Spliterator<T> spliterator()
          {
            return frozenSpliterator();
          }

          @NotNull
          @Override
          public Spliterator<T> frozenSpliterator()
          {
            return new IndexableSpliterator<>(getOuter(),
                                              fromIndex, toIndex,
                                              true);
          }
        };
      }

      @NotNull
      @Override
      public Indexable<T> frozen()
      {
        return this;  // already frozen
      }

      @NotNull
      @Override
      public Spliterator<T> spliterator()
      {
        return frozenSpliterator();
      }

      @NotNull
      @Override
      public Spliterator<T> frozenSpliterator()
      {
        return new IndexableSpliterator<>(this, 0, size(), true);
      }
    };
  }

  @NotNull
  static <T> Indexable.Base<T> frozenFromArray(@NotNull T[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return Indexable.emptyIndexable();
    }
    return new Indexable.Base<T>()
    {
      /**
       * Access to this base for inner classes.
       *
       * @return the outer base
       */
      @NotNull
      Indexable.Base<T> getOuter()
      {
        return this;
      }

      @Override
      public T get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @NotNull
      @Override
      public Indexable<T> frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Indexable<T> subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > size()) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        final int length = toIndex - fromIndex;
        return length == 0
                ? Indexable.emptyIndexable()
                : new Indexable.Base<T>()
        {
          @Override
          public int size()
          {
            return length;
          }

          @Override
          public T get(int index)
          {
            if (index < 0 || index >= length) {
              throw new IndexOutOfBoundsException("No such element: " + index);
            }
            return array[index + fromIndex];
          }

          @NotNull
          @Override
          public Indexable<T> subSet(int fromIdx, int toIdx)
          {
            if (fromIdx < 0) {
              throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
            }
            if (toIdx > length) {
              throw new IndexOutOfBoundsException("toIndex = " + toIdx);
            }
            if (fromIdx > toIdx) {
              throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                                 ") > toIndex(" + toIdx + ")");
            }
            return getOuter().subSet(fromIndex + fromIdx,
                                     fromIndex + toIdx);
          }

          @NotNull
          @Override
          public Indexable<T> frozen()
          {
            return this;
          }

          @NotNull
          @Override
          public Spliterator<T> spliterator()
          {
            return frozenSpliterator();
          }

          @NotNull
          @Override
          public Spliterator<T> frozenSpliterator()
          {
            return new IndexableSpliterator<>(getOuter(),
                                              fromIndex, toIndex,
                                              true);
          }
        };
      }
    };
  }

  /**
   * Get a double indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen double indexable view of the given {@code array}
   */
  @NotNull
  static DoubleIndexable.Base frozenFromArray(@NotNull double[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return DoubleIndexable.EMPTY;
    }
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public DoubleIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a double indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen double indexable view of the given {@code array}
   */
  @NotNull
  static DoubleIndexable.Base frozenFromArray(@NotNull double[] array,
                                              int start,
                                              int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }
    
    if (length == 0) {
      return DoubleIndexable.EMPTY;
    }
    
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for double[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public DoubleIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }

  /**
   * Get a float indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen float indexable view of the given {@code array}
   */
  @NotNull
  static FloatIndexable.Base frozenFromArray(@NotNull float[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return FloatIndexable.EMPTY;
    }
    return new FloatIndexable.Base()
    {
      @Override
      public float get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public FloatIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a float indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen float indexable view of the given {@code array}
   */
  @NotNull
  static FloatIndexable.Base frozenFromArray(@NotNull float[] array,
                                              int start,
                                              int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }
    
    if (length == 0) {
      return FloatIndexable.EMPTY;
    }
    
    return new FloatIndexable.Base()
    {
      @Override
      public float get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for float[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public FloatIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }

  /**
   * Get a long indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen long indexable view of the given {@code array}
   */
  @NotNull
  static LongIndexable.Base frozenFromArray(@NotNull long[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return LongIndexable.EMPTY;
    }
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public LongIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a long indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen long indexable view of the given {@code array}
   */
  @NotNull
  static LongIndexable.Base frozenFromArray(@NotNull long[] array,
                                             int start,
                                             int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }

    if (length == 0) {
      return LongIndexable.EMPTY;
    }

    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for long[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public LongIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }
  
  /**
   * Get an int indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen int indexable view of the given {@code array}
   */
  @NotNull
  static IntIndexable.Base frozenFromArray(@NotNull int[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return IntIndexable.EMPTY;
    }
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public IntIndexable.Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public IntIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get an int indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen int indexable view of the given {@code array}
   */
  @NotNull
  static IntIndexable.Base frozenFromArray(@NotNull int[] array,
                                             int start,
                                             int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }

    if (length == 0) {
      return IntIndexable.EMPTY;
    }

    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for int[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public IntIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public IntIndexable.Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }

  /**
   * Get a short indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen short indexable view of the given {@code array}
   */
  @NotNull
  static ShortIndexable.Base frozenFromArray(@NotNull short[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return ShortIndexable.EMPTY;
    }
    return new ShortIndexable.Base()
    {
      @Override
      public short get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public ShortIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a short indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen short indexable view of the given {@code array}
   */
  @NotNull
  static ShortIndexable.Base frozenFromArray(@NotNull short[] array,
                                             int start,
                                             int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }

    if (length == 0) {
      return ShortIndexable.EMPTY;
    }

    return new ShortIndexable.Base()
    {
      @Override
      public short get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for short[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public ShortIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }

  /**
   * Get a byte indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen byte indexable view of the given {@code array}
   */
  @NotNull
  static ByteIndexable.Base frozenFromArray(@NotNull byte[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return ByteIndexable.EMPTY;
    }
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public ByteIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a byte indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen byte indexable view of the given {@code array}
   */
  @NotNull
  static ByteIndexable.Base frozenFromArray(@NotNull byte[] array,
                                             int start,
                                             int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }

    if (length == 0) {
      return ByteIndexable.EMPTY;
    }

    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for byte[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public ByteIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }

  /**
   * Get a char indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen char indexable view of the given {@code array}
   */
  @NotNull
  static CharIndexable.Base frozenFromArray(@NotNull char[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return CharIndexable.EMPTY;
    }
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public CharIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a char indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen char indexable view of the given {@code array}
   */
  @NotNull
  static CharIndexable.Base frozenFromArray(@NotNull char[] array,
                                             int start,
                                             int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }

    if (length == 0) {
      return CharIndexable.EMPTY;
    }

    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for char[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public CharIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }

  /**
   * Get a boolean indexable view of an array which is frozen.
   * @param array array not expected to be changed
   * @return frozen boolean indexable view of the given {@code array}
   */
  @NotNull
  static BooleanIndexable.Base frozenFromArray(@NotNull boolean[] array)
  {
    final int size = array.length;
    if (size == 0) {
      return BooleanIndexable.EMPTY;
    }
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return array[index];
      }

      @Override
      public int size()
      {
        return size;
      }

      @Override
      public boolean isEmpty()
      {
        return false; // as size != 0
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex == 0  &&  toIndex == size) {
          return this;
        }
        return frozenFromArray(array, fromIndex, toIndex - fromIndex);
      }

      @NotNull
      @Override
      public BooleanIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a boolean indexable view of a part of an array which is frozen.
   * @param array array not expected to be changed
   * @param start start index into array
   * @param length number of values to be used from array
   * @return frozen boolean indexable view of the given {@code array}
   */
  @NotNull
  static BooleanIndexable.Base frozenFromArray(@NotNull boolean[] array,
                                             int start,
                                             int length)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start="+start);
    }
    if (length < 0) {
      throw new IndexOutOfBoundsException("length="+length);
    }
    if (start + length > array.length) {
      throw new IndexOutOfBoundsException("end="+ (start+length));
    }

    if (length == 0) {
      return BooleanIndexable.EMPTY;
    }

    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("index=%d for boolean[%d]", index, length));
        }
        return array[start + index];
      }

      @Override
      public int size()
      {
        return length;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public BooleanIndexable frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return frozenFromArray(array, start + fromIndex, toIndex - fromIndex);
      }
    };
  }
}
