// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.*;
import de.caff.generics.range.Range;
import de.caff.generics.tuple.*;

import java.util.*;
import java.util.function.*;
import java.util.stream.Stream;

import static de.caff.generics.Order.Ascending;
import static de.caff.generics.Order.Descending;

/**
 * Class which allows readonly access by index.
 * <p>
 * This interface describes something which is iterable, has a defined size
 * and allows direct indexed access to its elements.
 * </p>
 * <p>
 * Although this behaves similar to an unmodifiable {@link java.util.List}
 * its advantage is greater clearness because there are no methods to
 * modify this indexable, while {@code List} provides such methods but
 * will throw an exception when one of these method is called on an unmodifiable
 * list. Whenever you see a {@code List} in code you cannot be sure whether
 * it's meant to be modified or not. With an {@link Indexable} this is perfectly
 * clear. But note that in general you cannot be sure whether the underlying class
 * which implements this interface is immutable or not, so it might change,
 * although this is considered bad style.
 * </p>
 * <p>
 * The Java collection framework is built around {@link java.util.Collection},
 * which is already modifiable by design. The method {@link #asCollection()}
 * allows to use an indexable in cases where a {@code Collection} is expected,
 * and the {@link #asList()} method where a {@code List} is expected.
 * </p>
 * <p>
 * The other way round (get an indexable from a standard class or a standard array)
 * is given by {@link #viewList(List)} and {@link #viewArray(Object[])}. As the
 * returned Indeable is read-only, the former allows transparent conversion of the
 * element type to one of its super classes. The following code compiles without any
 * compiler warning:
 * </p>
 * <pre>
 * {@code
 *  List<Integer> integers = Arrays.asList(1, 2, 3);
 *  Indexable<Number> numbers = Indexable.viewList(integers);
 * }
 * </pre>
 *
 * <p>
 * The same is not possible with standard collection classes as their interface is basically
 * modifiable (even when wrapped in an unmodifiable wrapper). So a {@code List<Number>}
 * is different from a {@code ListInteger>} because to the former you could add
 * a double value, but to the latter not.
 * </p>
 * <h2>Factory Methods</h2>
 * <p>
 * The above factory methods <b>view</b> their content, and they even have counterparts which 
 * expect a {@code Function} which is mapping the incoming element type to a different
 * type which the {@code Indexable} exhibits. But viewing means that changes of the underlying
 * list or array will be reflected by the viewing indexable. To decouple the indexable
 * call its {@link Indexable#frozen()} method which will return an indexable which will not
 * change. Its elements itself might still change, though, if they are mutable.
 * </p>
 * <p>
 * Another way to get a frozen indexable is by using {@link Indexable#sorted(Comparator)},
 * which will return a frozen copy which is also sorted according to the given order
 * </p>
 * <p>
 * There are also factory methods which do a copy on creation, and which are also useful
 * to create an {@code Indexable} from non-indexable collections: 
 * </p>
 * <ul>
 *   <li>
 *     {@link Indexable#fromCollection(Collection)}:
 *     convert a collection into an indexable, can be called with a {@code List}
 *     if you want to make a copy on creation
 *   </li>
 *   <li>
 *     {@link Indexable#fromCollection(Collection, Function)}:
 *     convert a collection into an indexable while changing the elements,
 *     can be called with a {@code List} if you want to make a copy on creation
 *   </li>
 *   <li>
 *     {@link Indexable#fromCountable(Countable)}:
 *     convert a countable into an indexable
 *   </li>
 *   <li>
 *     {@link Indexable#fromCountable(Countable, Function)}:
 *     convert a countable into an indexable while changing the elements
 *   </li>
 *   <li>
 *     {@link Indexable#fromIterable(Iterable)}:
 *     convert any iterable into an indexable
 *   </li>
 *   <li>
 *     {@link Indexable#fromIterable(Iterable, int)}:
 *     convert any iterable into an indexable, with expected size
 *   </li>
 *   <li>
 *     {@link Indexable#fromIterable(Iterable, Function)}:
 *     convert any iterable into an indexable, while changing the element type
 *   </li>
 *   <li>
 *     {@link Indexable#fromIterable(Iterable, int, Function)}:
 *     convert any iterable into an indexable, while changing the element type, with expected size
 *   </li>
 *   <li>
 *     {@link Indexable#init(int, Supplier)}:
 *     creates an indexable with a given number of elements, provided by a supplier.
 *   </li>
 *   <li>
 *     {@link Indexable#viewByIndex(int, IntFunction)}:
 *     creates an indexable which will create the elements only when requested,
 *     but each time the same index is requested.
 *   </li>
 *   <li>
 *     {@link Indexable#initByIndex(int, IntFunction)}:
 *     creates an indexable with a given number of elements, created by an index-based initializer,
 *     where the elements are cached during creation.
 *   </li>
 *   <li></li>
 * </ul>
 * <p>
 * The {@link #emptyIndexable()} method provides a useful return value to indicate
 * emptiness.
 * </p>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 04, 2018
 * @param <T> type parameter
 * @see de.caff.generics.algorithm.FastSequenceSearch
 */
public interface Indexable<T>
        extends Countable<T>
{
  /**
   * Get the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @return element at the given index
   */
  T get(int index);

  /**
   * Check the index if it is between 0 (included) and the given size (excluded).
   * Throw an exception if not.
   * @param idx   index to check
   * @param size  size of indexable
   * @throws IndexOutOfBoundsException if {@code idx} is outside range
   */
  static void checkIndex(int idx, int size)
  {
    if (idx < 0  ||  idx >= size) {
      throw new IndexOutOfBoundsException(String.format("%s is not in range [0,%d[!",
                                                        idx, size));
    }
  }

  /**
   * Pythonesque get.
   * This allows accessing elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @return element at the given index
   */
  default T gyt(int index)
  {
    return get(Pythonesque.mapX(index, this));
  }

  /**
   * Get an element modulo size.
   * <p>
   * This maps the given {@code index} into the range of this indexable
   * by applying a modulo {@link #size()} operation. For empty indexable
   * this will throw an {@link IndexOutOfBoundsException} as there is no
   * possible index to get.
   * @param index index, possibly out of range, possibly even negative
   *              for Pythonesque access
   * @return element at the given, possibly modulated, index
   */
  default T getMod(int index)
  {
    final int size = size();
    if (size == 0) {
      throw new IndexOutOfBoundsException("No element for empty indexable!");
    }
    return gyt(index % size);
  }

  /**
   * Get this indexable but with inverted order.
   * @return a view of this indexable which accesses the elements in reverse order
   */
  @NotNull
  default Indexable<T> reverse()
  {
    return new Indexable.Base<T>()
    {
      @Override
      public int size()
      {
        return Indexable.this.size();
      }

      @Override
      public T get(int index)
      {
        return Indexable.this.get(Indexable.this.size() - index - 1);
      }

      @NotNull
      @Override
      public Indexable<T> reverse()
      {
        return Indexable.this;
      }
    };
  }

  /**
   * Is this indexable empty?
   * @return {@code true}: there are no elements in this indexable<br>
   *         {@code false}: this indexable has elements
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * Returns an iterator over elements of type {@code T}.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  default Iterator<T> iterator()
  {
    return listIterator();
  }

  /**
   * Get a subset iterator.
   * @param fromIndex  start index
   * @param toIndex    end index
   * @return subset iterator
   */
  @NotNull
  default Iterator<T> iterator(int fromIndex, int toIndex)
  {
    return subSet(fromIndex, toIndex).iterator();
  }

  /**
   * Returns a list iterator over elements of type {@code T}.
   *
   * @return a list iterator.
   */
  @NotNull
  default ListIterator<T> listIterator()
  {
    return new ListIterator<T>() {
      private int index = 0;

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public T next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public T previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(--index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(T t)
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void add(T t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  @Override
  default T first()
  {
    return get(0);
  }

  @Override
  default T last()
  {
    return gyt(-1);
  }

  /**
   * Get an indexable subset.
   * The subset includes indices {@code fromIndex} to {@code toIndex - 1}.
   * @param fromIndex start index of sub set
   * @param toIndex   index after last index
   * @return indexable subset view of this indexable
   */
  @NotNull
  default Indexable<T> subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0) {
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    }
    if (toIndex > size()) {
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    }
    if (fromIndex > toIndex) {
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    }
    final int length = toIndex - fromIndex;
    return length == 0
            ? emptyIndexable()
            : new Base<T>()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public T get(int index)
      {
        checkIndex(index, length);
        return Indexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public Indexable<T> subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return Indexable.this.subSet(fromIndex + fromIdx,
                                     fromIndex + toIdx);
      }
    };
  }

  /**
   * Get an indexable subset.
   * This is the Pythonesque version which allows negative indexes.
   * @param fromIndex start index of sub set
   * @param toIndex   end index of sub set
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Indexable<T> sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param fromIndex index to start with, negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Indexable<T> tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param toIndex index one after the end (equal to the length of the returned set),
   *                negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Indexable<T> headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Get a view of this indexable which behaves like a dictionary with {@code Integer} keys.
   * The valid keys are {@code 0} to {@code size() - 1}.
   * <p>
   * As the returned value is a view, it will reflect all changes of this indexable.
   * To avoid this either {@link Indexable#frozen() freeze this indexable} and call this method
   * on the result, or {@link Dict#frozen() freeze the returned Dict}.
   * The former is usually preferable as this may already be frozen, and then freezing will
   * just return {@code this}.
   *
   * @return dictionary view of this indexable
   */
  @NotNull
  default Dict<Integer, T> viewAsDict()
  {
    return new Dict.Base<Integer, T>() {
      @Nullable
      @Override
      public T get(Integer key)
      {
        if (key == null  ||  key < 0  ||  key >= size()) {
          return null;
        }
        return Indexable.this.get(key);
      }

      @Override
      public int size()
      {
        return Indexable.this.size();
      }

      @Override
      public boolean isEmpty()
      {
        return Indexable.this.isEmpty();
      }

      @NotNull
      @Override
      public Countable<T> values()
      {
        return Indexable.this;
      }

      @NotNull
      @Override
      public Countable<Integer> keys()
      {
        return Indexable.viewByIndex(size(), idx -> idx);
      }

      @Override
      public boolean hasKey(Integer key)
      {
        return key != null  &&  key >= 0  &&  key < size();
      }

      @NotNull
      @Override
      public Countable<Entry<Integer, T>> entries()
      {
        return Indexable.viewByIndex(size(), idx -> Entry.view(idx, Indexable.this.get(idx)));
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable collection.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   */
  @NotNull
  default Collection<T> asCollection()
  {
    return new AbstractCollection<T>()
    {
      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        return Indexable.this.iterator();
      }

      @Override
      public int size()
      {
        return Indexable.this.size();
      }
    };
  }

  /**
   * Get a view of this indexable as an unmodifiable list.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)} (but have a look at
   * {@link #addAllTo(Collection)}).
   * @return collection view
   * @see #toList()
   */
  @NotNull
  default List<T> asList()
  {
    return new ListView<>(this);
  }

  /**
   * Helper class for viewing a mutable indexable as a standard list.
   * The main purpose of this class is providing the RandomAccess modifier.
   */
  class ListView<T> extends AbstractList<T> implements RandomAccess
  {
    @NotNull
    private final Indexable<T> indexable;

    public ListView(@NotNull Indexable<T> indexable)
    {
      this.indexable = indexable;
    }

    public T get(int index)
    {
      return indexable.get(index);
    }

    @Override
    public int size()
    {
      return indexable.size();
    }

    @Override
    public boolean isEmpty()
    {
      return indexable.isEmpty();
    }

    @Override
    public Iterator<T> iterator()
    {
      return indexable.iterator();
    }
  }

  /**
   * Get the indexes of this indexable as an iterable range.
   * @return iterable indexes
   * @see #intIndexes()
   * @see Range#indexes(int)
   */
  @NotNull
  default Iterable<Integer> indexes()
  {
    return Range.indexes(size());
  }

  /**
   * Get the valid set of indexes as in integer indexable.
   * @return indexable indices of this indexable
   * @see #indexes()
   * @see IntIndexable#rangeFromSize(int)
   */
  @NotNull
  default IntIndexable intIndexes()
  {
    return IntIndexable.rangeFromSize(size());
  }

  /**
   * View this indexable as if it has another type.
   * @param mapper mapper used to convert the element type of this indexable to the expected type,
   *               will be applied on each access
   * @param <E> element type of returned view
   * @return view of this indexable with mapped elements, so any changes of this indexable will be reflected
   *         by the returned indexable. Use {@link #frozen()} on the resulting indexable to change this.
   */
  @NotNull
  @Override
  default <E> Indexable<E> view(@NotNull Function<? super T, ? extends E> mapper)
  {
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return Indexable.this.size();
      }

      @Override
      public E get(int index)
      {
        return mapper.apply(Indexable.this.get(index));
      }
    };
  }

  /**
   * View this indexable as if it has another type using a fragile mapping which might
   * throw a checked exception.
   * <p>
   * The returned indexable is evaluated lazily, so the exception might be triggered much later,
   * of even never if the element where the converted does throw an exception is never used.
   * <p>
   * If {@code mapper} is throwing an exception this exception is wrapped into a {@link WrappedFragileException}
   * which is thrown instead.
   * <p>
   * If the resulting indexable is only temporarily used, you can wrap its usage with a
   * try-catch construction (see below). If it is returned from your code the only way
   * to make the exception appear early is {@link #frozen() freezing} with the price
   * of an internal copy.
   * <p>
   * Example of wrapping the usage (assuming that method {@code Foo foo(Bar)} converts {@code Foo} objects
   * to {@code Bar} objects with the occasional {@code CheckedException} exception):
   * <pre>{@code
   *   try {
   *     Indexable<Bar> bars = // ...
   *     try {
   *       Indexable<Foo> foos = bars.viewFragile(bar -> foo(bar));
   *       // use foos only inside this block or
   *   //  return foos.frozen();
   *       // to trigger possible exceptions inside this block.
   *     } catch(WrappedFragileException x) {
   *       x.rethrow(CheckedException.class); // this will throw the caught exception
   *       // throw new RuntimeException("Never will reach here!");
   *     }
   *   }
   * }</pre>
   * @param mapper mapper from incoming to outgoing objects which might throw an exception
   * @param <R>    target type of the elements of the returned indexable
   * @param <E>    possible exception thrown by {@code mapper}
   * @return lazy view of this indexable which will seem to contain target type elements
   */
  @NotNull
  @Override
  default <R, E extends Exception> Indexable<R> viewFragile(@NotNull FragileFunction1<? extends R, E, ? super T> mapper)
  {
    return view(mapper.nonFragile());
  }

  /**
   * View this indexable as if it has another type.
   * Compared to {@link #view(Function)} this calls the mapper with an additional index argument.
   * @param mapper mapper which gets the index as first and the associated element as second argument,
   *               and will be applied on each access to return the element of the resulting indexable
   * @param <E> element type of returned view
   * @return view of this indexable with mapped elements, so any changes of this indexable will be reflected
   *         by the returned indexable. Use {@link #frozen()} on the resulting indexable to change this.
   */
  @NotNull
  default <E> Indexable<E> indexedView(@NotNull BiFunction<? super Integer, ? super T, ? extends E> mapper)
  {
    return new Base<E>()
    {
      @Override
      public E get(int index)
      {
        return mapper.apply(index, Indexable.this.get(index));
      }

      @Override
      public int size()
      {
        return Indexable.this.size();
      }
    };
  }

  /**
   * Add a part of the content of this indexable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  start position in the array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull T[] array, int arrayPos,
                         int index, int length)
  {
    for (T elem : subSet(index, index + length)) {
      array[arrayPos++] = elem;
    }
    return arrayPos;
  }

  /**
   * Add the complete content of this indexable to the gien array.
   * @param array    array where the content is added
   * @param arrayPos  start position in hte array where the content is added
   * @return array position after the added content
   */
  default int addToArray(@NotNull T[] array, int arrayPos)
  {
    return addToArray(array, arrayPos, 0, size());
  }

  /**
   * Add all elements of this indexable to the given collection.
   * @param collection collection for collecting this indexable
   * @deprecated use {@link #addAllTo(Collection)} instead
   */
  @Deprecated
  default void addToCollection(@NotNull Collection<? super T> collection)
  {
    forEach(collection::add);
  }

  /**
   * Call an action for each entry.
   * This is similar to {@link Iterable#forEach(Consumer)}, but also provides the index.
   * @param action action which will be called for each entry with index and element at that index
   * @see #forEachEntryFragile(FragileProcedure2)
   * @see #forEach(Consumer)
   */
  default void forEachEntry(@NotNull BiConsumer<Integer, ? super T> action)
  {
    final int sz = size();
    for (int index = 0;  index < sz;  ++index) {
      action.accept(index, get(index));
    }
  }

  /**
   * Call a fragile action for each entry.
   * This is similar to {@link #forEachEntry(BiConsumer)} but allows an action which might throw an exception.
   *
   * @param action action which will be called for each entry with index and associated element, might throw an exception
   * @param <E> exception type
   * @throws E if {@code action} throws
   */
  default <E extends Exception> void forEachEntryFragile(@NotNull FragileProcedure2<E, Integer, ? super T> action)
          throws E
  {
    final int sz = size();
    for (int index = 0;  index < sz;  ++index) {
      action.apply(index, get(index));
    }
  }

  /**
   * Get a spliterator on this indexable.
   * The returned spliterator will not have its {@link Spliterator#IMMUTABLE} flag set,
   * see {@link #frozenSpliterator()} for an alternative.
   * @return a spliterator on this indexable
   */
  @NotNull
  default Spliterator<T> spliterator()
  {
    return new IndexableSpliterator<>(this);
  }

  /**
   * Get an immutable spliterator on a frozen copy of this indexable.
   * If this indexable is already immutable (i.e. if {@link #frozen()} returns {@code this})
   * then this is the same as {@link #spliterator()} with the difference that the spliterator
   * returned here will have its {@link Spliterator#IMMUTABLE} flag set. If this is not immutable
   * a frozen copy will be created, and the spliterator will be operating on that. Please refer to
   * {@link #frozen()} to understand what level of immutabiliy it will provide.
   * @return an "immutable" spliterator with the possible cost of copying this indexable
   */
  @NotNull
  default Spliterator<T> frozenSpliterator()
  {
    final Indexable<T> frozen = frozen();
    return new IndexableSpliterator<>(frozen,
                                      0, frozen.size(),
                                      true);
  }

  /**
   * Is this indexable sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * See {@linkplain #isStrictlyOrdered(Ordering)} for a stricter alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isOrdered(@NotNull Ordering<? super T> order)
  {
    if (size() < 2) {
      return true;
    }
    T last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final T value = get(i);
      if (order.check(value, last) == Descending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered(Ordering)} for a more relaxed alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered(@NotNull Ordering<? super T> order)
  {
    if (size() < 2) {
      return true;
    }
    T last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final T value = get(i);
      if (order.check(value, last) != Ascending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Do a binary search in an ordered indexable.
   * This requires this indexable to be ordered in non-descending order
   * as defined by the given {@code order},
   * i.e. {@linkplain #isOrdered(Ordering)} has to return {@code true}
   * for this order.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(Ordering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @param order sort order
   * @return a positive integer defining the index where the given value was found,
   * and a negative integer if the value is not contained. The latter defines
   * the insertion point where the looked up value would match into this
   * sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(T value, @NotNull Ordering<? super T> order)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final T dp = get(mid);

      switch (order.check(dp, value)) {
      case Ascending:
        low = mid + 1;
        break;
      case Descending:
        hi = mid - 1;
        break;
      default:
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Create a (mutable) double indexable from the values of this indexable
   * which is ordered as defined by the given ordering.
   * @param order sort order
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableIndexable<T> ordered(@NotNull Ordering<? super T> order)
  {
    final MutableIndexable<T> result = MutableIndexable.fromIndexable(this);
    result.order(order);
    return result;
  }

  /**
   * Convert this into a base indexable.
   * Base indexables have useful default implementation for
   * {@link Object#equals(Object)}, {@link Object#hashCode()},
   * and {@link Object#toString()}.
   * @return base indexable
   */
  @NotNull
  default Base<T> asBase()
  {
    return new Indexable.Base<T>() {
      @Override
      public int size()
      {
        return Indexable.this.size();
      }

      @Override
      public T get(int index)
      {
        return Indexable.this.get(index);
      }

      @NotNull
      @Override
      public Indexable<T> reverse()
      {
        return Indexable.this.reverse();
      }

      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        return Indexable.this.iterator();
      }

      @NotNull
      @Override
      public ListIterator<T> listIterator()
      {
        return Indexable.this.listIterator();
      }

      @NotNull
      @Override
      public Indexable<T> subSet(int fromIndex, int toIndex)
      {
        return Indexable.this.subSet(fromIndex, toIndex);
      }
    };
  }

  /**
   * Do a linear forward search from the beginning.
   * This uses {@link Objects#deepEquals(Object, Object)} for
   * checking if a given element is the same as the searched one.
   * @param element  element to look for
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   */
  default int findFirst(T element)
  {
    return findNext(element, 0);
  }

  /**
   * Do a linear forward search from a given index and look for an
   * element which is the same as the given element.
   * This uses {@link Objects#deepEquals(Object, Object)} for
   * checking if a given element is the same as the searched one.
   * @param element  element to look for
   * @param start    start index to look from, pythonesque
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   */
  default int findNext(T element,
                       int start)
  {
    return findNext(element, Objects::deepEquals, start);
  }

  /**
   * Do a linear forward search from an element (or a substitute)
   * from the beginning.
   * @param object  element to look for
   * @param equality equality check used
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   * @param <U> type of search element
   */
  default <U> int findFirst(U object,
                            @NotNull Matcher<? super U, ? super T> equality)
  {
    return findNext(object, equality, 0);
  }

  /**
   * Do a linear forward search from a given index and look for an
   * element which is the same as the given object.
   * @param object  element to look for
   * @param equality equality check used
   * @param start    start index to look from, pythonesque
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   * @param <U> pattern object type, depending on the matcher this may differ
   *           from the element type
   */
  default <U> int findNext(U object,
                           @NotNull Matcher<? super U, ? super T> equality,
                           int start)
  {
    return findNext(equality.foldLeft(object), start);
  }

  /**
   * Do a linear forward search from the beginning.
   *
   * @param matcher test which returns {@code true} when a matching element is found
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   */
  default int findFirst(@NotNull Predicate<? super T> matcher)
  {
    return findNext(matcher, 0);
  }

  /**
   * Do a linear forward search from the beginning.
   * This method accepts a matcher which might throw a checked exception.
   * @param matcher test which returns {@code true} when a matching element is found
   * @param <X> exception type thrown by matcher
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   * @throws X if {@code matcher} throws
   */
  default <X extends Exception> int findFirstX(@NotNull FragilePredicate1<X, ? super T> matcher)
          throws X
  {
    return findNextX(matcher, 0);
  }

  /**
   * Do a linear forward search from the beginning.
   * @param matcher test which returns {@code true} when a matching element is found
   * @param start   start index where the search begins
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   */
  default int findNext(@NotNull Predicate<? super T> matcher, int start)
  {
    final int size = size();
    if (start < 0) {
      start += size;
      if (start < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           start - size, size));
      }
    }
    while (start < size) {
      if (matcher.test(get(start))) {
        return start;
      }
      ++start;
    }
    return -1;
  }

  /**
   * Do a linear forward search from the beginning.
   * @param matcher test which returns {@code true} when a matching element is found
   * @param start   start index where the search begins
   * @param <X> exception type thrown by matcher
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   * @throws X if {@code matcher} throws
   */
  default <X extends Exception> int findNextX(@NotNull FragilePredicate1<X, ? super T> matcher, int start)
          throws X
  {
    final int size = size();
    if (start < 0) {
      start += size;
      if (start < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           start - size, size));
      }
    }
    while (start < size) {
      if (matcher.test(get(start))) {
        return start;
      }
      ++start;
    }
    return -1;
  }

  /**
   * Get the first element which matches a given condition.
   * @param condition matching condition
   * @return the first element for which the condition returns {@code true},
   *         or {@code null} if no element fulfills the condition
   * @see Types#findFirst(Iterable, Predicate)
   */
  @Nullable
  default T firstMatch(@NotNull Predicate<? super T> condition)
  {
    return Types.findFirst(this, condition);
  }

  /**
   * Get the first element which matches a given fragile condition.
   * @param condition matching condition which might throw an exception
   * @return the first element for which the condition returns {@code true},
   *         or {@code null} if no element fulfills the condition
   * @param <X> exception type
   * @throws X if the condition throws
   */
  @Nullable
  default <X extends Exception> T firstMatchX(@NotNull FragilePredicate1<X, ? super T> condition)
          throws X
  {
    for (T elem : this) {
      if (condition.test(elem)) {
        return elem;
      }
    }
    return null;
  }

  /**
   * Get an adapted value from the first element which matches a given condition.
   * This method allows to avoid a {@code null} check in your code.
   * Indexables which may contain {@code null} values which the
   * {@code condition} considers a match should not use this method.
   * @param condition matching condition
   * @param adaptor   adaptor which adapts a possible match to your needs, only
   *                  called with non-null values
   * @param <R>       result type
   * @return first matching and adapted result or {@code null} if no match was found
   */
  @Nullable
  default <R> R firstMatch(@NotNull Predicate<? super T> condition,
                           @NotNull Function<? super T, ? extends R> adaptor)
  {
    final T match = firstMatch(condition);
    return match != null
            ? adaptor.apply(match)
            : null;
  }

  /**
   * Do a linear backward search from the end.
   * This uses {@link Objects#deepEquals(Object, Object)} for
   * checking if a given element is the same as the searched one.
   * @param element  element to look for
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   */
  default int findLast(T element)
  {
    return findPrevious(element, -1);
  }

  /**
   * Do a linear backward search from a given index and look for an
   * element which is the same as the given element.
   * This uses {@link Objects#deepEquals(Object, Object)} for
   * checking if a given element is the same as the searched one.
   * @param element  element to look for
   * @param start    start index to look from, pythonesque,
   *                 use {@code -1} to start with the last element
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   */
  default int findPrevious(T element,
                           int start)
  {
    return findPrevious(element, Objects::deepEquals, start);
  }

  /**
   * Do a linear backward search from an element (or a substitute)
   * from the beginning.
   * @param object  element to look for
   * @param equality equality check used
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   * @param <U> type of search element
   */
  default <U> int findLast(U object,
                           @NotNull Matcher<? super U, ? super T> equality)
  {
    return findPrevious(object, equality, -1);
  }

  /**
   * Do a linear backward search from a given index and look for an
   * element which is the same as the given object.
   * @param object  element to look for
   * @param equality equality check used
   * @param start    start index to look from, pythonesque
   * @return index found (non-pythonesque), {@code -1} if nothing was found
   * @param <U> pattern object type, depending on the matcher this may differ
   *           from the element type
   */
  default <U> int findPrevious(U object,
                               @NotNull Matcher<? super U, ? super T> equality,
                               int start)
  {
    return findPrevious(equality.foldLeft(object), start);
  }

  /**
   * Do a linear backward search from the beginning.
   * @param matcher matcher which returns {@code true} when a matching element is found
   * @return index of first matching element (non-pythonesque), {@code -1} if nothing was found
   */
  default int findLast(@NotNull Predicate<? super T> matcher)
  {
    return findPrevious(matcher, -1);
  }

  /**
   * Do a linear backward search from a given index and look for an
   * element which is the same as the given object.
   * @param matcher matcher which returns {@code true} when a matching element is found
   * @param start    start index to look from, pythonesque
   * @return index of first matching element (non-pythonesque), {@code -1} if nothing was found
   */
  default int findPrevious(@NotNull Predicate<? super T> matcher,
                           int start)
  {
    final int size = size();
    if (start < 0) {
      start += size;
      if (start < -1) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           start - size, size));
      }
    }
    while (start >= 0) {
      if (matcher.test(get(start))) {
        return start;
      }
      --start;
    }
    return -1;
  }

  /**
   * Find all elements that fulfill a given condition.
   * This will not return a view but a new copy containing
   * the filtered elements.
   * @param filter filter to apply
   * @return filtered elements
   */
  @NotNull
  default Indexable<T> findAll(@NotNull Predicate<? super T> filter)
  {
    return IndexableHelper.frozenFromList(Types.filter(this, filter));
  }

  /**
   * Find all elements that fulfill a given condition that might throw a checked exception.
   * This will not return a view but a new copy containing
   * the filtered elements.
   * @param filter filter to apply, might possibly throw
   * @param <X> exception type
   * @return filtered elements
   * @throws X if filter throws
   */
  @NotNull
  default <X extends Exception> Indexable<T> findAllX(@NotNull FragilePredicate1<X, ? super T> filter)
          throws X
  {
    final ArrayList<T> result = new ArrayList<>();
    for (T elem : this) {
      if (filter.test(elem)) {
        result.add(elem);
      }
    }
    return IndexableHelper.frozenFromList(result);
  }

  /**
   * Create a view of this indexable with an inserted element.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are inserted.
   * Use {@link ExpandableIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable might
   * result in a disaster when using the returned indexable!
   *
   * @param index index where the element is inserted before the current index
   *              (<b>not Pythonesque</b>, because it is possible to add an element at {@code index == size()},
   *              and it is expected that most insertion will happen at 0 or at the end)
   * @param item inserted item at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   * @see #withAppendedItem(Object)
   */
  @NotNull
  default Indexable<T> withInsertedItemAt(int index, T item)
  {
    final int insertIndex = index;
    final int newSize = size() + 1;
    if (insertIndex == 0) {
      return new Indexable.Base<T>() {
        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public T get(int index)
        {
          return index == 0
                  ? item
                  : Indexable.this.get(index - 1);
        }
      };
    }
    if (insertIndex == newSize - 1) {
      return new Indexable.Base<T>() {

        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public T get(int index)
        {
          return index == newSize - 1
                  ? item
                  : Indexable.this.get(index);
        }
      };
    }
    return new Indexable.Base<T>()
    {
      @Override
      public int size()
      {
        return newSize;
      }

      @Override
      public T get(int index)
      {
        if (index == insertIndex) {
          return item;
        }
        return Indexable.this.get(index < insertIndex
                                          ? index
                                          : index - 1);
      }
    };
  }

  /**
   * Create a view with of this indexable with another item added at the end.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are added.
   * @param item item to add
   * @return indexable view of this indexable, with the given item added at the end,
   *         so size is larger by 1
   * @see #withInsertedItemAt(int, Object)
   */
  @NotNull
  default Indexable<T> withAppendedItem(T item)
  {
    return withInsertedItemAt(size(), item);
  }

  @NotNull
  @Override
  default Indexable<T> withPrependedItem(T item)
  {
    return withInsertedItemAt(0, item);
  }

  /**
   * Create a view of this indexable with an exchanged element.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few items are exchanged.
   * Use {@link ExpandableIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable could
   * lead to unexpected results when using the returned indexable!
   *
   * @param index index where the element is exchanged, hiding the current element.
   *              (Pythonesque)
   * @param value exchanged value at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   */
  @NotNull
  default Indexable<T> withExchangedItemAt(int index, T value)
  {
    final int insertIndex = Pythonesque.mapX(index, this);
    return new Indexable.Base<T>()
    {
      @Override
      public T get(int index)
      {
        return index == insertIndex
                ? value
                : Indexable.this.get(index);
      }

      @Override
      public int size()
      {
        return Indexable.this.size();
      }
    };
  }

  /**
   * Create a view of this indexable with one item removed.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few items are removed.
   *
   * @param index index of the removed element (Pythonesque)
   * @return a view of this indexable where the item at the given index
   *         is removed
   */
  @NotNull
  default Indexable<T> withRemovedItemAt(int index)
  {
    final int removeIndex = Pythonesque.mapX(index, this);
    if (removeIndex == 0) {
      // remove first
      return tailSet(1);
    }
    final int sz = size() - 1;
    if (removeIndex == sz) {
      // remove last
      return headSet(-1);
    }
    return new Base<T>()
    {
      @Override
      public T get(int index)
      {
        return index < removeIndex
                ? Indexable.this.get(index)
                : Indexable.this.get(index + 1);
      }

      @Override
      public int size()
      {
        return sz;
      }
    };
  }

  /**
   * Create a view of this indexable with two items swapped.
   * Note that this is not efficient, but handy in certain situations
   * where only a few items are swapped.
   *
   * @param index1 index of the first element (Pythonesque)
   * @param index2 index of the second element (Pythonesque)
   * @return a view of this indexable where the first and second element
   *         have exchanged places
   */
  @NotNull
  default Indexable<T> withSwappedItemsAt(int index1, int index2)
  {
    final int swap1 = Pythonesque.mapX(index1, this);
    final int swap2 = Pythonesque.mapX(index2, this);
    if (swap1 == swap2) {
      return this;
    }
    return new Base<T>()
    {
      @Override
      public T get(int index)
      {
        if (index == swap1) {
          return Indexable.this.get(swap2);
        }
        if (index == swap2) {
          return Indexable.this.get(swap1);
        }
        return Indexable.this.get(index);
      }

      @Override
      public int size()
      {
        return Indexable.this.size();
      }
    };
  }

  /**
   * Get a rotated view of this indexable.
   * The returned indexable will show this indexable with rotated indexes.
   * A positive {@code steps} will rotate left, i.e. any index will be accessed
   * as if the number of steps is added before extraction (modulo length).
   * E.g. a rotation of {@code -1} will return the last element when index {@code 0} is
   * requested, and the first element for index {@code 1}.
   *
   * @param steps steps to rotate
   * @return rotated view of this indexable, use  {@link #frozen()}
   *         to create an indexable which no longer depends on this one
   */
  @NotNull
  default Indexable<T> rotated(int steps)
  {
    if (isEmpty()) {
      return this;
    }
    steps = steps % size();
    if (steps == 0) {
      return this;
    }
    if (steps < 0) {
      steps += size();
    }
    final int rotate = steps;
    return new Base<T>()
    {
      @Override
      public T get(int index)
      {
        return Indexable.this.get((index + rotate) % size());
      }

      @Override
      public int size()
      {
        return Indexable.this.size();
      }

      @NotNull
      @Override
      public Indexable<T> rotated(int steps)
      {
        return Indexable.this.rotated(rotate + steps);
      }
    };
  }

  /**
   * View this indexable as a double indexable.
   * @param extractor extractor for double values
   * @return double indexable view of this indexable
   */
  @NotNull
  default DoubleIndexable viewAsDoubleIndexable(@NotNull ToDoubleFunction<? super T> extractor)
  {
    return DoubleIndexable.viewIndexable(this, extractor);
  }

  /**
   * View this indexable as a float indexable.
   * @param extractor extractor for float values
   * @return float indexable view of this indexable
   */
  @NotNull
  default FloatIndexable viewAsFloatIndexable(@NotNull ToFloatFunction<? super T> extractor)
  {
    return FloatIndexable.viewIndexable(this, extractor);
  }

  /**
   * View this indexable as a long indexable.
   * @param extractor extractor for long values
   * @return long indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsLongIndexable(@NotNull ToLongFunction<? super T> extractor)
  {
    return LongIndexable.viewIndexable(this, extractor);
  }

  /**
   * View this indexable as an integer indexable.
   * @param extractor extractor for integer values
   * @return integer indexable view of this indexable
   */
  @NotNull
  default IntIndexable viewAsIntIndexable(@NotNull ToIntFunction<? super T> extractor)
  {
    return IntIndexable.viewIndexable(this, extractor);
  }

  /**
   * View this indexable as a short indexable.
   * @param extractor extractor for short values
   * @return short indexable view of this indexable
   */
  @NotNull
  default ShortIndexable viewAsShortIndexable(@NotNull ToShortFunction<? super T> extractor)
  {
    return ShortIndexable.viewIndexable(this, extractor);
  }

  /**
   * View this indexable as a char indexable.
   * @param extractor extractor for char values
   * @return char indexable view of this indexable
   */
  @NotNull
  default CharIndexable viewAsCharIndexable(@NotNull ToCharFunction<? super T> extractor)
  {
    return CharIndexable.viewIndexable(this, extractor);
  }

  /**
   * View this indexable as a byte indexable.
   * @param extractor extractor for byte values
   * @return byte indexable view of this indexable
   */
  @NotNull
  default ByteIndexable viewAsByteIndexable(@NotNull ToByteFunction<? super T> extractor)
  {
    return ByteIndexable.viewIndexable(this, extractor);
  }

  /**
   * View this indexable as a boolean indexable.
   * @param extractor extractor for boolean values
   * @return boolean indexable view of this indexable
   */
  @NotNull
  default BooleanIndexable viewAsBooleanIndexable(@NotNull Predicate<? super T> extractor)
  {
    return BooleanIndexable.viewIndexable(this, extractor);
  }

  /**
   * View a list as an Indexable.
   * <p>
   * Please note that this method is defined in a way that it allows transparent type conversion
   * of the elements' type to a super type. This is possible because the returned interface
   * is read-only. If you need a more complex conversion use {@link #viewList(List, Function)}.
   * @param list list used as a base
   * @param <E> element type
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   * @deprecated this was inconsistently named, use {@link #viewList(List)} instead
   */
  @NotNull
  @Deprecated
  static <E> Indexable<E> fromList(@NotNull List<? extends E> list)
  {
    return viewList(list);
  }

  /**
   * View a list as an Indexable.
   * <p>
   * Please note that this method is defined in a way that it allows transparent type conversion
   * of the elements' type to a super type. This is possible because the returned interface
   * is read-only. If you need a more complex conversion use {@link #viewList(List, Function)}.
   * <p>
   * You should only view larger lists when they are {@link RandomAccess random accessible}.
   * It is strongly recommended to {@link #frozen() freeze} views of other list types (like {@link LinkedList})
   * immediately.
   * <p>
   * The only guarantee given for non-random-accessible lists is that iterating is fast.
   * Using other methods of this interface might become awfully slow for non-random-accessible lists!
   * @param list list used as a base
   * @param <E> element type
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static <E> Indexable<E> viewList(@NotNull List<? extends E> list)
  {
    // NOTES:
    // Although it is generally bad to view a list w/o random access (like LinkedList) we try
    // to behave as gracefully as possible.
    // Therefore, a lot of methods are forwarded directly to the underlying list because the
    // default implementations use indexed access which will become unbearably slow for
    // larger linked lists.
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public E get(int index)
      {
        return list.get(index);
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }

      @NotNull
      @Override
      public Iterator<E> iterator()
      {
        // have to override to avoid exporting mutating method Iterator.remove()
        final Iterator<? extends E> iterator = list.iterator();
        return new Iterator<E>() // override to disable Iterator.remove()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public E next()
          {
            return iterator.next();
          }
        };
      }

      @NotNull
      @Override
      public ListIterator<E> listIterator()
      {
        // have to override to avoid exporting mutating methods ListIterator.remove(), .set() and .add()
        final ListIterator<? extends E> iterator = list.listIterator();
        return new ListIterator<E>()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public E next()
          {
            return iterator.next();
          }

          @Override
          public boolean hasPrevious()
          {
            return iterator.hasPrevious();
          }

          @Override
          public E previous()
          {
            return iterator.previous();
          }

          @Override
          public int nextIndex()
          {
            return iterator.nextIndex();
          }

          @Override
          public int previousIndex()
          {
            return iterator.previousIndex();
          }

          @Override
          public void remove()
          {
            throw new UnsupportedOperationException("remove");
          }

          @Override
          public void set(E e)
          {
            throw new UnsupportedOperationException("set");
          }

          @Override
          public void add(E e)
          {
            throw new UnsupportedOperationException("add");
          }
        };
      }

      @NotNull
      @Override
      public Indexable<E> subSet(int fromIndex, int toIndex)
      {
        return viewList(list.subList(fromIndex, toIndex));
      }

      @NotNull
      @Override
      public Collection<E> asCollection()
      {
        return Collections.unmodifiableCollection(list);
      }

      @NotNull
      @Override
      public List<E> asList()
      {
        return Collections.unmodifiableList(list);
      }
    };
  }

  /**
   * View a list which is possibly {@code null} as an indexable.
   * <p>
   * Please note that this method is defined in a way that it allows transparent type conversion
   * of the elements' type to a super type. This is possible because the returned interface
   * is read-only. If you need a more complex conversion use {@link #viewListN(List, Function)}.
   * @param list list used as a base
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable,
   *         or an {@linkplain #emptyIndexable() empty indexable}  if {@code list} is {@code null}
   * @param <E> element type
   */
  @NotNull
  static <E> Indexable<E> viewListN(@Nullable List<? extends E> list)
  {
    return list == null
            ? emptyIndexable()
            : viewList(list);
  }

  /**
   * View a list as an indexable of another type.
   * <p>
   * The mapper function will be applied on each access.
   * So it is strongly recommended to {@link #frozen() freeze} the result of this method
   * if it is used more than once.
   * <p>
   * You should only view larger lists when they are {@link RandomAccess random accessible}.
   * It is strongly recommended to {@link #frozen() freeze} views of other list types (like {@link LinkedList})
   * immediately. The implementation takes care to make iteration fast.
   * <p>
   * The only guarantee given for non-random-accessible lists is that iterating is fast.
   * Using other methods of this interface might become awfully slow for non-random-accessible lists!
   * @param list   list to view
   * @param mapper mapper from list element type to the element type of this indexable,
   *               it will be applied for each access
   * @param <E>  element type of this indexable
   * @param <V>  element type of the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned indexable
   */
  @NotNull
  static <E, V> Indexable<E> viewList(@NotNull List<V> list,
                                      @NotNull Function<? super V, ? extends E> mapper)
  {
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public E get(int index)
      {
        return mapper.apply(list.get(index));
      }
      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }

      @NotNull
      @Override
      public Iterator<E> iterator()
      {
        // have to override to avoid exporting mutating method Iterator.remove()
        final Iterator<V> iterator = list.iterator();
        return new Iterator<E>() // override to disable Iterator.remove()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public E next()
          {
            return mapper.apply(iterator.next());
          }
        };
      }

      @NotNull
      @Override
      public ListIterator<E> listIterator()
      {
        // have to override to avoid exporting mutating methods ListIterator.remove(), .set() and .add()
        final ListIterator<V> iterator = list.listIterator();
        return new ListIterator<E>()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public E next()
          {
            return mapper.apply(iterator.next());
          }

          @Override
          public boolean hasPrevious()
          {
            return iterator.hasPrevious();
          }

          @Override
          public E previous()
          {
            return mapper.apply(iterator.previous());
          }

          @Override
          public int nextIndex()
          {
            return iterator.nextIndex();
          }

          @Override
          public int previousIndex()
          {
            return iterator.previousIndex();
          }

          @Override
          public void remove()
          {
            throw new UnsupportedOperationException("remove");
          }

          @Override
          public void set(E e)
          {
            throw new UnsupportedOperationException("set");
          }

          @Override
          public void add(E e)
          {
            throw new UnsupportedOperationException("add");
          }
        };
      }

      @NotNull
      @Override
      public Indexable<E> subSet(int fromIndex, int toIndex)
      {
        return viewList(list.subList(fromIndex, toIndex), mapper);
      }

    };
  }

  /**
   * View a list as an indexable of another type.
   * <p>
   * Compared to {@linkplain #viewList(List, Function)} this accepts {@code null} for the list which is occasionally handy.
   * @param list   list to view
   * @param mapper mapper from list element type to the element type of this indexable,
   *               will be applied for each access
   * @param <E>  element type of this indexable
   * @param <V>  element type of the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned indexable,
   *         or an {@linkplain #emptyIndexable() empty indexable}  if {@code list} is {@code null}
   */
  @NotNull
  static <E, V> Indexable<E> viewListN(@Nullable List<V> list,
                                       @NotNull Function<? super V, ? extends E> mapper)
  {
    return list == null
            ? emptyIndexable()
            : viewList(list, mapper);
  }

  /**
   * View an array as an Indexable.
   * To view only part of the array use {@link #subSet(int, int)}.
   * @param array array used as base
   * @param <E> element type
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   * @deprecated this was inconsistently named, use {@link #viewArray(Object[])} instead
   */
  @SafeVarargs // because access is only readable
  @SuppressWarnings("varargs")
  @NotNull
  @Deprecated
  static <E> Indexable<E> fromArray(@NotNull E ... array)
  {
    return viewArray(array);
  }

  /**
   * View an array as an Indexable.
   * To view only part of the array use {@link #subSet(int, int)}.
   * @param array array used as base
   * @param <E> element type
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @SafeVarargs // because access is only readable
  @NotNull
  static <E> Indexable<E> viewArray(@NotNull E ... array)
  {
    return array.length == 0
            ? emptyIndexable()
            : new Base<E>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public E get(int index)
      {
        return array[index];
      }

      @NotNull
      @Override
      public Indexable<E> subSet(int fromIndex, int toIndex)
      {
        return Indexable.viewArray(array, fromIndex, toIndex - fromIndex);
      }
    };
  }

  /**
   * View a slice of an array.
   * @param array array to view
   * @param start start of array slice represented by the returned indexable
   * @param len   length of array slice represented by the returned indexable
   * @return indexable view of the given array slice, basically the same
   *         as {@code Indexable.viewArray(array).subset(start, start + len)}
   *
   * @param <E> array type
   */
  @NotNull
  static <E> Indexable<E> viewArray(@NotNull E[] array,
                                    int start,
                                    int len)
  {
    if (start < 0) {
      throw new IndexOutOfBoundsException("start has to be non-negative, but is "+start);
    }
    if (len < 0) {
      throw new IllegalArgumentException("len has to be non-negative, but is "+len);
    }
    if (start + len > array.length) {
      throw new IndexOutOfBoundsException(String.format("end will be outside array: %d + %d > %d",
                                                        start, len, array.length));
    }

    if (len == 0) {
      return emptyIndexable();
    }
    if (start == 0  &&  len == array.length) {
      return viewArray(array);
    }

    return new Base<E>()
    {
      @Override
      public E get(int index)
      {
        return array[start + index];
      }

      @Override
      public int size()
      {
        return len;
      }

      @NotNull
      @Override
      public Indexable<E> subSet(int fromIndex, int toIndex)
      {
        if (fromIndex < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
        }
        if (toIndex > size()) {
          throw new IndexOutOfBoundsException("toIndex = " + toIndex);
        }
        if (fromIndex > toIndex) {
          throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                             ") > toIndex(" + toIndex + ")");
        }
        return Indexable.viewArray(array,
                                   fromIndex + start,
                                   toIndex - fromIndex);
      }
    };
  }

  /**
   * Convert a collection into an indexable.
   * This will collect all elements from the {@code collection} and return them
   * as an indexable.
   * @param collection collection to iterate over
   * @param <E>      element type of the returned indexable, base type of the type
   *                 of the incoming collection
   * @return indexable created from the collection
   */
  @NotNull
  static <E> Indexable<E> fromCollection(@NotNull Collection<? extends E> collection)
  {
    return fromIterable(collection, collection.size());
  }

  /**
   * Convert a collection into an indexable.
   * This will iterate over the given {@code collection}, get its elements, convert them
   * using the {@code mapper}, and return the result as an indexable.  
   * @param collection collection to iterate over
   * @param mapper     mapper from incoming element type to outgoing element type
   * @param <E>        element type of the returned indexable
   * @param <V>        element type of the incoming iterable
   * @return indexable created from the collection
   */
  @NotNull
  static <E, V> Indexable<E> fromCollection(@NotNull Collection<V> collection,
                                            @NotNull Function<? super V, ? extends E> mapper)
  {
    return fromIterable(collection, collection.size(), mapper);
  }

  /**
   * Convert a countable into an indexable.
   * This will collect all elements from the {@code countable} and return them
   * as an indexable.
   * @param countable countable to iterate over
   * @param <E>      element type of the returned indexable, base type of the type
   *                 of the incoming countable
   * @return indexable created from the countable
   */
  @NotNull
  static <E> Indexable<E> fromCountable(@NotNull Countable<? extends E> countable)
  {
    return fromIterable(countable, countable.size());
  }

  /**
   * Convert a countable into an indexable.
   * This will iterate over the given {@code countable}, get its elements, convert them
   * using the {@code mapper}, and return the result as an indexable.  
   * @param countable countable to iterate over
   * @param mapper     mapper from incoming element type to outgoing element type
   * @param <E>        element type of the returned indexable
   * @param <V>        element type of the incoming iterable
   * @return indexable created from the countable
   */
  @NotNull
  static <E, V> Indexable<E> fromCountable(@NotNull Countable<V> countable,
                                           @NotNull Function<? super V, ? extends E> mapper)
  {
    return fromIterable(countable, countable.size(), mapper);
  }

  /**
   * Convert an iterable into an indexable.
   * This will collect all elements from the {@code iterable} and return them
   * as an indexable.
   * @param iterable iterable to iterate over
   * @param <E>      element type of the returned indexable, base type of the type
   *                 of the incoming iterable
   * @return indexable created from the iterable
   */
  @NotNull
  static <E> Indexable<E> fromIterable(@NotNull Iterable<? extends E> iterable)
  {
    if (iterable instanceof Sizeable) {
      if (iterable instanceof Indexable) {
        return Indexable.downCast(((Indexable<? extends E>) iterable).frozen());
      }
      return fromIterable(iterable, ((Sizeable)iterable).size(), e -> e);
    }

    if (iterable instanceof Collection) {
      return fromCollection((Collection<? extends E>)iterable);
    }
    
    return fromIterable(iterable, -1, e -> e);
  }

  /**
   * Convert an iterator to an indexable.
   * @param iterator iterator consumed by this method
   * @param <E>      element type of the returned indexable, base type of the type
   *                 of the incoming iterator
   * @return indexable created from the iterator
   */
  @NotNull
  static <E> Indexable<E> fromIterator(@NotNull Iterator<? extends E> iterator)
  {
    final ArrayList<E> list = new ArrayList<>();
    iterator.forEachRemaining(list::add);
    list.trimToSize();
    return viewList(list);
  }

  /**
   * Convert an iterable into an indexable.
   * This will iterate over the given {@code iterable}, get its elements, convert them
   * using the {@code mapper}, and return the result as an indexable.  
   * @param iterable iterable to iterate over
   * @param mapper   mapper from incoming element type to outgoing element type
   * @param <E>      element type of the returned indexable
   * @param <V>      element type of the incoming iterable
   * @return indexable created from the iterable
   */
  @NotNull
  static <E, V> Indexable<E> fromIterable(@NotNull Iterable<V> iterable,
                                          @NotNull Function<? super V, ? extends E> mapper)
  {
    return fromIterable(iterable, -1, mapper);
  }

  /**
   * Convert an iterable into an indexable.
   * This will collect all elements from the {@code iterable} and return them
   * as an indexable.
   * @param iterable iterable to iterate over
   * @param size     expected size of the iterable, use {@code 0} or negative for unknonw
   * @param <E>      element type of the returned indexable, base type of the type
   *                 of the incoming iterable
   * @return indexable created from the iterable
   */
  @NotNull
  static <E> Indexable<E> fromIterable(@NotNull Iterable<? extends E> iterable,
                                       int size)
  {
    return fromIterable(iterable, size, e -> e);
  }

  /**
   * Convert an iterable into an indexable.
   * This will iterate over the given {@code iterable}, get its elements, convert them
   * using the {@code mapper}, and return the result as an indexable.  
   * @param iterable iterable to iterate over
   * @param size     expected size of the iterable, use {@code 0} or negative for unknown
   * @param mapper   mapper from incoming element type to outgoing element type
   * @param <E>      element type of the returned indexable
   * @param <V>      element type of the incoming iterable
   * @return indexable created from the iterable
   */
  @NotNull
  static <E, V> Indexable<E> fromIterable(@NotNull Iterable<V> iterable,
                                          int size,
                                          @NotNull Function<? super V, ? extends E> mapper)
  {
    final ArrayList<E> list = Types.map(new ArrayList<>(size > 0 ? size : 8),
                                        iterable, mapper);
    list.trimToSize();
    return IndexableHelper.frozenFromList(list);
  }

  /**
   * Convert an enumeration into an indexable.
   * This will collect all elements from the {@code enumeration} and return them
   * as an indexale.
   * @param enumeration enumeration to iterate over
   * @param size     expected size of the enumeration, use {@code 0} or negative for unknonw
   * @param <E>      element type of the returned indexable, base type of the type
   *                 of the incoming enumeration
   * @return indexable created from the enumeration
   */
  @NotNull
  static <E> Indexable<E> fromEnumeration(@NotNull Enumeration<? extends E> enumeration,
                                          int size)
  {
    return fromEnumeration(enumeration, size, e -> e);
  }

  /**
   * Convert an enumeration into an indexable.
   * This will iterate over the given {@code enumeration}, get its elements, convert them
   * using the {@code mapper}, and return the result as an indexable.
   * @param enumeration enumeration to iterate over
   * @param size     expected size of the enumeration, use {@code 0} or negative for unknown
   * @param mapper   mapper from incoming element type to outgoing element type
   * @param <E>      element type of the returned indexable
   * @param <V>      element type of the incoming enumeration
   * @return indexable created from the enumeration
   */
  @NotNull
  static <E, V> Indexable<E> fromEnumeration(@NotNull Enumeration<V> enumeration,
                                             int size,
                                             @NotNull Function<? super V, ? extends E> mapper)
  {
    final ArrayList<E> list = Types.map(new ArrayList<>(size > 0 ? size : 8),
                                        enumeration, mapper);
    list.trimToSize();
    return IndexableHelper.frozenFromList(list);
  }

  /**
   * Downcast an indexable of a given type so that it appears at an indexable of a super type.
   * This method does nothing, it only returns its argument.
   * But using it keeps the compiler happy.
   * <p>
   * Downcasting indexables is no problem, as they don't allow adding new elements.
   * <p>
   * Example:
   * <blockquote><pre>{@code
   * Indexable<String> a = Indexable.viewArray("a", "b", "c");
   * Indexable<Object> b = Indexable.downCast(a); // no compiler warning here
   * assert a == b;
   * }</pre></blockquote>
   *
   * @param indexable indexable to be downcasted, needs to have a type which extends or implements {@code <E>}
   * @param <E> target type of returned indexable
   * @return {@code indexable}, but using a different element type
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Indexable does not allow adding
  static <E> Indexable<E> downCast(@NotNull Indexable<? extends E> indexable)
  {
    return (Indexable<E>)indexable;
  }

  /**
   * Create an indexable with only one element.
   * @param singleElement single element
   * @param <E> element type
   * @return indexable view of the given element
   */
  @NotNull
  static <E> Indexable<E> singleton(E singleElement)
  {
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public E get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException("index is "+index+" for size 1!");
        }
        return singleElement;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public Indexable<E> frozen()
      {
        return this;
      }

      @NotNull
      @Override
      public Indexable<E> reverse()
      {
        return this;
      }

      @NotNull
      @Override
      public Indexable<E> rotated(int steps)
      {
        return this;
      }

      @NotNull
      @Override
      public Collection<E> asCollection()
      {
        return Collections.singletonList(singleElement);
      }

      @NotNull
      @Override
      public List<E> asList()
      {
        return Collections.singletonList(singleElement);
      }

      @NotNull
      @Override
      public Iterable<Integer> indexes()
      {
        return IntIndexable.SINGLE_0;
      }

      @NotNull
      @Override
      public IntIndexable intIndexes()
      {
        return IntIndexable.SINGLE_0;
      }

      @NotNull
      @Override
      public Indexable<E> sorted(@NotNull Comparator<? super E> order)
      {
        return Indexable.singleton(singleElement);
      }

      @Override
      public boolean isSorted(@NotNull Comparator<? super E> order)
      {
        return true;
      }

      @Override
      public boolean isStrictlySorted(@NotNull Comparator<? super E> order)
      {
        return true;
      }

      @Override
      public E first()
      {
        return singleElement;
      }

      @Nullable
      @Override
      public E firstOrNull()
      {
        return singleElement;
      }

      @NotNull
      @Override
      public Optional<E> optFirst()
      {
        return Optional.of(singleElement);
      }

      @Override
      public E last()
      {
        return singleElement;
      }

      @Override
      public E lastOrNull()
      {
        return singleElement;
      }

      @NotNull
      @Override
      public Optional<E> optLast()
      {
        return Optional.of(singleElement);
      }

      @NotNull
      @Override
      public <K> Dict<K, Indexable<E>> groupingBy(@NotNull Function<? super E, ? extends K> keyExtractor)
      {
        return Dict.singleton(keyExtractor.apply(singleElement), Indexable.singleton(singleElement));
      }

      @NotNull
      @Override
      public <K, V> Dict<K, Indexable<V>> groupingBy(@NotNull Function<? super E, ? extends K> keyExtractor,
                                                     @NotNull Function<? super E, ? extends V> valueMapper)
      {
        return Dict.singleton(keyExtractor.apply(singleElement),
                              Indexable.singleton(valueMapper.apply(singleElement)));
      }

      @NotNull
      @Override
      public <K> Dict<K, E> mappingBy(boolean firstWins, @NotNull Function<? super E, ? extends K> keyExtractor)
      {
        return Dict.singleton(keyExtractor.apply(singleElement), singleElement);
      }

      @NotNull
      @Override
      public <K, V> Dict<K, V> mappingBy(boolean firstWins, @NotNull Function<? super E, ? extends K> keyExtractor,
                                         @NotNull Function<? super E, ? extends V> valueMapper)
      {
        return Dict.singleton(keyExtractor.apply(singleElement),
                              valueMapper.apply(singleElement));
      }
    };
  }

  /**
   * Get an indexable with a single or no element.
   *
   * @param optionalElement optional element
   * @param <E> element type
   * @return indexable with size 1 with the given element if it is not {@code null},
   *         or an empty countable if it is {@code null}
   */
  @NotNull
  static <E> Indexable<E> optional(@Nullable E optionalElement)
  {
    return optionalElement != null
            ? singleton(optionalElement)
            : emptyIndexable();
  }

  /**
   * Convert a standard Optional to an Indexable with either one or no element.
   * @param optional optional to convert to indexable
   * @param <E> optional/indexable type
   * @return indexable with either 1 element (if the optional has an element)
   *         or 0 elements (if the optional has no element)
   */
  @NotNull
  static <E> Indexable<E> fromOptional(@NotNull Optional<E> optional)
  {
    return optional.map(Indexable::singleton).orElseGet(Indexable::emptyIndexable);
  }

  /**
   * View an array as an indexable of another type.
   * @param array  array to view
   * @param mapper mapper from array element type to the element type of this indexable,
   *               will be applied for each access
   * @param <E>  element type of this indexable
   * @param <V>  element type of the array
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned indexable
   */
  @NotNull
  @SafeVarargs // because access is only readable
  static <E, V> Indexable<E> mapArray(@NotNull Function<? super V, ? extends E> mapper,
                                      @NotNull V ... array)
  {
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public E get(int index)
      {
        return mapper.apply(array[index]);
      }
    };
  }

  /**
   * View an indexable as if it has entries belonging to a super class.
   * This allows to use an indexable with a different element type in places where
   * an indexable with a super element type is expected.
   * @param indexable indexable to be viewed
   * @param <V> super type of the incoming {@code indexable}
   * @return {@code indexable}, but with adapted entry type
   */
  @NotNull
  @SuppressWarnings("unchecked") // as access is read-only
  static <V> Indexable<V> view(@NotNull Indexable<? extends V> indexable)
  {
    return (Indexable<V>)indexable;
  }

  /**
   * Create an indexable with a given size, and initialize its elements.
   * @param size     size of the indexable
   * @param producer element producer which will be called for each element
   * @param <V> element type
   * @return indexable with the given size containing the elements created by the producer
   */
  @NotNull
  @SuppressWarnings("unchecked") // because array is used read-only outside of this method
  static <V> Indexable<V> init(final int size,
                               @NotNull Supplier<? extends V> producer)
  {
    final Object[] array = new Object[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = producer.get();
    }
    return (Indexable<V>)IndexableHelper.frozenFromArray(array);
  }

  /**
   * Initialize an indexable to a given size while allowing exceptions.
   * @param size     size of indexable
   * @param creator  creator for the initial elements, called in order for each element, may throw an exception
   * @param <E> element type
   * @param <X> exception type
   * @return indexable with the given size
   * @throws X if {@code creator} throws
   */
  @NotNull
  @SuppressWarnings("unchecked") // okay because array restrictions are kept
  static <E, X extends Exception> Indexable<E> initFragile(int size,
                                                           @NotNull FragileFunction0<E, X> creator)
          throws X
  {
    final Object[] array = new Object[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = creator.apply();
    }
    return (Indexable<E>)IndexableHelper.frozenFromArray(array);
  }

  /**
   * Create an indexable with a given size which returns its elements by calling a producer.
   * This will always call the {@code producer} when an element is requested.
   * Use {@link #frozen()} on the result to create an indexable which avoids this.
   * @param size     size of the indexable
   * @param producer element producer which will be called for each index
   * @param <V> element type
   * @return an indexable with the given size which will return elements by calling the {@code producer}
   * @see #initByIndex(int, IntFunction)
   */
  @NotNull
  static <V> Indexable<V> viewByIndex(final int size,
                                      @NotNull IntFunction<? extends V> producer)
  {
    if (size == 0) {
      return emptyIndexable();
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new Base<V>()
    {
      @Override
      public V get(int index)
      {
        checkIndex(index, size);
        return producer.apply(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Create an indexable with a given size, and initialize its elements by index.
   * @param size     size of the indexable
   * @param producer element producer which will be called for each index
   * @param <V> element type
   * @return indexable with the given size containing the elements created by the producer
   * @see #viewByIndex(int, IntFunction)
   */
  @NotNull
  static <V> Indexable<V> initByIndex(final int size,
                                      @NotNull IntFunction<? extends V> producer)
  {
    final Indexable<V> indexable = viewByIndex(size, producer); // Java 8 needs this intermediate step
    return indexable.frozen();
  }

  /** String representation of an empty indexable. */
  String EMPTY_INDEXABLE_STRING = "[]";
  /**
   * An empty indexable.
   * Use {@link #emptyIndexable()} instead for compiler-safe access.
   */
  @NotNull
  Indexable<?> EMPTY = new Base<Object>()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public Object get(int index)
    {
      throw new IndexOutOfBoundsException("Empty indexable has no elements!");
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public Indexable<Object> frozen()
    {
      return this;
    }

    @NotNull
    @Override
    public Indexable<Object> sorted(@NotNull Comparator<? super Object> order)
    {
      return super.sorted(order);
    }

    @Override
    public boolean isSorted(@NotNull Comparator<? super Object> order)
    {
      return true;
    }

    @Override
    public boolean isStrictlySorted(@NotNull Comparator<? super Object> order)
    {
      return true;
    }

    @NotNull
    @Override
    public Object[] toArray()
    {
      return Empty.OBJECT_ARRAY;
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public Indexable<Object> reverse()
    {
      return this;
    }

    @NotNull
    @Override
    public Indexable<Object> rotated(int steps)
    {
      return this;
    }

    @NotNull
    @Override
    public Iterator<Object> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public ListIterator<Object> listIterator()
    {
      return Types.emptyListIterator();
    }

    @Override
    public void addAllTo(@NotNull Collection<? super Object> collection)
    {
    }

    @Override
    @NotNull
    public String toString()
    {
      return EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }

    @Override
    @SuppressWarnings("unchecked")
    public boolean equals(Object obj)
    {
      return (obj instanceof Indexable  &&  ((Indexable<Object>)obj).isEmpty());
    }

    @Override
    public void forEach(Consumer<? super Object> action)
    {
    }

    @Override
    public <E extends Exception> void forEachFragile(@NotNull FragileProcedure1<E, ? super Object> consumer) throws E
    {
    }

    @Override
    public <V> V foldLeft(V initialValue, @NotNull BiFunction<? super V, ? super Object, ? extends V> folder)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public Collection<Object> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Object> asList()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public <E> Indexable<E> view(@NotNull Function<? super Object, ? extends E> mapper)
    {
      return emptyIndexable();
    }

    @NotNull
    @Override
    public <R, E extends Exception> Indexable<R> viewFragile(@NotNull FragileFunction1<? extends R, E, ? super Object> mapper)
    {
      return emptyIndexable();
    }

    @NotNull
    @Override
    public <E> Indexable<E> indexedView(@NotNull BiFunction<? super Integer, ? super Object, ? extends E> mapper)
    {
      return emptyIndexable();
    }

    @NotNull
    @Override
    public Indexable<Object> frozen(@NotNull Function<? super Object, ?> elementCloner, boolean cloneOnGet)
    {
      return emptyIndexable();
    }

    @Override
    public int addToArray(@NotNull Object[] array, int arrayPos, int index, int length)
    {
      return arrayPos;
    }

    @Override
    public int addToArray(@NotNull Object[] array, int arrayPos)
    {
      return arrayPos;
    }

    @Override
    public void addToCollection(@NotNull Collection<? super Object> collection)
    {
    }

    @Override
    public void forEachEntry(@NotNull BiConsumer<Integer, ? super Object> action)
    {
    }

    @Override
    public <E extends Exception> void forEachEntryFragile(@NotNull FragileProcedure2<E, Integer, ? super Object> action)
            throws E
    {
    }

    @Override
    public int findFirst(Object element)
    {
      return -1;
    }

    @Override
    public boolean containsEq(Object elem)
    {
      return false;
    }

    @Override
    public boolean containsRef(Object elem)
    {
      return false;
    }

    @Override
    public boolean hasAny(@NotNull Predicate<? super Object> check)
    {
      return false;
    }

    @Nullable
    @Override
    public Object getAny(@NotNull Predicate<? super Object> check)
    {
      return null;
    }

    @Override
    public boolean hasAll(@NotNull Predicate<? super Object> check)
    {
      return true;
    }

    @Override
    public Object first()
    {
      throw new NoSuchElementException("No first element in empty Indexable!");
    }

    @Nullable
    @Override
    public Object firstOrNull()
    {
      return null;
    }

    @NotNull
    @Override
    public Optional<Object> optFirst()
    {
      return Optional.empty();
    }

    @Override
    public Object last()
    {
      throw new NoSuchElementException("No last element in empty Indexable!");
    }

    @Override
    public Object lastOrNull()
    {
      return null;
    }

    @NotNull
    @Override
    public Optional<Object> optLast()
    {
      return Optional.empty();
    }

    @Override
    public <U> int findFirst(U object, @NotNull Matcher<? super U, ? super Object> equality)
    {
      return -1;
    }

    @Override
    public int findFirst(@NotNull Predicate<? super Object> matcher)
    {
      return -1;
    }

    @Override
    public int findLast(Object element)
    {
      return -1;
    }

    @Override
    public <U> int findLast(U object, @NotNull Matcher<? super U, ? super Object> equality)
    {
      return -1;
    }

    @Override
    public int findLast(@NotNull Predicate<? super Object> matcher)
    {
      return -1;
    }

    @NotNull
    @Override
    public Indexable<Object> findAll(@NotNull Predicate<? super Object> filter)
    {
      return Indexable.emptyIndexable();
    }

    @NotNull
    @Override
    public Dict<Integer, Object> viewAsDict()
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public DoubleIndexable viewAsDoubleIndexable(@NotNull ToDoubleFunction<? super Object> extractor)
    {
      return DoubleIndexable.EMPTY;
    }

    @NotNull
    @Override
    public FloatIndexable viewAsFloatIndexable(@NotNull ToFloatFunction<? super Object> extractor)
    {
      return FloatIndexable.EMPTY;
    }

    @NotNull
    @Override
    public LongIndexable viewAsLongIndexable(@NotNull ToLongFunction<? super Object> extractor)
    {
      return LongIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable viewAsIntIndexable(@NotNull ToIntFunction<? super Object> extractor)
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public ShortIndexable viewAsShortIndexable(@NotNull ToShortFunction<? super Object> extractor)
    {
      return ShortIndexable.EMPTY;
    }

    @NotNull
    @Override
    public CharIndexable viewAsCharIndexable(@NotNull ToCharFunction<? super Object> extractor)
    {
      return CharIndexable.EMPTY;
    }

    @NotNull
    @Override
    public ByteIndexable viewAsByteIndexable(@NotNull ToByteFunction<? super Object> extractor)
    {
      return ByteIndexable.EMPTY;
    }

    @NotNull
    @Override
    public BooleanIndexable viewAsBooleanIndexable(@NotNull Predicate<? super Object> extractor)
    {
      return BooleanIndexable.EMPTY;
    }

    @NotNull
    @Override
    public Object[] toArray(@NotNull Class<Object> type)
    {
      return Empty.OBJECT_ARRAY;
    }

    @NotNull
    @Override
    public Iterable<Object> filtered(@NotNull Predicate<? super Object> filter)
    {
      return this;
    }

    @NotNull
    @Override
    public Countable<Object> filterToCountable(@NotNull Predicate<? super Object> filter)
    {
      return this;
    }

    @NotNull
    @Override
    public Indexable<Object> filteredToIndexable(@NotNull Predicate<? super Object> filter)
    {
      return this;
    }

    @NotNull
    @Override
    public <K> Dict<K, Indexable<Object>> groupingBy(@NotNull Function<? super Object, ? extends K> keyExtractor)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public <K, V> Dict<K, Indexable<V>> groupingBy(@NotNull Function<? super Object, ? extends K> keyExtractor,
                                                   @NotNull Function<? super Object, ? extends V> valueMapper)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public <K> Dict<K, Object> mappingBy(boolean firstWins, @NotNull Function<? super Object, ? extends K> keyExtractor)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public <K, V> Dict<K, V> mappingBy(boolean firstWins, @NotNull Function<? super Object, ? extends K> keyExtractor,
                                       @NotNull Function<? super Object, ? extends V> valueMapper)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public Spliterator<Object> spliterator()
    {
      return Spliterators.emptySpliterator();
    }

    @NotNull
    @Override
    public Stream<Object> stream()
    {
      return Stream.empty();
    }

    @NotNull
    @Override
    public Stream<Object> parellelStream()
    {
      return Stream.empty();
    }
  };

  /**
   * Get an empty indexable set.
   * @param <E> element type
   * @return empty indexable
   */
  @SuppressWarnings("unchecked")
  @NotNull
  static <E> Indexable.Base<E> emptyIndexable()
  {
    return (Indexable.Base<E>)EMPTY; // no problem as there are no elements
  }

  /**
   * Combine a set of indexables into one large indexable.
   *
   * @param parts parts of the returned indexable
   * @return combined indexable
   * @param <E> element type of the returned indexable
   */
  @NotNull
  @SafeVarargs
  static <E> Indexable<E> combine(@NotNull Indexable<? extends E>... parts)
  {
    if (parts.length == 0) {
      return Indexable.emptyIndexable();
    }
    
    return new Indexable.Base<E>()
    {
      @Override
      public E get(int index)
      {
        if (index < 0) {
          throw new IndexOutOfBoundsException("Non-positive: "+index);
        }
        int runIdx = index;
        for (Indexable<? extends E> part : parts) {
          if (runIdx < part.size()) {
            return part.get(runIdx);
          }
          runIdx -= part.size();
        }
        throw new IndexOutOfBoundsException(String.format("Index %d too large for size %d", index, size()));
      }

      @Override
      public int size()
      {
        int sz = 0;
        for (Indexable<? extends E> part : parts) {
          sz += part.size();
          if (sz < 0) {
            throw new RuntimeException("Combination of indexables has become too large!");
          }
        }
        return sz;
      }
    };
  }

  /**
   * Create a string representation of the given indexable.
   * @param indexable indexable
   * @return string representation
   * @deprecated use {@link Countable#toString(Countable)}
   */
  @NotNull
  @Deprecated
  static String toString(@NotNull Indexable<?> indexable)
  {
    return Countable.toString(indexable);
  }

  /**
   * Are two indexables equal?
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @param equalityChecker checker for equality of both objects
   * @param <E> parameter type of equality checker, and super class of
   *           both indexables' element type
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   * @deprecated  use {@link Countable#equal(Countable, Countable, BiPredicate)} instead
   */
  @Deprecated
  static <E> boolean equal(@NotNull Indexable<? extends E> indexable1,
                           @NotNull Indexable<? extends E> indexable2,
                           @NotNull BiPredicate<E, E> equalityChecker)
  {
    return Countable.equal(indexable1, indexable2, equalityChecker);
  }

  /**
   * Compare two indexables.
   * This compares the two indexables lexically element by element
   * using {@code comparator}
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @param comparator type
   * @param <E> comparator type, needs to be a base type of
   *           both indexables' element type
   * @return {@code < 0>} if {@code indexable1 < indexable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   * @deprecated  use {@link Types#lexicalCompare(Iterable, Iterable, Comparator)} instead
   */
  @Deprecated
  static <E> int compare(@NotNull Indexable<? extends E> indexable1,
                         @NotNull Indexable<? extends E> indexable2,
                         @NotNull Comparator<E> comparator)
  {
    return Types.lexicalCompare(indexable1, indexable2, comparator);
  }

  /**
   * Calculate a hashcode for a generic indexable.
   * @param indexable indexable for which the hash code is required
   * @return hash code for the given indexable
   * @deprecated use {@link Types#hash(Iterable)} instead
   */
  @Deprecated
  static int hash(@NotNull Indexable<?> indexable)
  {
    return Types.hash(indexable);
  }

  /**
   * Wrap an indexable with one which caches the hash value.
   * This is useful if indexables are used as hash keys because hash value calculation
   * is expensive.
   * <p>
   * The wrapped indexable must not change after it is wrapped, otherwise strange things are expected to happen.
   * @param indexable wrapped indexable, must not change
   * @param <E> element type of indexable
   * @return indexable which forwards most methods to {@code indexable}, but also provides
   *         useful implementations for {@link Object#hashCode()}, {@link Object#equals(Object)},
   *         and {@link Object#toString()}
   */
  @NotNull
  static <E> Indexable<E> withCachedHash(@NotNull Indexable<E> indexable)
  {
    final int hashCode = hash(indexable);
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public E get(int index)
      {
        return indexable.get(index);
      }

      @NotNull
      @Override
      public Iterator<E> iterator()
      {
        return indexable.iterator();
      }

      @NotNull
      @Override
      public ListIterator<E> listIterator()
      {
        return indexable.listIterator();
      }

      @NotNull
      @Override
      public Indexable<E> subSet(int fromIndex, int toIndex)
      {
        return indexable.subSet(fromIndex, toIndex);
      }

      @Override
      public int hashCode()
      {
        return hashCode;
      }
    };
  }

  /**
   * View a 2-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 2 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple2<? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple2!");
        }
      }

      @Override
      public int size()
      {
        return 2;
      }
    };
  }

  /**
   * View a 3-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 3 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple3<? extends C, ? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        case 2:
          return tuple._3();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple3!");
        }
      }

      @Override
      public int size()
      {
        return 3;
      }
    };
  }

  /**
   * View a 4-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 4 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple4<? extends C, ? extends C, ? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        case 2:
          return tuple._3();
        case 3:
          return tuple._4();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple4!");
        }
      }

      @Override
      public int size()
      {
        return 4;
      }
    };
  }

  /**
   * View a 5-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 5 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple5<? extends C, ? extends C, ? extends C, ? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        case 2:
          return tuple._3();
        case 3:
          return tuple._4();
        case 4:
          return tuple._5();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple5!");
        }
      }

      @Override
      public int size()
      {
        return 5;
      }
    };
  }

  /**
   * View a 6-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 6 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple6<? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        case 2:
          return tuple._3();
        case 3:
          return tuple._4();
        case 4:
          return tuple._5();
        case 5:
          return tuple._6();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple6!");
        }
      }

      @Override
      public int size()
      {
        return 6;
      }
    };
  }

  /**
   * View a 7-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 7 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple7<? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        case 2:
          return tuple._3();
        case 3:
          return tuple._4();
        case 4:
          return tuple._5();
        case 5:
          return tuple._6();
        case 6:
          return tuple._7();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple7!");
        }
      }

      @Override
      public int size()
      {
        return 7;
      }
    };
  }

  /**
   * View a 8-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 8 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple8<? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        case 2:
          return tuple._3();
        case 3:
          return tuple._4();
        case 4:
          return tuple._5();
        case 5:
          return tuple._6();
        case 6:
          return tuple._7();
        case 7:
          return tuple._8();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple8!");
        }
      }

      @Override
      public int size()
      {
        return 8;
      }
    };
  }

  /**
   * View a 9-element tuple as an indexable.
   * @param tuple tuple to view
   * @param <C> common super type of the tuple's types
   * @return indexable with 9 elements
   */
  @NotNull
  static <C> Indexable<C> viewTuple(@NotNull ITuple9<? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C, ? extends C> tuple)
  {
    return new Base<C>()
    {
      @Override
      public C get(int index)
      {
        switch (index) {
        case 0:
          return tuple._1();
        case 1:
          return tuple._2();
        case 2:
          return tuple._3();
        case 3:
          return tuple._4();
        case 4:
          return tuple._5();
        case 5:
          return tuple._6();
        case 6:
          return tuple._7();
        case 7:
          return tuple._8();
        case 8:
          return tuple._9();
        default:
          throw new IndexOutOfBoundsException("Trying to get ["+index+"] from Tuple9!");
        }
      }

      @Override
      public int size()
      {
        return 9;
      }
    };
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   * @param <TT> element type
   * @see Indexable#asBase()
   */
  abstract class Base<TT>
          extends Countable.Base<TT>
          implements Indexable<TT>
  {
    @Override
    @NotNull
    public Indexable.Base<TT> asBase()
    {
      return this;
    }
  }

  /**
   * A spliterator for indexables.
   * @param <TElem> element type
   */
  class IndexableSpliterator<TElem>
          implements Spliterator<TElem>
  {
    @NotNull
    private final Indexable<TElem> indexable;
    private int index;
    private final int fence;
    private final int character;

    /**
     * Instantiate a mutable spliterator for an indexable.
     * @param indexable indexable of this spliterator
     */
    public IndexableSpliterator(@NotNull Indexable<TElem> indexable)
    {
      this(indexable, 0, indexable.size(), false);
    }

    /**
     * Instantiate a spliterator for part of an indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param immutable is the underlying indexable immutable? Use with care!
     */
    public IndexableSpliterator(@NotNull Indexable<TElem> indexable,
                                int start,
                                int fence,
                                boolean immutable)
    {
      this(indexable, start, fence,
           immutable
                   ? IMMUTABLE | ORDERED | SIZED | SUBSIZED
                   : ORDERED | SIZED | SUBSIZED);
    }

    /**
     * Instantiate a spliterator for an indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param character characteristics of this spliterator
     */
    private IndexableSpliterator(@NotNull Indexable<TElem> indexable,
                                 int start,
                                 int fence,
                                 int character)
    {
      this.indexable = indexable;
      index = start;
      this.fence = fence;
      this.character = character;
    }

    @Override
    public boolean tryAdvance(Consumer<? super TElem> action)
    {
      if (index < fence) {
        action.accept(indexable.get(index++));
        return true;
      }
      return false;
    }

    @Override
    public IndexableSpliterator<TElem> trySplit()
    {
      final int here = index;
      final int mid = (here + fence) / 2;
      if (here < mid) {
        index = mid;
        return new IndexableSpliterator<>(indexable,
                                          here, mid,
                                          character);
      }
      return null;
    }

    @Override
    public long estimateSize()
    {
      return fence - index;
    }

    @Override
    public int characteristics()
    {
      return character;
    }
  }
}
