// ============================================================================
// File:               IdentityHashSet
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2025  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            5/13/25 11:58 AM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Stream;

/**
 * Hash set based on identities.
 * Although useful, this is missing in the Java standard library.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since May 13, 2025
 */
public class IdentityHashSet<T>
        extends AbstractSet<T>
{
  @NotNull
  private final Map<T, Boolean> map;

  /**
   * Default constructor.
   */
  public IdentityHashSet()
  {
    map = new IdentityHashMap<>();
  }

  /**
   * Constructor.
   * @param initialCapacity initial capacity of this set
   */
  public IdentityHashSet(int initialCapacity)
  {
    map = new IdentityHashMap<>(initialCapacity);
  }

  /**
   * Copy constructor.
   * This copies the values of the {@code iterable} to this set.
   * @param iterable iterable providing the initial values
   */
  public IdentityHashSet(@NotNull Iterable<? extends T> iterable)
  {
    this();
    iterable.forEach(this::add);
  }

  /**
   * Copy constructor.
   * This copies the values of the {@code collection} to this set.
   * @param collection collection providing the initial values
   */
  public IdentityHashSet(@NotNull Collection<? extends T> collection)
  {
    this(collection.size());
    this.addAll(collection);
  }

  /**
   * Copy constructor.
   * This copies the values of the {@code countable} to this set.
   * @param countable countable providing the initial values
   */
  public IdentityHashSet(@NotNull Countable<? extends T> countable)
  {
    this(countable.size());
    countable.addAllTo(this);
  }

  @NotNull
  @Override
  public Iterator<T> iterator()
  {
    return map.keySet().iterator();
  }

  @Override
  public int size()
  {
    return map.size();
  }

  @NotNull
  @Override
  public Spliterator<T> spliterator()
  {
    return map.keySet().spliterator();
  }

  @Override
  public boolean isEmpty()
  {
    return map.isEmpty();
  }

  @Override
  public boolean removeAll(@NotNull Collection<?> c)
  {
    return map.keySet().removeAll(c);
  }

  @Override
  public boolean contains(Object o)
  {
    return map.containsKey(o);
  }

  @NotNull
  @Override
  public Object[] toArray()
  {
    return map.keySet().toArray();
  }

  @NotNull
  @Override
  public <T1> T1[] toArray(@NotNull T1[] a)
  {
    return map.keySet().toArray(a);
  }

  @Override
  public boolean add(T t)
  {
    if (map.containsKey(t)) {
      return false;
    }
    map.put(t, Boolean.TRUE) ;
    return true;
  }

  @Override
  public boolean remove(Object o)
  {
    return map.remove(o) != null;
  }

  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    return map.keySet().containsAll(c);
  }

  @Override
  public boolean addAll(@NotNull Collection<? extends T> c)
  {
    boolean changed = false;
    for (T t : c) {
      changed = changed | add(t); // bit or
    };
    return changed;
  }

  @Override
  public void clear()
  {
    map.clear();
  }

  @Override
  public boolean removeIf(@NotNull Predicate<? super T> filter)
  {
    return map.keySet().removeIf(filter);
  }

  @NotNull
  @Override
  public Stream<T> stream()
  {
    return map.keySet().stream();
  }

  @NotNull
  @Override
  public Stream<T> parallelStream()
  {
    return map.keySet().parallelStream();
  }

  @Override
  public void forEach(Consumer<? super T> action)
  {
    map.keySet().forEach(action);
  }
}
