// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Function1;

import java.util.*;

/**
 * A hashset implementation which allows to provide
 * a dedicated hashcode implementation for its elements.
 * <p>
 * Especially for mutable objects you should take care
 * of providing a useful copier with either the
 * {@code HashCoder} argument in
 * {@link HashCoderSet#HashCoderSet(HashCoder)}
 * or directly with {@link HashCoderSet#HashCoderSet(Function1)}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Mai 18, 2020
 * @param <V> element type
 */
public class HashCoderSet<V>
        implements Set<V>
{
  @NotNull
  private final HashCoder<V> hashCoder;
  @NotNull
  private final Set<HashKey> hashSet;

  /**
   * Hash key substitute for original key class.
   */
  private class HashKey
  {
    private final V originalKey;
    private final int hashCode;

    private HashKey(V originalKey)
    {
      this.originalKey = hashCoder.copy(originalKey);
      this.hashCode = hashCoder.getHashCode(originalKey);
    }

    @Override
    @SuppressWarnings("unchecked") // safe casting because used only internally
    public boolean equals(Object o)
    {
      if (this == o) {
        return true;
      }
      if (!(o instanceof HashCoderSet.HashKey)) {
        return false;
      }
      final HashKey hashKey = (HashKey)o;
      return hashCoder.areEqual(originalKey,
                                hashKey.originalKey);
    }

    @Override
    public int hashCode()
    {
      return hashCode;
    }

    /**
     * Get the original key.
     * @return the original key
     */
    public V getOriginalKey()
    {
      return hashCoder.copy(originalKey);
    }
  }

  /**
   * Constructor.
   * @param hashCoder hash coder used for hashcode calculation,
   *                  equality checks and copying
   */
  public HashCoderSet(@NotNull HashCoder<V> hashCoder)
  {
    this.hashCoder = hashCoder;
    hashSet = new HashSet<>();
  }

  /**
   * Constructor.
   * @param hashCoder hash coder used for hashcode calculation,
   *                  equality checks and copying
   * @param initialCapacity initial map capacity
   */
  public HashCoderSet(@NotNull HashCoder<V> hashCoder,
                      int initialCapacity)
  {
    this.hashCoder = hashCoder;
    hashSet = new HashSet<>(initialCapacity);
  }

  /**
   * Constructor.
   * @param hashCoder hash coder used for hashcode calculation,
   *                  equality checks and copying
   * @param initialCapacity initial map capacity
   * @param loadFactor      map load factor
   */
  public HashCoderSet(@NotNull HashCoder<V> hashCoder,
                      int initialCapacity,
                      float loadFactor)
  {
    this.hashCoder = hashCoder;
    hashSet = new HashSet<>(initialCapacity, loadFactor);
  }

  /**
   * Copy constructor.
   * @param hashCoder hash coder used for
   * @param sourceCollection source collection which's elements are copied to this set
   */
  public HashCoderSet(@NotNull HashCoder<V> hashCoder,
                      @NotNull Collection<? extends V> sourceCollection)
  {
    this.hashCoder = hashCoder;
    hashSet = new HashSet<>(sourceCollection.size());
    addAll(sourceCollection);
  }

  public HashCoderSet(@NotNull HashCoder<V> hashCoder,
                      @NotNull Iterable<? extends V> source)
  {
    this.hashCoder = hashCoder;
    hashSet = new HashSet<>();
  }

  /**
   * Constructor.
   * This uses {@link java.util.Objects#equals(Object, Object)} for
   * equality checks.
   * @param calculator hash code calculator
   */
  public HashCoderSet(@NotNull HashCodeCalculator<? super V> calculator)
  {
    this(new HashCoder<>(calculator));
  }

  /**
   * Constructor.
   * @param calculator hash code calculator
   * @param matcher    equality checker fpr hash keys
   */
  public HashCoderSet(@NotNull HashCodeCalculator<? super V> calculator,
                      @NotNull Matcher<? super V, ? super V> matcher)
  {
    this(new HashCoder<>(calculator,
                         matcher));
  }

  /**
   * Constructor.
   * @param calculator hash code calculator
   * @param matcher    equality checker fpr hash keys
   * @param copier     copier for keys, useful to decouple the keys from their basically mutable source objects
   */
  public HashCoderSet(@NotNull HashCodeCalculator<? super V> calculator,
                      @NotNull Matcher<? super V, ? super V> matcher,
                      @NotNull Function1<V, V> copier)
  {
    this(new HashCoder<>(calculator,
                         matcher,
                         copier));
  }

  /**
   * Constructor for creating a hashmap with keys created from mutable objects.
   * As this decouples the internally used keys from incoming and outgoing
   * keys the basic hashmap requirements are always guaranteed.
   * <p>
   *   Otherwise this uses the {@link Objects#hashCode()}
   * </p>
   * @param copier copier for keys
   */
  public HashCoderSet(@NotNull Function1<V, V> copier)
  {
    this(new HashCoder<>(copier));
  }

  /**
   * Get the hash coder used in this map.
   * @return hash coder used for hashcode calculation, equality checks and copying
   */
  @NotNull
  public HashCoder<? super V> getHashCoder()
  {
    return hashCoder;
  }

  @Override
  public int size()
  {
    return hashSet.size();
  }

  @Override
  public boolean isEmpty()
  {
    return hashSet.isEmpty();
  }

  @Override
  @SuppressWarnings("unchecked")
  public boolean contains(Object o)
  {
    return hashSet.contains(new HashKey((V)o));
  }

  @NotNull
  @Override
  public Iterator<V> iterator()
  {
    return new IteratorConverter<>(hashSet.iterator(),
                                   HashKey::getOriginalKey);
  }

  @NotNull
  @Override
  public Object[] toArray()
  {
    final int sz = size();
    if (sz == 0) {
      return Empty.OBJECT_ARRAY;
    }
    final Object[] result = new Object[sz];
    int i = 0;
    for (HashKey k : hashSet) {
      result[i++] = (k.originalKey);
    }
    return result;
  }

  @NotNull
  @Override
  public <T> T[] toArray(@NotNull T[] a)
  {
    final List<V> tmp = Types.map(new ArrayList<>(size()),
                                  hashSet,
                                  HashKey::getOriginalKey);
    return tmp.toArray(a);
  }

  @Override
  public boolean add(V v)
  {
    return hashSet.add(new HashKey(v));
  }

  @Override
  @SuppressWarnings("unchecked")
  public boolean remove(Object o)
  {
    return hashSet.remove(new HashKey((V)o));
  }

  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    for (Object o : c) {
      if (!contains(c)) {
        return false;
      }
    }
    return true;
  }

  @Override
  public boolean addAll(Collection<? extends V> c)
  {
    boolean changed = false;
    for (V obj : c) {
      if (hashSet.add(new HashKey(obj))) {
        changed = true;
      }
    }
    return changed;
  }

  /**
   * Add all elements of the given iterable.
   * @param it iterable
   * @return {@code true} when this collection was changed by the addition<br>
   *         {@code false} when it was not changed
   */
  public boolean addAll(@NotNull Iterable<? extends V> it)
  {
    boolean changed = false;
    for (V obj : it) {
      if (hashSet.add(new HashKey(obj))) {
        changed = true;
      }
    }
    return changed;
  }

  @Override
  @SuppressWarnings("unchecked")
  public boolean retainAll(@NotNull Collection<?> c)
  {
    return hashSet.retainAll(Types.map(c,
                                       o -> new HashKey((V)o)));
  }

  @Override
  public boolean removeAll(@NotNull Collection<?> c)
  {
    boolean changed = false;
    for (Object v : c) {
      if (remove(v)) {
        changed = true;
      }
    }
    return changed;
  }

  @Override
  public void clear()
  {
    hashSet.clear();
  }
}
