// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Function1;

import java.util.*;
import java.util.function.Function;

/**
 * A hashmap implementation that allows to provide
 * a dedicated hashcode implementation for its elements.
 * <p>
 * Especially for mutable objects you should take care
 * of providing a useful copier with either the
 * {@code HashCoder} argument in
 * {@link HashCoderMap#HashCoderMap(HashCoder)}
 * or directly with {@link HashCoderMap#HashCoderMap(Function)}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Mai 18, 2020
 * @param <K> type of hash key
 * @param <V> value type
 */
public class HashCoderMap<K, V>
        implements Map<K, V>
{
  @NotNull
  private final HashCoder<K> hashCoder;
  @NotNull
  private final HashMap<HashKey, V> hashMap;

  /**
   * Hash key substitute for original key class.
   */
  private class HashKey
  {
    private final K originalKey;
    private final int hashCode;

    private HashKey(K originalKey)
    {
      this.originalKey = hashCoder.copy(originalKey);
      this.hashCode    = hashCoder.getHashCode(originalKey);
    }

    @Override
    @SuppressWarnings("unchecked") // safe casting because use only internally
    public boolean equals(Object o)
    {
      if (this == o) {
        return true;
      }
      if (!(o instanceof HashCoderMap.HashKey)) {
        return false;
      }
      final HashKey hashKey = (HashKey)o;
      return hashCoder.areEqual(originalKey,
                                hashKey.originalKey);
    }

    @Override
    public int hashCode()
    {
      return hashCode;
    }

    /**
     * Get the original key.
     * @return the original key
     */
    public K getOriginalKey()
    {
      return hashCoder.copy(originalKey);
    }
  }

  /**
   * Constructor.
   * @param hashCoder hash coder used for hashcode calculation,
   *                  equality checks and copying
   */
  public HashCoderMap(@NotNull HashCoder<K> hashCoder)
  {
    this.hashCoder = hashCoder;
    hashMap = new HashMap<>();
  }

  /**
   * Constructor.
   * @param hashCoder hash coder used for hashcode calculation,
   *                  equality checks and copying
   * @param initialCapacity initial map capacity
   */
  public HashCoderMap(@NotNull HashCoder<K> hashCoder,
                      int initialCapacity)
  {
    this.hashCoder = hashCoder;
    hashMap = new HashMap<>(initialCapacity);
  }

  /**
   * Constructor.
   * @param hashCoder hash coder used for hashcode calculation,
   *                  equality checks and copying
   * @param initialCapacity initial map capacity
   * @param loadFactor      map load factor
   */
  public HashCoderMap(@NotNull HashCoder<K> hashCoder,
                      int initialCapacity,
                      float loadFactor)
  {
    this.hashCoder = hashCoder;
    hashMap = new HashMap<>(initialCapacity, loadFactor);
  }

  /**
   * Copy constructor.
   * @param hashCoder hash coder used for
   * @param sourceMap source map which is copied to this map
   */
  public HashCoderMap(@NotNull HashCoder<K> hashCoder,
                      @NotNull Map<? extends K, ? extends V> sourceMap)
  {
    this.hashCoder = hashCoder;
    hashMap = new HashMap<>(sourceMap.size());
    putAll(sourceMap);
  }

  /**
   * Constructor.
   * This uses {@link java.util.Objects#equals(Object, Object)} for
   * equality checks.
   * @param calculator hash code calculator
   */
  public HashCoderMap(@NotNull HashCodeCalculator<? super K> calculator)
  {
    this(new HashCoder<>(calculator));
  }

  /**
   * Constructor.
   * @param calculator hash code calculator
   * @param matcher    equality checker fpr hash keys
   */
  public HashCoderMap(@NotNull HashCodeCalculator<? super K> calculator,
                      @NotNull Matcher<? super K, ? super K> matcher)
  {
    this(new HashCoder<>(calculator,
                         matcher));
  }

  /**
   * Constructor.
   * @param calculator hash code calculator
   * @param matcher    equality checker fpr hash keys
   * @param copier     copier for keys, useful to decouple the keys from their basically mutable source objects
   */
  public HashCoderMap(@NotNull HashCodeCalculator<? super K> calculator,
                      @NotNull Matcher<? super K, ? super K> matcher,
                      @NotNull Function1<K, K> copier)
  {
    this(new HashCoder<>(calculator,
                         matcher,
                         copier));
  }

  /**
   * Constructor for creating a hashmap with keys created from mutable objects.
   * As this decouples the internally used keys from incoming and outgoing
   * keys the basic hashmap requirements are always guaranteed.
   * <p>
   *   Otherwise this uses the {@link Objects#hashCode()}
   * </p>
   * @param copier copier for keys
   */
  public HashCoderMap(@NotNull Function<K, K> copier)
  {
    this(new HashCoder<>(copier));
  }

  /**
   * Get the hash coder used in this map.
   * @return hash coder used for hashcode calculation, equality checks and copying
   */
  @NotNull
  public HashCoder<? super K> getHashCoder()
  {
    return hashCoder;
  }

  @Override
  public int size()
  {
    return hashMap.size();
  }

  @Override
  public boolean isEmpty()
  {
    return hashMap.isEmpty();
  }

  @Override
  @SuppressWarnings("unchecked")
  public boolean containsKey(Object key)
  {
    return hashMap.containsKey(new HashKey((K)key));
  }

  @Override
  public boolean containsValue(Object value)
  {
    return hashMap.containsValue(value);
  }

  @Override
  @SuppressWarnings("unchecked")
  public V get(Object key)
  {
    return hashMap.get(new HashKey((K)key));
  }

  @Override
  public V put(K key, V value)
  {
    return hashMap.put(new HashKey(key),
                       value);
  }

  @Override
  @SuppressWarnings("unchecked")
  public V remove(Object key)
  {
    return hashMap.remove(new HashKey((K)key));
  }

  @Override
  public void putAll(Map<? extends K, ? extends V> m)
  {
    for (Entry<? extends K, ? extends V> entry : m.entrySet()) {
      hashMap.put(new HashKey(entry.getKey()), entry.getValue());
    }
  }

  @Override
  public void clear()
  {
    hashMap.clear();
  }

  @NotNull
  @Override
  public Set<K> keySet()
  {
    return new HashCoderSet<K>(hashCoder,
                               Types.view(hashMap.keySet(),
                                          HashCoderMap.HashKey::getOriginalKey));
  }

  @NotNull
  @Override
  public Collection<V> values()
  {
    return hashMap.values();
  }

  @NotNull
  @Override
  public Set<Entry<K, V>> entrySet()
  {
    return Types.map(new HashSet<>(),
                     hashMap.entrySet(),
                     e -> new Entry<K, V>() {
                       @Override
                       public K getKey()
                       {
                         return e.getKey().getOriginalKey();
                       }

                       @Override
                       public V getValue()
                       {
                         return e.getValue();
                       }

                       @Override
                       public V setValue(V value)
                       {
                         return e.setValue(value);
                       }
                     });
  }
}
