// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Objects;
import java.util.function.Function;

/**
 * Helper interface for {@link HashCoderMap}.
 * <p>
 * This class provides substitute hash values and equals
 * implementation. To be {@code null}-safe all three
 * arguments of {@link HashCoder#HashCoder(HashCodeCalculator, Matcher, Function)}
 * need to be {@code null}-safe. The default implementations used
 * in the constructors with fewer arguments are {@code null}-safe.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Mai 18, 2020
 * @param <T> type of hashed object
 */
public class HashCoder<T>
  implements HashCodeCalculator<T>,
             UniformMatcher<T>
{
  private static final HashCoder<Object> NATURAL = new HashCoder<>(HashCodeCalculator.NATURAL,
                                                                   UniformMatcher.NATURAL,
                                                                   Function.identity());
  private static final HashCoder<Object> NATURAL_NULL_SAFE = new HashCoder<>(HashCodeCalculator.NATURAL_NULL_SAFE,
                                                                             UniformMatcher.NATURAL_NULL_SAFE,
                                                                             Function.identity());

  @NotNull
  private final HashCodeCalculator<? super T> codeCalculator;
  @NotNull
  private final Matcher<? super T, ? super T> matcher;
  @NotNull
  private final Function<T, T> copier;

  /**
   * Constructor.
   * @param codeCalculator calculator for hash codes
   * @param matcher        equality checker
   * @param copier         copier for incoming and outgoing keys,
   *                       useful to make keys immutable which
   *                       is a basic requirements for hash keys
   */
  public HashCoder(@NotNull HashCodeCalculator<? super T> codeCalculator,
                   @NotNull Matcher<? super T, ? super T> matcher,
                   @NotNull Function<T, T> copier)
  {
    this.codeCalculator = codeCalculator;
    this.matcher = matcher;
    this.copier = copier;
  }

  /**
   * Constructor.
   * <p>
   * This used simple copying for the keys.
   * @param codeCalculator calculator for hash codes
   * @param matcher        equality checker
   */
  public HashCoder(@NotNull HashCodeCalculator<? super T> codeCalculator,
                   @NotNull Matcher<? super T, ? super T> matcher)
  {
    this(codeCalculator, matcher, obj -> obj);
  }

  /**
   * Constructor.
   * This uses {@link Objects#equals} and simple copying for the keys.
   * @param codeCalculator hash code calculator
   */
  public HashCoder(@NotNull HashCodeCalculator<? super T> codeCalculator)
  {
    this(codeCalculator, Objects::equals, obj -> obj);
  }

  /**
   * Constructor.
   * This is useful to decouple hash keys, allowing to use
   * mutable objects for keys in a hash map or set. Obviuosly
   * then the copier has to really copy the object.
   * <p>
   *   Hashcode and equality checks are just using the standard
   *   implementation from the key objects.
   * </p>
   * @param copier         copier for incoming and outgoing keys
   */
  public HashCoder(@NotNull Function<T, T> copier)
  {
    this(Objects::hashCode, Objects::equals, copier);
  }

  @Override
  public int getHashCode(T obj)
  {
    return codeCalculator.getHashCode(obj);
  }

  @Override
  public boolean areEqual(T object1, T object2)
  {
    return matcher.areEqual(object1, object2);
  }

  /**
   * Copy a key.
   * @param key  key to copy
   * @return copied key, possible the incoming one
   */
  public T copy(T key)
  {
    return copier.apply(key);
  }

  /**
   * Get a fast natural hash coder using just the object's methods
   * for hashing and equality, but not checking for {@code null} values.
   * @return natural hash coder
   * @param <V> value type
   */
  @NotNull
  @SuppressWarnings("unchecked") // because of read-only usage
  public static <V> HashCoder<V> natural()
  {
    return (HashCoder<V>)NATURAL;
  }

  /**
   * Get a natural hash coder using just the object's methods
   * for hashing and equality, allowing for {@code null} values.
   * @return natural hash coder
   * @param <V> value type
   */
  @NotNull
  @SuppressWarnings("unchecked") // because of read-only usage
  public static <V> HashCoder<V> naturalNullSafe()
  {
    return (HashCoder<V>)NATURAL_NULL_SAFE;
  }
}
