// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2013-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Objects;

/**
 * Calculator interface for generalized hash code calculation.
 * <p>
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see Matcher
 */
public interface HashCodeCalculator<T>
{
  /**
   * Default hash code calculator which just uses
   * {@link java.util.Objects#hashCode(java.lang.Object)}.
   */
  HashCodeCalculator<Object> OBJECTS = Objects::hashCode;

  /**
   * Default hash code calculator which just uses
   * the object's own hash code implementation.
   * Note that this requires non-{@code null} objects.
   */
  HashCodeCalculator<Object> NATURAL = Object::hashCode;

  /**
   * Default hash code calculator which just uses
   * the object's own hash code implementation, but takes care of
   * {@code null} values.
   */
  HashCodeCalculator<Object> NATURAL_NULL_SAFE = obj -> obj == null ? 0 : obj.hashCode();

  /**
   * Get the hash code value of the given object.
   * @param obj object for which the hash code is requested
   * @return hash code
   */
  int getHashCode(T obj);

  /**
   * Get a hash code calculator which uses
   * {@link java.util.Objects#hashCode(java.lang.Object)}.
   * @return  natural hashCode implementation
   * @param <K> object type
   */
  @NotNull
  @SuppressWarnings("unchecked") // because or read-only nature of the result
  static <K> HashCodeCalculator<K> objects()
  {
    return (HashCodeCalculator<K>)OBJECTS;
  }

  /**
   * Get a hash code calculator which uses
   * the object's own hashcode implementation.
   * Note that this requires non-{@code null} objects.
   * Use {@link #naturalNullSafe()} to avoid this.
   * @return  natural hashCode implementation
   * @param <K> object type
   */
  @NotNull
  @SuppressWarnings("unchecked") // because or read-only nature of the result
  static <K> HashCodeCalculator<K> natural()
  {
    return (HashCodeCalculator<K>)NATURAL;
  }

  /**
   * Get a hash code calculator which uses
   * the object's own hashcode implementation,
   * but takes care of {@code null} values.
   * @return  natural hashCode implementation
   * @param <K> object type
   */
  @NotNull
  @SuppressWarnings("unchecked") // because or read-only nature of the result
  static <K> HashCodeCalculator<K> naturalNullSafe()
  {
    return (HashCodeCalculator<K>)NATURAL_NULL_SAFE;
  }
}
