// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.matcher.Match;

/**
 * Helper class which is caching a hash value.
 * <p>
 * Usage of this class is only making sense if the hash values of the
 * underlying objects don't change, but that is true when depending
 * on hashes anyway. Caching the hash is best if its calculation is
 * expensive or the costs are unknown.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class HashCacheWrapper<T>
{
  /** The wrapped object. */
  private final T wrapped;
  /** The cached hash value. */
  private final int hashCodeValue;
  /** The matcher to use to check for equality of the underlying object. */
  @NotNull
  private final UniformMatcher<? super T> matcher;

  /**
   * Constructor.
   * @param wrapped        wrapped object
   * @param hashCodeValue  hash value to use
   * @param matcher        matcher to use for equality checks, if {@code null}
   *                       {@link de.caff.generics.matcher.Match#DEFAULT_MATCHER} will be used
   */
  public HashCacheWrapper(T wrapped, int hashCodeValue, @Nullable UniformMatcher<? super T> matcher)
  {
    this.wrapped = wrapped;
    this.hashCodeValue = hashCodeValue;
    if (matcher == null) {
      this.matcher = Match.DEFAULT_MATCHER;
    }
    else {
      this.matcher = matcher;
    }
  }

  /**
   * Constructor.
   * This will use the default hash value of the object and its equals method for
   * equality checks.
   * @param wrapped wrapped object
   */
  public HashCacheWrapper(@NotNull T wrapped)
  {
    this(wrapped, wrapped.hashCode(), null);
  }

  /**
   * Constructor.
   * This will use the default hash value of the object and its equals method for
   * equality checks.
   * @param wrapped wrapped object
   * @param matcher matcher used for equality checks
   */
  public HashCacheWrapper(@NotNull T wrapped, @Nullable UniformMatcher<? super T> matcher)
  {
    this(wrapped, wrapped.hashCode(), matcher);
  }

  /**
   * Constructor.
   * This will use a dedicated hash code calculator and a dedicated matcher for
   * equality checks.
   * @param wrapped            wrapped object
   * @param hashCodeCalculator hash code calculator
   * @param matcher            matcher for equality checks, if {@code null}
   *                           {@link de.caff.generics.matcher.Match#DEFAULT_MATCHER}
   *                           will be used
   */
  public HashCacheWrapper(T wrapped,
                          @NotNull HashCodeCalculator<? super T> hashCodeCalculator,
                          @Nullable UniformMatcher<? super T> matcher)
  {
    this(wrapped, hashCodeCalculator.getHashCode(wrapped), matcher);
  }

  /**
   * Get the wrapped object.
   * @return wrapped object
   */
  public T getWrapped()
  {
    return wrapped;
  }

  @Override
  @SuppressWarnings("unchecked")
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    final HashCacheWrapper<T> that = (HashCacheWrapper<T>)o;

    return hashCodeValue == that.hashCodeValue &&
           matcher.areEqual(this.wrapped, that.wrapped);
  }

  @Override
  public int hashCode()
  {
    return hashCodeValue;
  }
}
