// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FloatFunction2;
import de.caff.generics.function.FloatOperator2;
import de.caff.generics.function.FloatPredicate2;
import de.caff.generics.tuple.Tuple;
import de.caff.generics.tuple.Tuple2;

import java.io.Serializable;
import java.util.Locale;
import java.util.Objects;

import static de.caff.generics.Primitives.areEqual;

/**
 * A pair of primitive float values.
 * <p>
 * This is similar to {@link Pair}, but for raw {@code float} values.
 * As these values are defined final, this class is also immutable.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 04, 2020
 */
public final class FloatPair
        implements Serializable,
                   FloatIndexable
{
  private static final long serialVersionUID = -832244696920409438L;
  /** The first value of this pair. */
  public final float first;
  /** The second value of this pair. */
  public final float second;

  /**
   * Constructor.
   * @param first    first value of pair
   * @param second   second value of pair
   */
  public FloatPair(float first, float second)
  {
    this.first = first;
    this.second = second;
  }

  /**
   * Convert this pair of {@code float}s into a pair of {@code double}s.
   * @return double pair with same values
   */
  @NotNull
  public DoublePair toDoublePair()
  {
    return new DoublePair(first, second);
  }

  /**
   * Get this pair with a different first value.
   * @param first new first value
   * @return pair with the given {@code first} value,
   *         and the {@link #second} value of this pair
   */
  @NotNull
  public FloatPair withFirst(float first)
  {
    if (Float.compare(this.first, first) == 0) {
      return this;
    }
    return new FloatPair(first, this.second);
  }

  /**
   * Get this pair with a different first value.
   * @param second new second value
   * @return pair with the {@link #first} value of this pair,
   *         and the given {@code second} value
   */
  @NotNull
  public FloatPair withSecond(float second)
  {
    if (Float.compare(this.second, second) == 0) {
      return this;
    }
    return new FloatPair(this.first, second);
  }

  /**
   * Get this pair with swapped values.
   * @return pair with swapped {@link #first} and {@link #second} values
   */
  @NotNull
  public FloatPair swapped()
  {
    if (Float.compare(first, second) == 0) {
      return this;
    }
    return new FloatPair(second, first);
  }

  @Override
  public int size()
  {
    return 2;
  }

  @Override
  public float get(int index)
  {
    switch (index) {
    case 0:
      return first;
    case 1:
      return second;
    default:
      throw new IndexOutOfBoundsException(String.format("Pair has only 2 elements, so %d is out of bounds!", index));
    }
  }

  /**
   * Apply a 2-float function on this pair.
   * @param function function to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @param <T> output type
   * @return result of applying the function
   */
  public <T> T applyAsFloats(@NotNull FloatFunction2<T> function)
  {
    return function.applyAsFloat(first, second);
  }

  /**
   * Call an 2-float operator on this pair.
   * @param operator operator to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @return result of applying the operator
   */
  public float operate(@NotNull FloatOperator2 operator)
  {
    return operator.applyAsFloat(first, second);
  }

  /**
   * Call a 2-float predicate on this pair. 
   * @param predicate predicate to apply, {@link #first} becomes its first, {@link #second} its second argument 
   * @return result of testing this pair
   */
  public boolean test(@NotNull FloatPredicate2 predicate)
  {
    return predicate.testFloats(first, second);
  }

  /**
   * Convert this into a tuple.
   * @return tuple created from this pair
   */
  @NotNull
  public Tuple2<Float, Float> toTuple()
  {
    return Tuple.of(first, second);
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final FloatPair that = (FloatPair)o;
    return areEqual(that.first, first)  &&
           areEqual(that.second, second);
  }

  @Override
  public int hashCode()
  {
    return Objects.hash(Primitives.boxed(first), Primitives.boxed(second));
  }

  @Override
  @NotNull
  public String toString()
  {
    return String.format(Locale.US, "FloatPair(%s, %s)", first, second);
  }
}
