// ============================================================================
// File:               FilteringIteratorWrapper
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2021-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            30.09.21 13:56
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.function.Predicate;

/**
 * A wrapper for a standard iterator which only iterates over
 * selected elements.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 30, 2021
 */
public class FilteringIteratorWrapper<T>
        implements Iterator<T>
{
  /** The basic iterator. */
  @NotNull
  private final Iterator<T> iterator;
  /** The filter. */
  @NotNull
  private final Predicate<? super T> filter;

  private T nextElement;
  private boolean hasNext = false;

  /**
   * Create a filtering iterator.
   * @param iterator basic iterator
   * @param filter   filter which selects the elements which are returned by this iterator
   */
  public FilteringIteratorWrapper(@NotNull Iterator<T> iterator, @NotNull Predicate<? super T> filter)
  {
    this.iterator = iterator;
    this.filter = filter;
    forward();
  }

  /**
   * Go forward to the next valid element.
   * Sets {@link #nextElement} and {@link #hasNext}.
   */
  private void forward()
  {
    while (iterator.hasNext()) {
      nextElement = iterator.next();
      if (filter.test(nextElement)) {
        hasNext = true;
        return;
      }
    }
    hasNext = false;
    nextElement = null;
  }

  @Override
  public boolean hasNext()
  {
    return hasNext;
  }

  @Override
  public T next()
  {
    if (!hasNext) {
      throw new NoSuchElementException();
    }
    try {
      return nextElement;
    } finally {
      forward();
    }
  }
}
