// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.EmptyStackException;

/**
 * Implementation of a functional stack.
 * See {@link FStack} for more information.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 22, 2020
 * @param <T> element type
 */
class FStackImpl<T>
        implements FStack<T>
{
  /**
   * Empty stack.
   * Don't use this directly, use {@link FStack#empty()} instead.
   */
  static FStack<Object> EMPTY = new FStack<Object>()
  {
    @Override
    public Object top()
    {
      throw new EmptyStackException();
    }

    @NotNull
    @Override
    public FStack<Object> push(Object elem)
    {
      return new FStackImpl<>(elem, this);
    }

    @NotNull
    @Override
    public FStack<Object> pop()
    {
      throw new EmptyStackException();
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @Override
    public int size()
    {
      return 0;
    }
  };

  private final T element;
  @NotNull
  private final FStack<T> tail;

  /**
   * Constructor.
   * @param element element at the top of this stack
   * @param tail tails elements of this stack
   */
  FStackImpl(T element, @NotNull FStack<T> tail)
  {
    this.element = element;
    this.tail = tail;
  }

  @Override
  public T top()
  {
    return element;
  }

  @NotNull
  @Override
  public FStack<T> push(T elem)
  {
    return new FStackImpl<>(elem, this);
  }

  @NotNull
  @Override
  public FStack<T> pop()
  {
    return tail;
  }

  @Override
  public boolean isEmpty()
  {
    return false;
  }

  @Override
  public int size()
  {
    return 1 + tail.size();
  }
}
