// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;

/**
 * A functional stack.
 * <p>
 * This implements a functional stack which itself is immutable.
 * Stack operations are defined in a way that most of the stack is
 * reused in copies of it. That means that when you change ony
 * element on the stack it will change in all copies of this
 * stack and the stacks created from it via {@link #pop()}
 * and {@link #push(Object)}. If you don't change the elements
 * (preferable they are immutable, too) this stack is perfectly
 * thread-safe.
 * <p>
 * The only way to use this stack in multi-threaded
 * environments for potentially changeable objects
 * is by making copies of each element when it is pushed and when
 * it is requested via {@link #top()}. This stack can be used this
 * way by providing a copier or using {@link Copyable copiable} elements.
 * See item list below for how this is done.
 * <p>
 * This stack also extends {@link Iterable}. The returned {@link Iterator}
 * will traverse the stack from top to bottom.
 * <p>
 * Creating a stack from scratch is done by
 * <ul>
 *   <li>
 *     {@link #empty()} which will return an empty stack with the required
 *     type. Internally this is always the same object, but the Java type
 *     system takes care of handling this correctly.
 *   </li>
 *   <li>
 *     {@link #empty(Copier)} which will return an empty copying stack
 *     with the required type. This is the only way to use a stack of
 *     mutable elements securely in multi-threaded environments.
 *   </li>
 *   <li>
 *     {@link #emptyCopyable()} which will return an empty copying stack with
 *     the required {@link Copyable copyable} type. This is the way to use
 *     copyable elements securely in multi-threaded environments.
 *   </li>
 *   <li>
 *     {@link #from(Iterable)} which will return a stack with the elements
 *     from the given iterable. Because of the lifo characteristics of
 *     stacks popping the elements one by one (or iterating over the stack)
 *     will present the elements in the reverse order compared to the
 *     iterable.
 *   </li>
 *   <li>
 *     {@link #from(Copier, Iterable)} which will return a stack with the
 *     copied elements from the given iterable. Because of the lifo characteristics of
 *     stacks popping the elements one by one (or iterating over the stack)
 *     will present the elements in the reverse order compared to the
 *     iterable. This is a way to use a preinitialized stack of
 *     mutable elements securely in multi-threaded environments.
 *   </li>
 *   <li>
 *     {@link #fromCopyable(Iterable)} which will return a stack with the
 *     copied elements from the given iterable. Because of the lifo characteristics of
 *     stacks popping the elements one by one (or iterating over the stack)
 *     will present the elements in the reverse order compared to the
 *     iterable. This is a way to use a preinitialized stack of
 *     {@link Copyable copyable} mutable elements securely in multi-threaded environments.
 *   </li>
 *   <li>
 *     {@link #reverseFrom(Iterable)} which will return a stack with the
 *     elements from the given iterable, but take care that they are
 *     pushed in reverse order. Therefore popping the elements one by
 *     one (or iterating over the stack) will present the same order
 *     of elements that the iterable provided.
 *   </li>
 *   <li>
 *     {@link #reverseFrom(Copier, Iterable)} which will return a stack with the
 *     copied elements from the given iterable but take care that they are
 *     pushed in reverse order. Therefore popping the elements one by
 *     one (or iterating over the stack) will present the same order
 *     of elements that the iterable provided. This is a way to use a
 *     preinitialized stack of mutable elements securely in
 *     multi-threaded environments.
 *   </li>
 *   <li>
 *     {@link #reverseFromCopyable(Iterable)} which will return a stack with
 *     the copied elements from the given iterable but take care that they are pushed
 *     in reverse order.  Therefore popping the elements one by
 *     one (or iterating over the stack) will present the same order
 *     of elements that the iterable provided. This is a way to use a preinitialized stack of
 *     {@link Copyable copyable} mutable elements securely in multi-threaded environments.
 *   </li>
 * </ul>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 22, 2020
 * @param <T> element type
 */
public interface FStack<T>
        extends Iterable<T>
{
  /**
   * Get the top element of the stack.
   * @return top element
   * @throws java.util.EmptyStackException if this stack is {@link #isEmpty() empty}
   */
  T top();

  /**
   * Push an element to the top of the stack.
   * This will not change this stack but return
   * a new stack with the given element at
   * the {@link #top()}.
   * @param elem element to push to the stack
   * @return new stack with {@code elem} at the top
   *         and the same elements as this stack below it
   */
  @NotNull
  FStack<T> push(T elem);

  /**
   * Get a stack which is the result of pushing all elements
   * to the top of this stack.
   * This will not change this stack but return
   * a new stack with the given elements at
   * the {@link #top()}.
   * <p>
   * Because of the lifo characteristics of stacks the order
   * of elements when popped from the returned stack is
   * the reverse of the order of the iterable. If you
   * prefer otherwise use {@link #reversePushAll(Iterable)}.
   * @param elements elements to push to this stack
   * @return stack with the elements pushed to its top
   */
  @NotNull
  default FStack<T> pushAll(@NotNull Iterable<? extends T> elements)
  {
    FStack<T> stack = this;
    for (T elem : elements) {
      stack = stack.push(elem);
    }
    return stack;
  }

  /**
   * Get a stack which is the result of pushing all elements
   * to the top of this stack.
   * This will not change this stack but return
   * a new stack with the given elements at
   * the {@link #top()}.
   * <p>
   * This method reverses the order of pushing so if
   * you pop the elements from the returned stack
   * they will have the same order as they had in the
   * iterable's sequence. If you
   * prefer otherwise use {@link #pushAll(Iterable)}.
   * @param elements elements to push to this stack
   * @return stack with the elements pushed to its top
   */
  @NotNull
  default FStack<T> reversePushAll(@NotNull Iterable<? extends T> elements)
  {
    // Note: this intermediate copy is awful, but there's no simple way around this
    final FStack<T> intermediate = empty();
    for (T elem : elements) {
      intermediate.push(elem);
    }
    return pushAll(intermediate);
  }

  /**
   * Pop the stack.
   * This will remove the top-most element from the stack
   * and return a new stack without it.
   * @return a new stack without the top element of this stack
   * @throws java.util.EmptyStackException if this stack is {@link #isEmpty() empty}
   */
  @NotNull
  FStack<T> pop();

  /**
   * Is this stack empty?
   * An empty stack does not have a top element, therefore
   * {@link #top()} and {@link #pop()} are not allowed and
   * will result in {@link java.util.EmptyStackException}s
   * @return {@code}
   */
  boolean isEmpty();

  /**
   * Get the size of this stack.
   * This is an expensive operation!
   * @return number of elements in this stack
   */
  int size();

  @NotNull
  @Override
  default Iterator<T> iterator()
  {
    return new Iterator<T>() {
      private FStack<T> remaining = FStack.this;

      @Override
      public boolean hasNext()
      {
        return !remaining.isEmpty();
      }

      @Override
      public T next()
      {
        try {
          final T elem = remaining.top();
          remaining = remaining.pop();
          return elem;
        } catch (EmptyStackException e) {
          throw new NoSuchElementException();
        }
      }
    };
  }

  /**
   * Get an empty stack.
   * This stack will only behave well if elements are either immutable or if they are never changed.
   * See
   * @param <E> element type of the returned stack
   * @return empty stack
   */
  @NotNull
  @SuppressWarnings("unchecked") // as EMPTY does not have any elements, and the
                                 // Java type system takes care of handling pushed elements correctly
  static <E> FStack<E> empty()
  {
    return (FStack<E>) FStackImpl.EMPTY;
  }

  /**
   * Get an empty stack which uses a copier to allow using changeable elements
   * securely in multi-threaded environments.
   * <p>
   * The returned functional stack is itself immutable, and will copy elements
   * when pushed and requested.
   * @param copier copier applied on incoming and outgoing elements
   * @param <E> element type of returned stack
   * @return functional stack which decouples the elements by copying them when necessary
   * @see #empty()
   * @see #emptyCopyable()
   */
  @NotNull
  static <E> FStack<E> empty(@NotNull Copier<E> copier)
  {
    return new CopierFStackImpl.Empty<>(copier);
  }

  /**
   * Get an empty stack which can be used securely in multi-threaded environemnts
   * because it copies it copyable elements.
   * @param <E> copyable element type
   * @return functional stack which decouples its elements by copying them when necessary
   * @see #empty()
   * @see #empty(Copier)
   */
  @NotNull
  static <E extends Copyable<E>> FStack<E> emptyCopyable()
  {
    return empty(Copyable<E>::getCopy);
  }

  /**
   * Create a stack from pushing all elements of an iterable to this stack.
   * Because of the lifo characteristics of stacks iterating over the returned
   * stack will be done in the reverse order of the given {@code iterable}.
   * If you prefer this the other way round use {@link #reverseFrom(Iterable)}.
   * @param iterable iterable to push onto the stack
   * @param <E> element type of the iterable and the returned stack
   * @return stack initialized from the given iterable
   * @see #from(Copier, Iterable)
   * @see #fromCopyable(Iterable)
   */
  @NotNull
  static <E> FStack<E> from(@NotNull Iterable<E> iterable)
  {
    return FStack.<E>empty().pushAll(iterable);
  }

  /**
   * Create a stack from pushing copies of all elements of an iterable to this stack.
   * Because of the lifo characteristics of stacks iterating over the returned
   * stack will be done in the reverse order of the given {@code iterable}.
   * If you prefer this the other way round use {@link #reverseFrom(Copier, Iterable)}.
   * @param copier copier applied to all incoming and outgoing elements
   * @param iterable iterable to push onto the stack
   * @param <E> element type of the iterable and the returned stack
   * @return stack initialized from the given iterable
   * @see #from(Iterable)
   * @see #from(Copier, Iterable)
   */
  @NotNull
  static <E> FStack<E> from(@NotNull Copier<E> copier,
                            @NotNull Iterable<E> iterable)
  {
    return FStack.<E>empty(copier).pushAll(iterable);
  }

  /**
   * Create a stack from pushing copies of all elements of an iterable to this stack.
   * Because of the lifo characteristics of stacks iterating over the returned
   * stack will be done in the reverse order of the given {@code iterable}.
   * If you prefer this the other way round use {@link #reverseFromCopyable(Iterable)}.
   * @param iterable iterable to push onto the stack
   * @param <E> copyable element type of the iterable and the returned stack
   * @return stack initialized from the given iterable
   * @see #from(Iterable)
   * @see #from(Copier, Iterable)
   */
  @NotNull
  static <E extends Copyable<E>> FStack<E> fromCopyable(@NotNull Iterable<E> iterable)
  {
    return FStack.<E>emptyCopyable().pushAll(iterable);
  }

  /**
   * Create a stack from pushing all elements of an iterable to this stack.
   * This method basically pushes the elements in the reverse order,
   * so that iterating over the stack will present the same sequence
   * of elements the {@code iterable} presents.
   * If you prefer this the other way round use {@link #from(Iterable)}.
   * @param iterable iterable to push onto the stack
   * @param <E> element type of the iterable and the returned stack
   * @return stack initialized from the given iterable
   * @see #reverseFrom(Copier, Iterable)
   * @see #reverseFromCopyable(Iterable)
   */
  @NotNull
  static <E> FStack<E> reverseFrom(@NotNull Iterable<E> iterable)
  {
    return FStack.<E>empty().reversePushAll(iterable);
  }

  /**
   * Create a stack from pushing copies of all elements of an iterable to this stack.
   * This method basically pushes the elements in the reverse order,
   * so that iterating over the stack will present the same sequence
   * of elements the {@code iterable} presents.
   * If you prefer this the other way round use {@link #from(Copier, Iterable)}.
   * @param copier copier applied to all incoming and outgoing stack elements
   * @param iterable iterable to push onto the stack
   * @param <E> element type of the iterable and the returned stack
   * @return stack initialized from the given iterable
   * @see #reverseFrom(Iterable)
   * @see #reverseFromCopyable(Iterable)
   */
  @NotNull
  static <E> FStack<E> reverseFrom(@NotNull Copier<E> copier,
                                   @NotNull Iterable<E> iterable)
  {
    return FStack.<E>empty(copier).reversePushAll(iterable);
  }

  /**
   * Create a stack from pushing copies of all elements of an iterable to this stack.
   * This method basically pushes the elements in the reverse order,
   * so that iterating over the stack will present the same sequence
   * of elements the {@code iterable} presents.
   * If you prefer this the other way round use {@link #fromCopyable(Iterable)}.
   * @param iterable iterable to push onto the stack
   * @param <E> copyable element type of the iterable and the returned stack
   * @return stack initialized from the given iterable
   * @see #reverseFrom(Iterable)
   * @see #reverseFromCopyable(Iterable)
   */
  @NotNull
  static <E extends Copyable<E>> FStack<E> reverseFromCopyable(@NotNull Iterable<E> iterable)
  {
    return FStack.<E>emptyCopyable().reversePushAll(iterable);
  }

}
