// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.IntUnaryOperator;

/**
 * Immutable expandable primitive integer indexable.
 * <p>
 * This interface defines an integer indexable which allows
 * to change (set, add, remove) its values but is immutable
 * nevertheless. All changing methods return copies, thus
 * guaranteeing immutability.
 * <p>
 * The internal implementation tries to keep copying a cheap
 * operation, but some access patterns may result in performance
 * degration. It is therefore recommended not to use this in
 * time-critical code.
 * <p>
 * Initial creation is done by using the following methods:
 * <ul>
 *   <li>{@link #empty()}</li>
 *   <li>{@link #zeroed(int)}</li>
 *   <li>{@link #from(int...)}</li>
 *   <li>{@link #from(int[], int, int)}</li>
 *   <li>{@link #from(IntIndexable)}</li>
 *   <li>{@link #from(Indexable)}</li>
 *   <li>{@link #from(Collection)}</li>
 *   <li>{@link #fromNumbers(Number[])}</li>
 * </ul>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 24, 2020
 */
public interface ExpandableIntIndexable
        extends IntIndexable
{
  /**
   * Return an expandable int indexable Insert where the the given value
   * was inserted at the given index.
   * @param index index in the range {@code 0} (insert at first position)
   *              to {@code size()} (insert after last position)
   * @param value value to insert at the given position
   * @return new integer indexable with the given value inserted
   */
  @NotNull
  ExpandableIntIndexable add(int index, int value);

  /**
   * Return an expandable int indexable where the value at the given index
   * was removed.
   * @param index index to remove, in the range {@code 0} to
   *              {@code size() - 1}
   * @return new expandable integer indexable with the given index removed
   */
  @NotNull
  ExpandableIntIndexable remove(int index);

  /**
   * Return an expandable int indexable where the given
   * value was added to the end of this indexable.
   * @param value value to add
   * @return new integer indexable with added value
   */
  @NotNull
  default ExpandableIntIndexable add(int value)
  {
    return add(size(), value);
  }

  /**
   * Add all values.
   * @param values values to add
   * @return expandable int indexable where all the values are added
   */
  @NotNull
  default ExpandableIntIndexable addAll(@NotNull int... values)
  {
    ExpandableIntIndexable result = this;
    for (int v : values) {
      result = result.add(v);
    }
    return result;
  }

  /**
   * Pythonesque insertion allowing negative indexes to insert from the end.
   * @param index positive values behave the same way as {@link #add(int, int)},
   *              but negative values count from the back, so {@code -1}
   *              will insert before the last element
   * @param value value to insert
   * @return new integer indexable with the given value inserted
   */
  @NotNull
  default ExpandableIntIndexable ydd(int index, int value)
  {
    if (index >= 0) {
      return add(index, value);
    }
    else {
      final int realIndex = index + size();
      if (realIndex < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           index, size()));
      }
      return add(realIndex, value);
    }
  }

  /**
   * Pythonesque remove allowing negative indexes to remove from the end.
   * @param index positive values behave the same as {@link #remove(int)},
   *              while negative indexes count from the back, so {@code -1}
   *              will remove the last element
   * @return new expandable integer indexable with the given index removed
   */
  @NotNull
  default ExpandableIntIndexable remyve(int index)
  {
    if (index >= 0) {
      return remove(index);
    }
    else {
      final int realIndex = index + size();
      if (realIndex < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           index, size()));
      }
      return remove(realIndex);
    }
  }

  /**
   * Convenience method to create an empty expandable int indexable.
   * @return empty expandable int indexable
   */
  @NotNull
  static ExpandableIntIndexable empty()
  {
    return from();
  }

  /**
   * Create an expandable int indexable of the given size initialized with {@code 0}.
   * @param size size of returned integer indexable
   * @return expandable integer indexable of the given size
   */
  @NotNull
  static ExpandableIntIndexable zeroed(int size)
  {
    return from(new IntIndexable.Base()
    {
      @Override
      public int size()
      {
        return size;
      }

      @Override
      public int get(int index)
      {
        return 0;
      }

      @Override
      public int addToArray(@NotNull int[] array, int arrayPos, int index, int length)
      {
        Arrays.fill(array, arrayPos, arrayPos + length, 0);
        return arrayPos + length;
      }
    });
  }

  /**
   * Create an expandable int indexable with a given size, and initialize its elements by index.
   * @param size     size of the indexable
   * @param producer element producer which will be called for each index
   * @return expandable int indexable with the given size containing the elements created by the producer
   */
  @NotNull
  static IntIndexable initByIndex(int size, @NotNull IntUnaryOperator producer)
  {
    return from(IntIndexable.viewByIndex(size, producer));
  }

  /**
   * Create an expandable integer indexable from an initial array of values.
   * @param values array of values
   * @return expandable integer indexable
   */
  @NotNull
  static ExpandableIntIndexable from(@NotNull int ... values)
  {
    return ExpandableIntIndexableImpl.from(values);
  }

  /**
   * Create an expandable integer indexable from part of an int array.
   * @param array    integer array
   * @param start    start position in array
   * @param length   number of byte in array
   * @return expandable integer indexable
   */
  @NotNull
  static ExpandableIntIndexable from(@NotNull int[] array, int start, int length)
  {
    return ExpandableIntIndexableImpl.from(array, start, length);
  }

  /**
   * Get an expandable integer indexable from a normal
   * integer indexable.
   * @param indexable  standard integer indexable
   * @return expandable copy of the given indexable
   */
  @NotNull
  static ExpandableIntIndexable from(@NotNull IntIndexable indexable)
  {
    return ExpandableIntIndexableImpl.from(indexable);
  }

  /**
   * Get an expandable integer indexable from an array of numbers.
   * @param values number values
   * @param <N>    number type, prefarably {@link Integer}
   * @return expandable int indexable
   */
  @NotNull
  @SafeVarargs // because read only
  @SuppressWarnings("varargs")
  static <N extends Number> ExpandableIntIndexable fromNumbers(@NotNull N... values)
  {
    return from(IntIndexable.viewNumberArray(values));
  }

  /**
   * Get an expandable integer indexable from an indexable of numbers.
   * @param indexable numbers indexable
   * @return expandable int indexable
   */
  @NotNull
  static ExpandableIntIndexable from(@NotNull Indexable<? extends Number> indexable)
  {
    return from(IntIndexable.viewIndexable(indexable));
  }

  /**
   * Create an expandable integer indexable from a collection of numbers.
   *
   * @param values collection of numbers, {@link java.util.List} is strongly
   *               recommended for better performance
   * @return expandable integer indexable
   */
  @NotNull
  @SuppressWarnings("unchecked")
  static ExpandableIntIndexable from(@NotNull Collection<? extends Number> values)
  {
    if (values instanceof java.util.List) {
      return from(Indexable.viewList((List<? extends Number>)values));
    }
    // need fast access, so no way around converting
    final Number[] numbers = values.toArray(Empty.NUMBER_ARRAY);
    return from(IntIndexable.viewNumberArray(numbers));
  }
}
