// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.*;

/**
 * Default implementation used for immutable expandable indexable.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 24, 2020
 */
class ExpandableIndexableImpl
{
  /** Partition size. */
  static final int PART_SIZE = 64;

  static final int LARGE_SIZE_LIMIT = Integer.MAX_VALUE / PART_SIZE;

  public static int size(@NotNull Indexable<?>[] indexables)
  {
    int size = 0;
    for (Indexable<?> indexable : indexables) {
      size += indexable.size();
    }
    return size;
  }

  private static abstract class ExpandableIndexableNode<T>
          extends Indexable.Base<T>
          implements ExpandableIndexable<T>
  {
    /**
     * Get the capacity of this node.
     * @return capacity
     */
    public abstract long getCapacity();

    /**
     * Get the weight of this node.
     * This is either the number of sub nodes ore
     * the number of elements.
     * @return weight
     */
    public abstract int getWeight();

    /**
     * Try to insert the value at the given index.
     * @param index index
     * @param value value to insert
     * @return the given value
     */
    @Nullable
    public abstract ExpandableIndexableNode<T> tryAdd(int index, T value);

    @NotNull
    @Override
    public abstract ExpandableIndexableNode<T> add(int index, T value);

    @NotNull
    @Override
    public abstract ExpandableIndexableNode<T> set(int index, T value);

    @NotNull
    @Override
    public abstract ExpandableIndexableNode<T> remove(int index);

    /**
     * Get the depth of the tree.
     * @return tree depth
     */
    public abstract int depth();

    /**
     * Is this node filled?
     * @return the answer
     */
    public abstract boolean isFilled();

    @Override
    public abstract int addToArray(@NotNull T[] array, int arrayPos, int index, int length);
  }

  private static ExpandableIndexableNode<Object> EMPTY =
          new ExpandableIndexableNode<Object>()
          {
            @NotNull
            @Override
            public ExpandableIndexableNode<Object> add(int index, Object value)
            {
              if (index != 0) {
                throw new IndexOutOfBoundsException("Cannot add outside range [0,0] but index is "+index+"!");
              }
              return new Leaf<>(value);
            }

            @Nullable
            @Override
            public ExpandableIndexableNode<Object> tryAdd(int index, Object value)
            {
              return add(index, value);
            }

            @NotNull
            @Override
            public ExpandableIndexableNode<Object> remove(int index)
            {
              throw new IndexOutOfBoundsException("No element to remove in empty indexable!");
            }

            @NotNull
            @Override
            public ExpandableIndexableNode<Object> set(int index, Object value)
            {
              throw new IndexOutOfBoundsException("No element to set in empty indexable!");
            }

            @NotNull
            @Override
            public ExpandableIndexable<Object> syt(int index, Object value)
            {
              throw new IndexOutOfBoundsException("No element to set in empty indexable!");
            }

            @Override
            public int size()
            {
              return 0;
            }

            @Override
            public Object get(int index)
            {
              throw new IndexOutOfBoundsException("No element to get in empty indexable!");
            }

            @Override
            public int getWeight()
            {
              return 0;
            }

            @Override
            public int depth()
            {
              return 1;
            }

            @NotNull
            @Override
            public Object[] toArray()
            {
              return Empty.OBJECT_ARRAY;
            }

            @Override
            public int addToArray(@NotNull Object[] array, int arrayPos, int index, int length)
            {
              return arrayPos;
            }

            @Override
            public boolean isFilled()
            {
              return false;  // because this is a special kind of Leaf
            }

            @Override
            public long getCapacity()
            {
              return PART_SIZE; // because this is a special kind of leaf
            }

            @Override
            public String toString()
            {
              return Indexable.EMPTY_INDEXABLE_STRING;
            }
          };

  @NotNull
  @SuppressWarnings("unchecked") // because EMPTY has not elements
  private static <E> ExpandableIndexableNode<E> empty()
  {
    return (ExpandableIndexableNode<E>)EMPTY;
  }

  private static class Leaf<T>
          extends ExpandableIndexableNode<T>
  {
    @NotNull
    private final Object[] values;  // can use Object here because values is not changing and never seen in the outside world

    /**
     * Constructor.
     * @param values values of this leaf
     */
    @SafeVarargs
    @SuppressWarnings("varargs")
    public Leaf(@NotNull T ... values)
    {
      this.values = values;
    }

    public Leaf(@NotNull T[] array, int start, int length)
    {
      this.values = Arrays.copyOfRange(array, start, start + length);
    }

    public Leaf(T addValue, @NotNull T[] array, int start, int length)
    {
      values = new Object[length + 1];
      values[0] = addValue;
      System.arraycopy(array, start, values, 1, length);
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked")  // see comment of field this.values
    public ExpandableIndexableNode<T> add(int index, T value)
    {
      if (index < 0  ||  index > values.length) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, values.length));
      }
      if (values.length == PART_SIZE) {
        // have to split
        if (index == PART_SIZE) {
          return new Node<>(this,
                            new Leaf<>(value));
        }
        else {
          return new Node<>(new Leaf<>((T[])values, 0, index),
                            new Leaf<>(value, (T[])values, index, values.length - index));
        }
      }
      final Object[] array = new Object[values.length + 1];
      if (index > 0) {
        System.arraycopy(values, 0, array, 0, index);
      }
      array[index] = value;
      if (index < values.length) {
        System.arraycopy(values, index, array, index + 1, values.length - index);
      }
      return new Leaf<T>((T[])array);
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked")  // see comment of field this.values
    public ExpandableIndexableNode<T> set(int index, T value)
    {
      if (index < 0  ||  index > values.length) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, values.length));
      }
      if (value == values[index]) {
        return this;
      }
      final Object[] newValues = values.clone();
      newValues[index] = value;
      return new Leaf<T>((T[])newValues);
    }

    @Override
    public ExpandableIndexableNode<T> tryAdd(int index, T value)
    {
      return values.length < PART_SIZE
              ? add(index, value)
              : null;
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked") // see comment of field this.values
    public ExpandableIndexableNode<T> remove(int index)
    {
      final int len = values.length;
      if (index < 0 || index >= len) {
        throw new IndexOutOfBoundsException(String.format("Index %d out of range [0,%d[!", index, values.length));
      }
      if (len == 1) {
        return empty();
      }
      if (index == len - 1) {
        return new Leaf<>((T[])Arrays.copyOf(values, len - 1));
      }
      final Object[] array = new Object[len - 1];
      if (index > 0) {
        System.arraycopy(values, 0, array, 0, index);
      }
      if (index < len - 1) {
        System.arraycopy(values, index + 1, array, index, len - index - 1);
      }
      return new Leaf<>((T[])array); // this works because array is never directly accessed from the outsi
    }

    @Override
    public int size()
    {
      return values.length;
    }

    @Override
    public int getWeight()
    {
      return values.length;
    }

    @Override
    public int depth()
    {
      return 1;
    }

    @Override
    @SuppressWarnings("unchecked") // see comment of field this.values
    public T get(int index)
    {
      if (index < 0  ||  index >= values.length) {
        throw new IndexOutOfBoundsException(String.format("Index %d out of range [0,%d[!", index, values.length));
      }
      return (T)values[index];
    }

    @Override
    @SuppressWarnings("unchecked") // see comment of field this.values
    public int addToArray(@NotNull T[] array, int arrayPos, int index, int length)
    {
      System.arraycopy((T[])values, index, array, arrayPos, length);
      return arrayPos + length;
    }

    @Override
    public boolean isFilled()
    {
      return values.length == PART_SIZE;
    }

    @Override
    public long getCapacity()
    {
      return PART_SIZE;
    }

    @Override
    public String toString()
    {
      return Countable.toString(this);
    }
  }

  private static class SubNode<T>
  {
    private final int offset;
    @NotNull
    private final ExpandableIndexableNode<T> sub;

    public SubNode(int offset, @NotNull ExpandableIndexableNode<T> sub)
    {
      this.offset = offset;
      this.sub = sub;
    }

    public int getEndIndex()
    {
      return offset + sub.size();
    }
  }

  private static class Node<T>
          extends ExpandableIndexableNode<T>
  {
    private final int numValues;
    private final long capacity;
    @NotNull
    private final SubNode<?>[] subNodes;

    @SuppressWarnings("unchecked") // see comment on field Leaf.values
    public Node(@NotNull ExpandableIndexableNode<T>... subNodes)
    {
      this.subNodes = new SubNode<?>[subNodes.length];
      long maxSubCap = PART_SIZE;
      int index = 0;
      int len = 0;
      for (ExpandableIndexableNode<?> sub : subNodes) {
         this.subNodes[index++] = new SubNode<>(len, (ExpandableIndexableNode<T>)sub);
         len += sub.size();
         maxSubCap = Math.max(maxSubCap, sub.getCapacity());
      }
      this.numValues = len;
      this.capacity = PART_SIZE * maxSubCap;
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked")  // see comment on field Lead.values
    public ExpandableIndexableNode<T> add(int index, T value)
    {
      if (numValues == Integer.MAX_VALUE) {
        throw new RuntimeException("Size is becoming too large for 32bit integer!");
      }
      if (index < 0  || index > numValues) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, numValues));
      }
      final ExpandableIndexableNode<T> trial = tryAdd(index, value);
      if (trial != null) {
        return trial;
      }

      if (index == numValues) {
        return new Node<T>(this, new Leaf<T>(value));
      }

      if (subNodes.length == PART_SIZE  &&  numValues <= LARGE_SIZE_LIMIT) {
        if (numValues < capacity / PART_SIZE) {
          // can't add something although quite empty, so let's recreate everything
          return recreateWithInsertedValue(index, value);
        }
      }

      final int subNodeIndex = subNodeIndex(index);
      final SubNode<T> oldSubNode = (SubNode<T>)subNodes[subNodeIndex];
      final int oldSubNodeIndex = index - oldSubNode.offset;
      final ExpandableIndexableNode<T> newSubNode1 = fromIndexable(oldSubNode.sub.headSet(oldSubNodeIndex));
      final Indexable<T> oldRest = oldSubNode.sub.tailSet(oldSubNodeIndex);
      final int size2 = oldRest.size() + 1;
      final ExpandableIndexableNode<T> newSubNode2 = fromIndexable(new Indexable.Base<T>()
      {
        @Override
        public int size()
        {
          return size2;
        }

        @Override
        public T get(int index)
        {
          return index == 0
                  ? value
                  : oldRest.get(index - 1);
        }

        @Override
        public int addToArray(@NotNull T[] array, int arrayPos, int index, int length)
        {
          if (index == 0) {
            array[arrayPos++] = value;
          }
          if (length > 1) {
            arrayPos += oldRest.addToArray(array, arrayPos, 0, length - 1);
          }
          return arrayPos;
        }
      });
      final ExpandableIndexableNode<?>[] newNodes = new ExpandableIndexableNode<?>[subNodes.length];
      for (int i = 0;  i < subNodeIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      final ExpandableIndexableNode<T> moveSubNode;
      newNodes[subNodeIndex] = newSubNode1;
      if (subNodeIndex == subNodes.length - 1) {
        moveSubNode = newSubNode2;
      }
      else {
        moveSubNode = (ExpandableIndexableNode<T>)subNodes[subNodes.length - 1].sub;
        newNodes[subNodeIndex + 1] = newSubNode2;

        for (int i = subNodeIndex + 2; i < subNodes.length; ++i) {
          newNodes[i] = subNodes[i - 1].sub;
        }
      }
      return new Node<>(new Node<>((ExpandableIndexableNode<T>[])newNodes),
                        new Node<>(moveSubNode));
    }

    @Override
    @SuppressWarnings("unchecked")
    public ExpandableIndexableNode<T> tryAdd(int index, T value)
    {
      final int subIndex = subNodeIndex(index);
      final SubNode<T> subNode = (SubNode<T>)subNodes[subIndex];
      ExpandableIndexableNode<T> newSub = subNode.sub.tryAdd(index - subNode.offset, value);
      if (newSub != null) {
        return withExchangedSubNode(subIndex, newSub);
      }

      if (index == subNode.getEndIndex() && subIndex < subNodes.length - 1) {
        // try to add to the start of the following node
        final SubNode<T> nextSub = (SubNode<T>)subNodes[subIndex + 1];
        newSub = nextSub.sub.tryAdd(index - nextSub.offset, value);
        if (newSub != null) {
          return withExchangedSubNode(subIndex + 1, newSub);
        }
      }

      if (index == subNode.offset  &&  subIndex > 0) {
        // try to add to the end of previous node
        final SubNode<T> prevSub = (SubNode<T>)subNodes[subIndex - 1];
        newSub = prevSub.sub.tryAdd(index - prevSub.offset, value);
        if (newSub != null) {
          return withExchangedSubNode(subIndex - 1, newSub);
        }
      }

      if (index == size()) {
        if (subNodes.length == PART_SIZE) {
          return null;
        }
        return withInsertedSubNode(subNodes.length, new Leaf<T>(value));
      }

      if (isFilled()) {
        return null;
      }

      // recreate this node
      return recreateWithInsertedValue(index, value);
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked")  // see comment of field this.values
    public ExpandableIndexableNode<T> set(int index, T value)
    {
      if (index < 0  ||  index >= numValues) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, numValues));
      }
      final int subNodeIndex = subNodeIndex(index);
      final SubNode<T> subNode = (SubNode<T>)subNodes[subNodeIndex];
      final int subIndex = index - subNode.offset;
      final ExpandableIndexableNode<T> newSubNode = subNode.sub.set(subIndex, value);
      return withExchangedSubNode(subNodeIndex, newSubNode);
    }

    @NotNull
    private ExpandableIndexableNode<T> recreateWithInsertedValue(int pos, T value)
    {
      /* To avoid creating a large array we use a view to this Node which just virtually inserts the given value. */
      final int size = numValues + 1;
      return fromIndexable(new Indexable.Base<T>()
      {
        @Override
        public int size()
        {
          return size;
        }

        @Override
        public T get(int idx)
        {
          if (idx < pos) {
            return ExpandableIndexableImpl.Node.this.get(idx);
          }
          if (idx > pos) {
            return ExpandableIndexableImpl.Node.this.get(idx - 1);
          }
          return value;
        }

        @Override
        public int addToArray(@NotNull T[] array, int arrayPos, int index, int length)
        {
          final int endIndex = index + length;
          if (index < pos) {
            final int copyLen = Math.min(length, pos - index);
            arrayPos = ExpandableIndexableImpl.Node.this.addToArray(array, arrayPos,
                                                                    index, copyLen);
            length -= copyLen;
          }
          if (index <= pos  &&  endIndex > pos) {
            array[arrayPos++] = value;
            --length;
          }
          if (endIndex > pos + 1) {
            arrayPos = ExpandableIndexableImpl.Node.this.addToArray(array, arrayPos,
                                                                    pos, length);
          }
          return arrayPos;
        }
      });
    }

    @NotNull
    @SuppressWarnings("unchecked")
    private ExpandableIndexableNode<T> withExchangedSubNode(int subIndex,
                                                            @NotNull ExpandableIndexableNode<T> substNode)
    {
      final ExpandableIndexableNode<?>[] newNodes = new ExpandableIndexableNode<?>[subNodes.length];
      for (int i = 0;  i < subIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      newNodes[subIndex] = substNode;
      for (int i = subIndex + 1;  i < newNodes.length;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      return new Node<T>((ExpandableIndexableNode<T>[])newNodes);
    }

    @NotNull
    @SuppressWarnings("unchecked")
    private ExpandableIndexableNode<T> withInsertedSubNode(int subIndex,
                                                           @NotNull ExpandableIndexableNode<T> substNode)
    {
      final ExpandableIndexableNode<?>[] newNodes = new ExpandableIndexableNode<?>[subNodes.length + 1];
      for (int i = 0;  i < subIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      newNodes[subIndex] = substNode;
      for (int i = subIndex + 1;  i < newNodes.length;  ++i) {
        newNodes[i] = subNodes[i - 1].sub;
      }
      return new Node<>((ExpandableIndexableNode<T>[])newNodes);
    }

    /**
     * Calculate the sub node index for a given element index.
     * @param elementIndex element index
     * @return sub node index
     */
    @SuppressWarnings("unchecked")
    private int subNodeIndex(int elementIndex)
    {
      if (elementIndex == numValues) {
        return subNodes.length - 1;
      }
      int minSubIndex = 0;
      int maxSubIndex = subNodes.length;
      while (maxSubIndex > minSubIndex - 1)  {
        final int midSubIndex = (maxSubIndex + minSubIndex) / 2;
        final SubNode<T> sub = (SubNode<T>)subNodes[midSubIndex];
        if (elementIndex < sub.offset) {
          maxSubIndex = midSubIndex;
        }
        else if (elementIndex >= sub.offset + sub.sub.size()) {
          minSubIndex = midSubIndex + 1;
        }
        else {
          return midSubIndex;
        }
      }
      return minSubIndex;
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked")
    public ExpandableIndexableNode<T> remove(int index)
    {
      if (index < 0  ||  index >= numValues) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, numValues));
      }
      final int subNodeIndex = subNodeIndex(index);
      final SubNode<T> subNode = (SubNode<T>)subNodes[subNodeIndex];
      final ExpandableIndexableNode<T> newSubNode = subNode.sub.remove(index - subNode.offset);
      if (newSubNode.isEmpty()) {
        if (subNodes.length == 1) {
          return empty();
        }
        final ExpandableIndexableNode<?>[] newNodes = new ExpandableIndexableNode<?>[subNodes.length - 1];
        for (int i = 0;  i < subNodeIndex;  ++i) {
          newNodes[i] = subNodes[i].sub;
        }
        for (int i = subNodeIndex;  i < newNodes.length;  ++i) {
          newNodes[i] = subNodes[i+1].sub;
        }
        return new ExpandableIndexableImpl.Node<T>((ExpandableIndexableNode<T>[])newNodes);
      }

      final ExpandableIndexableNode<?>[] newNodes = new ExpandableIndexableNode<?>[subNodes.length];
      for (int i = 0;  i < subNodeIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      newNodes[subNodeIndex] = newSubNode;
      for (int i = subNodeIndex + 1;  i < newNodes.length;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      return new ExpandableIndexableImpl.Node<T>((ExpandableIndexableNode<T>[])newNodes);
    }

    @Override
    public int size()
    {
      return numValues;
    }

    @Override
    public int getWeight()
    {
      return subNodes.length;
    }

    @Override
    public int depth()
    {
      int maxDepth = 0;
      for (SubNode<?> subNode : subNodes) {
        maxDepth = Math.max(maxDepth, subNode.sub.depth());
      }
      return maxDepth + 1;
    }

    @Override
    @SuppressWarnings("unchecked")
    public T get(int index)
    {
      if (index < 0  ||  index >= numValues) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d[!", index, numValues));
      }
      final SubNode<T> subNode = (SubNode<T>)subNodes[subNodeIndex(index)];
      return subNode.sub.get(index - subNode.offset);
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked")
    public Iterator<T> iterator(int from, int to)
    {
      if (subNodes.length == 1) {
        return ((ExpandableIndexableNode<T>)subNodes[0].sub).iterator(from, to);
      }
      final int fromSubIndex = subNodeIndex(from);
      final int toSubIndex = subNodeIndex(to);

      final ExpandableIndexableImpl.SubNode<T> subNode = (SubNode<T>)subNodes[fromSubIndex];
      if (fromSubIndex == toSubIndex) {
        return subNode.sub.iterator(from - subNode.offset, to - subNode.offset);
      }
      final Collection<Iterator<T>> subIterators = new ArrayList<>(toSubIndex - fromSubIndex + 1);
      subIterators.add(subNode.sub.iterator(from - subNode.offset, subNode.sub.size()));
      for (int i = fromSubIndex + 1;  i < toSubIndex;  ++i) {
        final ExpandableIndexableImpl.SubNode<T> node = (SubNode<T>)subNodes[i];
        subIterators.add(node.sub.iterator());
      }
      final ExpandableIndexableImpl.SubNode<T> sub = (SubNode<T>)subNodes[toSubIndex];
      subIterators.add(sub.sub.iterator(0, to - sub.offset)) ;

      final Iterator<Iterator<T>> subIteratorIterator = subIterators.iterator();
      return new Iterator<T>()
      {
        private Iterator<T> subIterator = subIteratorIterator.hasNext()
                ? subIteratorIterator.next()
                : null;

        @Override
        public T next()
        {
          if (subIterator == null) {
            throw new NoSuchElementException("Cannot iterate past the end!");
          }
          final T value = subIterator.next();
          while (!subIterator.hasNext()) {
            if (subIteratorIterator.hasNext()) {
              subIterator = subIteratorIterator.next();
            }
            else {
              subIterator = null;
              break;
            }
          }
          return value;
        }

        @Override
        public boolean hasNext()
        {
          return subIterator != null;
        }
      };
    }

    @Override
    @SuppressWarnings("unchecked")
    public int addToArray(@NotNull T[] array, int arrayPos, int index, int length)
    {
      final int start = index;
      final int end   = index + length;
      final int startSubIndex = subNodeIndex(start);
      final int endSubIndex   = subNodeIndex(end);
      if (startSubIndex == endSubIndex) {
        final SubNode<T> node = (SubNode<T>)subNodes[startSubIndex];
        return node.sub.addToArray(array, arrayPos, index - node.offset, length);
      }
      final SubNode<T> startNode = (SubNode<T>)subNodes[startSubIndex];
      final int startIndex = index - startNode.offset;
      arrayPos = startNode.sub.addToArray(array, arrayPos, startIndex, startNode.sub.size() - startIndex);
      for (int i = startSubIndex + 1;  i < endSubIndex;  ++i) {
        final SubNode<T> node = (SubNode<T>)subNodes[i];
        arrayPos = node.sub.addToArray(array, arrayPos);
      }
      final SubNode<T> endNode = (SubNode<T>)subNodes[endSubIndex];
      return endNode.sub.addToArray(array, arrayPos, 0, end - endNode.offset);
    }

    @Override
    public boolean isFilled()
    {
      if (subNodes.length < PART_SIZE) {
        return true;
      }
      for (SubNode<?> subNode : subNodes) {
        if (!subNode.sub.isFilled()) {
          return false;
        }
      }
      return true;
    }

    @Override
    public long getCapacity()
    {
      return capacity;
    }
  }

  /**
   * Create an expandable int indexable node from the given values.
   * @param values integer array with values
   * @param start start index into array to first element to be copied
   * @param length number of elements to copy
   * @param <E> element type
   * @return expandable int indexable
   */
  @NotNull
  @SuppressWarnings("unchecked")
  static <E> ExpandableIndexableNode<E> fromArray(E[] values, int start, int length)
  {
    if (length <= PART_SIZE) {
      return new Leaf<>(values, start, length);
    }
    int nextSize = PART_SIZE * PART_SIZE;
    int size = PART_SIZE;
    while (nextSize < length) {
      size = nextSize;
      nextSize = Integer.MAX_VALUE / size < PART_SIZE
              ? size * PART_SIZE
              : Integer.MAX_VALUE;
    }
    final ExpandableIndexableNode<?>[] nodes = new ExpandableIndexableNode<?>[(length + size - 1) / size];
    int nodeIndex = 0;
    int pos;
    final int boundary = length - size;
    for (pos = 0;  pos < boundary  &&  pos >= 0;  pos += size) { // >=0 is for very large arrays
      nodes[nodeIndex++] = fromArray(values, start + pos, size);
    }
    nodes[nodeIndex] = fromArray(values, start + pos, length - pos);
    return new Node<>((ExpandableIndexableNode<E>[])nodes);
  }

  /**
   * Create an expandable int indexable node from the given values.
   * @param values integer array with values
   * @param <E> element type
   * @return expandable int indexable
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  static <E> ExpandableIndexableNode<E> fromArray(E ... values)
  {
    final int length = values.length;
    if (length == 0) {
      return empty();
    }
    if (length <= PART_SIZE) {
      return new Leaf<>(values);
    }
    else {
      return fromArray(values, 0, length);
    }
  }

  /**
   * Create an expandable integer indexable node from a given indexable.
   * @param indexable indexable defining the values of the returned indexable
   * @param <E> element type
   * @return expandable int indexable
   */
  @NotNull
  @SuppressWarnings("unchecked")
  static <E> ExpandableIndexableNode<E> fromIndexable(@NotNull Indexable<? extends E> indexable)
  {
    if (indexable.isEmpty()) {
      return empty();
    }
    if (indexable instanceof ExpandableIndexableNode) {
      // this works because ExpandableIndexableNode is immutable
      return (ExpandableIndexableNode<E>)indexable;
    }
    final int length = indexable.size();
    if (length <= PART_SIZE) {
      return new Leaf<>((E[])indexable.toArray());
    }
    int nextSize = PART_SIZE * PART_SIZE;
    int size = PART_SIZE;
    while (nextSize < length) {
      size = nextSize;
      nextSize = Integer.MAX_VALUE / size < PART_SIZE
              ? size * PART_SIZE
              : Integer.MAX_VALUE;
    }
    final ExpandableIndexableNode<?>[] nodes = new ExpandableIndexableNode<?>[(length + size - 1) / size];
    int nodeIndex = 0;
    int pos;
    final int boundary = length - size;
    for (pos = 0;  pos >= 0   &&  pos < boundary;  pos += size) { // >=0 is for very large arrays
      nodes[nodeIndex++] = fromIndexable(indexable.subSet(pos, pos + size));
    }
    nodes[nodeIndex] = fromIndexable(indexable.tailSet(pos));
    return new Node<>((ExpandableIndexableNode<E>[])nodes);
  }
}
