// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;
import java.util.function.IntFunction;

/**
 * Immutable expandable indexable.
 * <p>
 * This interface defines an indexable which allows
 * to change (set, add, remove) its values but is immutable
 * nevertheless. All changing methods return copies, thus
 * guaranteeing immutability.
 * <p>
 * The immutablility guarantee is only valid for the indexable
 * implementation, mutable elements which are changed will
 * lead to strange effects, eg they are changed in all copies.
 * <p>
 * The internal implementation tries to keep copying a cheap
 * operation, but some access patterns may result in performance
 * degration. It is therefore recommended not to use this in
 * time-critical code.
 * <p>
 * Initial creation is done by using the following methods:
 * <ul>
 *   <li>{@link #empty()}</li>
 *   <li>{@link #nulled(int)}</li>
 *   <li>{@link #fromArray(Object...)}</li>
 *   <li>{@link #fromArray(Object[], int, int)}</li>
 *   <li>{@link #fromIndexable(Indexable)}</li>
 *   <li>{@link #fromCollection(Collection)}</li>
 * </ul>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 24, 2020
 */
public interface ExpandableIndexable<T>
        extends Indexable<T>
{
  /**
   * Return an expandable indexable  where the the given value
   * was inserted at the given index.
   * @param index index in the range {@code 0} (insert at first position)
   *              to {@code size()} (insert after last position)
   * @param value value to insert at the given position
   * @return new indexable with the given value inserted
   */
  @NotNull
  ExpandableIndexable<T> add(int index, T value);

  /**
   * Return an expandable int indexable where the value at the given index
   * was removed.
   * @param index index to remove, in the range {@code 0} to
   *              {@code size() - 1}
   * @return new expandable indexable with the given index removed
   */
  @NotNull
  ExpandableIndexable<T> remove(int index);

  /**
   * Set the value at the given index.
   * @param index  index
   * @param value  new value
   * @return new expandable indexable with the given value exchanged
   */
  @NotNull
  ExpandableIndexable<T> set(int index, T value);

  /**
   * Pythonesque set the value at the given index.
   * @param index index, negative to count from the back
   * @param value value to set
   * @return new expandable indexable with the given value exchanged
   */
  @NotNull
  default ExpandableIndexable<T> syt(int index, T value)
  {
    return set(index < 0 ? size() - index : index, value);
  }

  /**
   * Return an expandable int indexable where the given
   * value was added to the end of this indexable.
   * @param value value to add
   * @return new expandable indexable with added value
   */
  @NotNull
  default ExpandableIndexable<T> add(T value)
  {
    return add(size(), value);
  }

  /**
   * Pythonesque insertion allowing negative indexes to insert from the end.
   * @param index positive values behave the same way as {@link #add(int, Object)},
   *              but negative values count from the back, so {@code -1}
   *              will insert before the last element
   * @param value value to insert
   * @return new expandable indexable with the given value inserted
   */
  @NotNull
  default ExpandableIndexable<T> ydd(int index, T value)
  {
    if (index >= 0) {
      return add(index, value);
    }
    else {
      final int realIndex = index + size();
      if (realIndex < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           index, size()));
      }
      return add(realIndex, value);
    }
  }

  /**
   * Pythonesque remove allowing negative indexes to remove from the end.
   * @param index positive values behave the same as {@link #remove(int)},
   *              while negative indexes count from the back, so {@code -1}
   *              will remove the last element
   * @return new expandable indexable with the given index removed
   */
  @NotNull
  default ExpandableIndexable<T> remyve(int index)
  {
    if (index >= 0) {
      return remove(index);
    }
    else {
      final int realIndex = index + size();
      if (realIndex < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           index, size()));
      }
      return remove(realIndex);
    }
  }

  /**
   * Add all elements to this indexable and return the result.
   * @param elements elements to add
   * @return resulting expandable indexable
   */
  @NotNull
  default ExpandableIndexable<T> addAll(@NotNull Iterable<? extends T> elements)
  {
    ExpandableIndexable<T> result = this;
    for (T elem : elements) {
      result = result.add(elem);
    }
    return result;
  }

  @NotNull
  @Override
  default Iterator<T> iterator()
  {
    return iterator(0, size());
  }

  /**
   * Convenience method to create an empty expandable int indexable.
   * @param <E> element type
   * @return empty expandable int indexable
   */
  @NotNull
  static <E> ExpandableIndexable<E> empty()
  {
    return fromArray();
  }

  /**
   * Create an expandable indexable with a given size, and initialize its elements by index.
   * @param size     size of the indexable
   * @param producer element producer which will be called for each index
   * @param <E> element type
   * @return expandable indexable with the given size containing the elements created by the producer
   * @see #viewByIndex(int, IntFunction)
   */
  @NotNull
  static <E> ExpandableIndexable<E> initByIndex(int size, @NotNull IntFunction<? extends E> producer)
  {
    return fromIndexable(Indexable.viewByIndex(size, producer));
  }

  /**
   * Create an expandable indexable with a given size, and initialize all its elements to {@code null}.
   * @param size size of the returned indexable
   * @param <E> element type
   * @return expandable indexable with the given size and {@code null} elements
   */
  @NotNull
  static <E> ExpandableIndexable<E> nulled(int size)
  {
    return initByIndex(size, idx -> null);
  }

  /**
   * Create an expandable indexable from an initial array of values.
   * @param values array of values
   * @param <E> element type
   * @return expandable integer indexable
   * @deprecated overlaps with other {@code from()} methods, use {@link #fromArray(Object[])}  instead
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  @Deprecated
  static <E> ExpandableIndexable<E> from(@NotNull E ... values)
  {
    return fromArray(values);
  }

  /**
   * Create an expandable integer indexable from an initial array of values.
   * @param values array of values
   * @param <E> element type
   * @return expandable integer indexable
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  static <E> ExpandableIndexable<E> fromArray(@NotNull E ... values)
  {
    return ExpandableIndexableImpl.fromArray(values);
  }

  /**
   * Create an expandable integer indexable from part of an int array.
   * @param array    integer array
   * @param start    start position in array
   * @param length   number of byte in array
   * @param <E> element type
   * @return expandable integer indexable
   * @deprecated overlaps with other {@code from()} methods, use {@link #fromArray(Object[], int, int)}  instead
   */
  @NotNull
  @Deprecated
  static <E> ExpandableIndexable<E> from(@NotNull E[] array, int start, int length)
  {
    return ExpandableIndexableImpl.fromArray(array, start, length);
  }

  /**
   * Create an expandable integer indexable from part of an int array.
   * @param array    integer array
   * @param start    start position in array
   * @param length   number of byte in array
   * @param <E> element type
   * @return expandable integer indexable
   */
  @NotNull
  static <E> ExpandableIndexable<E> fromArray(@NotNull E[] array, int start, int length)
  {
    return ExpandableIndexableImpl.fromArray(array, start, length);
  }

  /**
   * Get an expandable integer indexable from a normal
   * integer indexable.
   * @param indexable  standard integer indexable
   * @param <E> element type
   * @return expandable copy of the given indexable
   * @deprecated overlaps with other {@code from()} methods, use {@link #fromIndexable(Indexable)}  instead
   */
  @NotNull
  @Deprecated
  static <E> ExpandableIndexable<E> from(@NotNull Indexable<E> indexable)
  {
    return ExpandableIndexableImpl.fromIndexable(indexable);
  }

  /**
   * Get an expandable integer indexable from a normal
   * integer indexable.
   * @param indexable  standard integer indexable
   * @param <E> element type
   * @return expandable copy of the given indexable
   */
  @NotNull
  static <E> ExpandableIndexable<E> fromIndexable(@NotNull Indexable<E> indexable)
  {
    return ExpandableIndexableImpl.fromIndexable(indexable);
  }

  /**
   * Create an expandable integer indexable from a collection of numbers.
   *
   * @param values collection of numbers, {@link List} is strongly
   *               recommended for better performance
   * @param <E> element type
   * @return expandable integer indexable
   * @deprecated overlaps with other {@code from()} methods, use {@link #fromCollection(Collection)}  instead
   */
  @NotNull
  @Deprecated
  static <E> ExpandableIndexable<E> from(@NotNull Collection<? extends E> values)
  {
    return fromCollection(values);
  }

  /**
   * Create an expandable integer indexable from a collection of numbers.
   *
   * @param values collection of numbers, {@link List} is strongly
   *               recommended for better performance
   * @param <E> element type
   * @return expandable integer indexable
   */
  @NotNull
  @SuppressWarnings("unchecked")
  static <E> ExpandableIndexable<E> fromCollection(@NotNull Collection<? extends E> values)
  {
    if (values instanceof List) {
      return fromIndexable(Indexable.viewList((List<? extends E>)values));
    }
    // need fast access, so no way around converting
    return ExpandableIndexableImpl.fromArray((E[])values.toArray());
  }
}
