// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.*;

/**
 * Default implementation used for immutable expandable double indexable.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 24, 2020
 */
class ExpandableDoubleIndexableImpl
{
  /** Partition size. */
  static final int PART_SIZE = 64;

  static final int LARGE_SIZE_LIMIT = Integer.MAX_VALUE / PART_SIZE;

  public static int size(@NotNull DoubleIndexable[] indexables)
  {
    int size = 0;
    for (DoubleIndexable indexable : indexables) {
      size += indexable.size();
    }
    return size;
  }

  private static abstract class ExpandableDoubleIndexableNode
          extends DoubleIndexable.Base
          implements ExpandableDoubleIndexable
  {
    /**
     * Get the capacity of this node.
     * @return capacity
     */
    public abstract long getCapacity();

    /**
     * Get the weight of this node.
     * This is either the number of sub nodes ore
     * the number of elements.
     * @return weight
     */
    public abstract int getWeight();

    /**
     * Try to insert the value at the given index.
     * @param index index
     * @param value value to insert
     * @return the given value
     */
    @Nullable
    public abstract ExpandableDoubleIndexableNode tryAdd(int index, double value);

    @NotNull
    @Override
    public abstract ExpandableDoubleIndexableNode add(int index, double value);

    @NotNull
    @Override
    public abstract ExpandableDoubleIndexableNode remove(int index);

    /**
     * Get the depth of the tree.
     * @return tree depth
     */
    public abstract int depth();

    /**
     * Is this node filled?
     * @return the answer
     */
    public abstract boolean isFilled();

    @Override
    public abstract int addToArray(@NotNull double[] array, int arrayPos, int index, int length);
  }

  private static ExpandableDoubleIndexableNode EMPTY =
          new ExpandableDoubleIndexableNode()
          {
            @NotNull
            @Override
            public ExpandableDoubleIndexableNode add(int index, double value)
            {
              if (index != 0) {
                throw new IndexOutOfBoundsException("Cannot add outside range [0,0] but index is "+index+"!");
              }
              return new Leaf(value);
            }

            @Override
            public ExpandableDoubleIndexableNode tryAdd(int index, double value)
            {
              return add(index, value);
            }

            @NotNull
            @Override
            public ExpandableDoubleIndexableNode remove(int index)
            {
              throw new IndexOutOfBoundsException("No element to remove in empty indexable!");
            }

            @NotNull
            @Override
            public ExpandableDoubleIndexable getCopy()
            {
              return this;
            }

            @Override
            public int size()
            {
              return 0;
            }

            @Override
            public double get(int index)
            {
              throw new IndexOutOfBoundsException("No element to get in empty indexable!");
            }

            @Override
            public int getWeight()
            {
              return 0;
            }

            @Override
            public int depth()
            {
              return 1;
            }

            @Override
            public int addToArray(@NotNull double[] array, int pos)
            {
              return pos;
            }

            @Override
            public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
            {
              return arrayPos;
            }

            @NotNull
            @Override
            public double[] toArray()
            {
              return Empty.DOUBLE_ARRAY;
            }

            @Override
            public boolean isFilled()
            {
              return false;  // because this is a special kind of Leaf
            }

            @Override
            public long getCapacity()
            {
              return PART_SIZE; // because this is a special kind of leaf
            }

            @Override
            public String toString()
            {
              return Indexable.EMPTY_INDEXABLE_STRING;
            }
          };

  private static class Leaf
          extends ExpandableDoubleIndexableNode
  {
    @NotNull
    private final double[] values;

    /**
     * Constructor.
     * @param values values of this leaf
     */
    public Leaf(@NotNull double... values)
    {
      this.values = values;
    }

    public Leaf(@NotNull double[] array, int start, int length)
    {
      this.values = Arrays.copyOfRange(array, start, start + length);
    }

    public Leaf(double addValue, @NotNull double[] array, int start, int length)
    {
      values = new double[length + 1];
      values[0] = addValue;
      System.arraycopy(array, start, values, 1, length);
    }

    @NotNull
    @Override
    public ExpandableDoubleIndexableNode add(int index, double value)
    {
      if (index < 0  ||  index > values.length) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, values.length));
      }
      if (values.length == PART_SIZE) {
        // have to split
        if (index == PART_SIZE) {
          return new Node(this,
                          new Leaf(value));
        }
        else {
          return new Node(new Leaf(values, 0, index),
                          new Leaf(value, values, index, values.length - index));
        }
      }
      final double[] array = new double[values.length + 1];
      if (index > 0) {
        System.arraycopy(values, 0, array, 0, index);
      }
      array[index] = value;
      if (index < values.length) {
        System.arraycopy(values, index, array, index + 1, values.length - index);
      }
      return new Leaf(array);
    }

    @Override
    public ExpandableDoubleIndexableNode tryAdd(int index, double value)
    {
      return values.length < PART_SIZE
              ? add(index, value)
              : null;
    }

    @NotNull
    @Override
    public ExpandableDoubleIndexableNode remove(int index)
    {
      final int len = values.length;
      if (index < 0 || index >= len) {
        throw new IndexOutOfBoundsException(String.format("Index %d out of range [0,%d[!", index, values.length));
      }
      if (len == 1) {
        return ExpandableDoubleIndexableImpl.EMPTY;
      }
      if (index == len - 1) {
        return new Leaf(Arrays.copyOf(values, len - 1));
      }
      final double[] array = new double[len - 1];
      if (index > 0) {
        System.arraycopy(values, 0, array, 0, index);
      }
      if (index < len - 1) {
        System.arraycopy(values, index + 1, array, index, len - index - 1);
      }
      return new Leaf(array);
    }

    @NotNull
    @Override
    public ExpandableDoubleIndexable getCopy()
    {
      return this;
    }

    @Override
    public int size()
    {
      return values.length;
    }

    @Override
    public int getWeight()
    {
      return values.length;
    }

    @Override
    public int depth()
    {
      return 1;
    }

    @Override
    public double get(int index)
    {
      if (index < 0  ||  index >= values.length) {
        throw new IndexOutOfBoundsException(String.format("Index %d out of range [0,%d[!", index, values.length));
      }
      return values[index];
    }

    @Override
    public int addToArray(@NotNull double[] array, int pos)
    {
      System.arraycopy(values, 0, array, pos, values.length);
      return pos + values.length;
    }

    @Override
    public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
    {
      System.arraycopy(values, index, array, arrayPos, length);
      return arrayPos + length;
    }

    @NotNull
    @Override
    public double[] toArray()
    {
      return values.clone();
    }

    @Override
    public boolean isFilled()
    {
      return values.length == PART_SIZE;
    }

    @Override
    public long getCapacity()
    {
      return PART_SIZE;
    }

    @Override
    public String toString()
    {
      return DoubleIndexable.toString(this);
    }
  }

  private static class SubNode
  {
    private final int offset;
    @NotNull
    private final ExpandableDoubleIndexableNode sub;

    public SubNode(int offset, @NotNull ExpandableDoubleIndexableNode sub)
    {
      this.offset = offset;
      this.sub = sub;
    }

    public int getEndIndex()
    {
      return offset + sub.size();
    }
  }

  private static class Node
          extends ExpandableDoubleIndexableNode
  {
    private final int numValues;
    private final long capacity;
    @NotNull
    private final SubNode[] subNodes;

    public Node(@NotNull ExpandableDoubleIndexableNode... subNodes)
    {
      this.subNodes = new SubNode[subNodes.length];
      long maxSubCap = PART_SIZE;
      int index = 0;
      int len = 0;
      for (ExpandableDoubleIndexableNode sub : subNodes) {
         this.subNodes[index++] = new SubNode(len, sub);
         len += sub.size();
         maxSubCap = Math.max(maxSubCap, sub.getCapacity());
      }
      this.numValues = len;
      this.capacity = PART_SIZE * maxSubCap;
    }

    @NotNull
    @Override
    public ExpandableDoubleIndexableNode add(int index, double value)
    {
      if (numValues == Integer.MAX_VALUE) {
        throw new RuntimeException("Size is becoming too large for 32bit integer!");
      }
      if (index < 0  || index > numValues) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, numValues));
      }
      final ExpandableDoubleIndexableNode trial = tryAdd(index, value);
      if (trial != null) {
        return trial;
      }

      if (index == numValues) {
        return new Node(this, new Leaf(value));
      }

      if (subNodes.length == PART_SIZE  &&  numValues <= LARGE_SIZE_LIMIT) {
        if (numValues < capacity / PART_SIZE) {
          // can't add something although quite empty, so let's recreate everything
          return recreateWithInsertedValue(index, value);
        }
      }

      final int subNodeIndex = subNodeIndex(index);
      final SubNode oldSubNode = subNodes[subNodeIndex];
      final int oldSubNodeIndex = index - oldSubNode.offset;
      final ExpandableDoubleIndexableNode newSubNode1 = from(oldSubNode.sub.headSet(oldSubNodeIndex));
      final DoubleIndexable oldRest = oldSubNode.sub.tailSet(oldSubNodeIndex);
      final int size2 = oldRest.size() + 1;
      final ExpandableDoubleIndexableNode newSubNode2 = from(new DoubleIndexable.Base()
      {
        @Override
        public int size()
        {
          return size2;
        }

        @Override
        public double get(int index)
        {
          return index == 0
                  ? value
                  : oldRest.get(index - 1);
        }

        @Override
        public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
        {
          if (index == 0) {
            array[arrayPos++] = value;
          }
          if (length > 1) {
            arrayPos += oldRest.addToArray(array, arrayPos, 0, length - 1);
          }
          return arrayPos;
        }
      });
      final ExpandableDoubleIndexableNode[] newNodes = new ExpandableDoubleIndexableNode[subNodes.length];
      for (int i = 0;  i < subNodeIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      final ExpandableDoubleIndexableNode moveSubNode;
      newNodes[subNodeIndex] = newSubNode1;
      if (subNodeIndex == subNodes.length - 1) {
        moveSubNode = newSubNode2;
      }
      else {
        moveSubNode = subNodes[subNodes.length - 1].sub;
        newNodes[subNodeIndex + 1] = newSubNode2;

        for (int i = subNodeIndex + 2; i < subNodes.length; ++i) {
          newNodes[i] = subNodes[i - 1].sub;
        }
      }
      return new Node(new Node(newNodes),
                      new Node(moveSubNode));
    }

    @Override
    public ExpandableDoubleIndexableNode tryAdd(int index, double value)
    {
      final int subIndex = subNodeIndex(index);
      final SubNode subNode = subNodes[subIndex];
      ExpandableDoubleIndexableNode newSub = subNode.sub.tryAdd(index - subNode.offset, value);
      if (newSub != null) {
        return withExchangedSubNode(subIndex, newSub);
      }

      if (index == subNode.getEndIndex() && subIndex < subNodes.length - 1) {
        // try to add to the start of the following node
        final SubNode nextSub = subNodes[subIndex + 1];
        newSub = nextSub.sub.tryAdd(index - nextSub.offset, value);
        if (newSub != null) {
          return withExchangedSubNode(subIndex + 1, newSub);
        }
      }

      if (index == subNode.offset  &&  subIndex > 0) {
        // try to add to the end of previous node
        final SubNode prevSub = subNodes[subIndex - 1];
        newSub = prevSub.sub.tryAdd(index - prevSub.offset, value);
        if (newSub != null) {
          return withExchangedSubNode(subIndex - 1, newSub);
        }
      }

      if (index == size()) {
        if (subNodes.length == PART_SIZE) {
          return null;
        }
        return withInsertedSubNode(subNodes.length, new Leaf(value));
      }

      if (isFilled()) {
        return null;
      }

      // recreate this node
      return recreateWithInsertedValue(index, value);
    }

    @NotNull
    private ExpandableDoubleIndexableNode recreateWithInsertedValue(int pos, double value)
    {
      /* To avoid creating a large array we use a view to this Node which just virtually inserts the given value. */
      final int size = numValues + 1;
      return from(new DoubleIndexable.Base()
      {
        @Override
        public int size()
        {
          return size;
        }

        @Override
        public double get(int idx)
        {
          if (idx < pos) {
            return ExpandableDoubleIndexableImpl.Node.this.get(idx);
          }
          if (idx > pos) {
            return ExpandableDoubleIndexableImpl.Node.this.get(idx - 1);
          }
          return value;
        }

        @Override
        public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
        {
          final int endIndex = index + length;
          if (index < pos) {
            final int copyLen = Math.min(length, pos - index);
            arrayPos = ExpandableDoubleIndexableImpl.Node.this.addToArray(array, arrayPos,
                                                                          index, copyLen);
            length -= copyLen;
          }
          if (index <= pos  &&  endIndex > pos) {
            array[arrayPos++] = value;
            --length;
          }
          if (endIndex > pos + 1) {
            arrayPos = ExpandableDoubleIndexableImpl.Node.this.addToArray(array, arrayPos,
                                                                          pos, length);
          }
          return arrayPos;
        }
      });
    }

    @NotNull
    private ExpandableDoubleIndexableNode withExchangedSubNode(int subIndex,
                                                               @NotNull ExpandableDoubleIndexableNode substNode)
    {
      final ExpandableDoubleIndexableNode[] newNodes = new ExpandableDoubleIndexableNode[subNodes.length];
      for (int i = 0;  i < subIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      newNodes[subIndex] = substNode;
      for (int i = subIndex + 1;  i < newNodes.length;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      return new Node(newNodes);
    }

    @NotNull
    private ExpandableDoubleIndexableNode withInsertedSubNode(int subIndex,
                                                              @NotNull ExpandableDoubleIndexableNode substNode)
    {
      final ExpandableDoubleIndexableNode[] newNodes = new ExpandableDoubleIndexableNode[subNodes.length + 1];
      for (int i = 0;  i < subIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      newNodes[subIndex] = substNode;
      for (int i = subIndex + 1;  i < newNodes.length;  ++i) {
        newNodes[i] = subNodes[i - 1].sub;
      }
      return new Node(newNodes);
    }


    /**
     * Calculate the sub node index for a given element index.
     * @param elementIndex element index
     * @return sub node index
     */
    private int subNodeIndex(int elementIndex)
    {
      if (elementIndex == numValues) {
        return subNodes.length - 1;
      }
      int minSubIndex = 0;
      int maxSubIndex = subNodes.length;
      while (maxSubIndex > minSubIndex - 1)  {
        final int midSubIndex = (maxSubIndex + minSubIndex) / 2;
        final SubNode sub = subNodes[midSubIndex];
        if (elementIndex < sub.offset) {
          maxSubIndex = midSubIndex;
        }
        else if (elementIndex >= sub.offset + sub.sub.size()) {
          minSubIndex = midSubIndex + 1;
        }
        else {
          return midSubIndex;
        }
      }
      return minSubIndex;
    }

    @NotNull
    @Override
    public ExpandableDoubleIndexableNode remove(int index)
    {
      if (index < 0  ||  index >= numValues) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d]!", index, numValues));

      }
      final int subNodeIndex = subNodeIndex(index);
      final SubNode subNode = subNodes[subNodeIndex];
      final ExpandableDoubleIndexableNode newSubNode = subNode.sub.remove(index - subNode.offset);
      if (newSubNode.isEmpty()) {
        if (subNodes.length == 1) {
          return ExpandableDoubleIndexableImpl.EMPTY;
        }
        final ExpandableDoubleIndexableNode[] newNodes = new ExpandableDoubleIndexableNode[subNodes.length - 1];
        for (int i = 0;  i < subNodeIndex;  ++i) {
          newNodes[i] = subNodes[i].sub;
        }
        for (int i = subNodeIndex;  i < newNodes.length;  ++i) {
          newNodes[i] = subNodes[i+1].sub;
        }
        return new ExpandableDoubleIndexableImpl.Node(newNodes);
      }

      final ExpandableDoubleIndexableNode[] newNodes = new ExpandableDoubleIndexableNode[subNodes.length];
      for (int i = 0;  i < subNodeIndex;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      newNodes[subNodeIndex] = newSubNode;
      for (int i = subNodeIndex + 1;  i < newNodes.length;  ++i) {
        newNodes[i] = subNodes[i].sub;
      }
      return new ExpandableDoubleIndexableImpl.Node(newNodes);
    }

    @NotNull
    @Override
    public ExpandableDoubleIndexable getCopy()
    {
      return this;
    }

    @Override
    public int size()
    {
      return numValues;
    }

    @Override
    public int getWeight()
    {
      return subNodes.length;
    }

    @Override
    public int depth()
    {
      int maxDepth = 0;
      for (SubNode subNode : subNodes) {
        maxDepth = Math.max(maxDepth, subNode.sub.depth());
      }
      return maxDepth + 1;
    }

    @Override
    public double get(int index)
    {
      if (index < 0  ||  index >= numValues) {
        throw new IndexOutOfBoundsException(String.format("Index %d is out of range [0,%d[!", index, numValues));

      }
      final SubNode subNode = subNodes[subNodeIndex(index)];
      return subNode.sub.get(index - subNode.offset);
    }

    @NotNull
    @Override
    public Iterator<Double> iterator()
    {
      return doubleIterator();
    }

    @NotNull
    @Override
    public PrimitiveIterator.OfDouble doubleIterator(int from, int to)
    {
      if (subNodes.length == 1) {
        return subNodes[0].sub.doubleIterator(from, to);
      }
      final int fromSubIndex = subNodeIndex(from);
      final int toSubIndex = subNodeIndex(to);

      final SubNode subNode = subNodes[fromSubIndex];
      if (fromSubIndex == toSubIndex) {
        return subNode.sub.doubleIterator(from - subNode.offset, to - subNode.offset);
      }
      final Collection<PrimitiveIterator.OfDouble> subIterators = new ArrayList<>(toSubIndex - fromSubIndex + 1);
      subIterators.add(subNode.sub.doubleIterator(from - subNode.offset, subNode.sub.size()));
      for (int i = fromSubIndex + 1;  i < toSubIndex;  ++i) {
        final SubNode node = subNodes[i];
        subIterators.add(node.sub.doubleIterator());
      }
      final SubNode sub = subNodes[toSubIndex];
      subIterators.add(sub.sub.doubleIterator(0, to - sub.offset)) ;

      final Iterator<PrimitiveIterator.OfDouble> doubleIteratorIterator = subIterators.iterator();
      return new PrimitiveIterator.OfDouble()
      {
        private OfDouble doubleIterator = doubleIteratorIterator.hasNext()
                ? doubleIteratorIterator.next()
                : null;

        @Override
        public double nextDouble()
        {
          if (doubleIterator == null) {
            throw new NoSuchElementException("Cannot iterate past the end!");
          }
          final double value = doubleIterator.nextDouble();
          while (!doubleIterator.hasNext()) {
            if (doubleIteratorIterator.hasNext()) {
              doubleIterator = doubleIteratorIterator.next();
            }
            else {
              doubleIterator = null;
              break;
            }
          }
          return value;
        }

        @Override
        public boolean hasNext()
        {
          return doubleIterator != null;
        }
      };
    }

    @Override
    public int addToArray(@NotNull double[] array, int pos)
    {
      for (SubNode subNode : subNodes) {
        pos = subNode.sub.addToArray(array, pos);
      }
      return pos;
    }

    @Override
    public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
    {
      final int start = index;
      final int end   = index + length;
      final int startSubIndex = subNodeIndex(start);
      final int endSubIndex   = subNodeIndex(end);
      if (startSubIndex == endSubIndex) {
        final SubNode node = subNodes[startSubIndex];
        return node.sub.addToArray(array, arrayPos, index - node.offset, length);
      }
      final SubNode startNode = subNodes[startSubIndex];
      final int startIndex = index - startNode.offset;
      arrayPos = startNode.sub.addToArray(array, arrayPos, startIndex, startNode.sub.size() - startIndex);
      for (int i = startSubIndex + 1;  i < endSubIndex;  ++i) {
        final SubNode node = subNodes[i];
        arrayPos = node.sub.addToArray(array, arrayPos);
      }
      final SubNode endNode = subNodes[endSubIndex];
      return endNode.sub.addToArray(array, arrayPos, 0, end - endNode.offset);
    }

    @Override
    public boolean isFilled()
    {
      if (subNodes.length < PART_SIZE) {
        return true;
      }
      for (SubNode subNode : subNodes) {
        if (!subNode.sub.isFilled()) {
          return false;
        }
      }
      return true;
    }

    @Override
    public long getCapacity()
    {
      return capacity;
    }

    @Override
    public String toString()
    {
      return DoubleIndexable.toString(this);
    }
  }

  /**
   * Create an expandable double indexable node from the given values.
   * @param values double array with values
   * @param start start index of values from array
   * @param length number of elements to take from array
   * @return expandable double indexable
   */
  @NotNull
  static ExpandableDoubleIndexableNode from(double[] values, int start, int length)
  {
    if (length <= PART_SIZE) {
      return new Leaf(values, start, length);
    }
    int nextSize = PART_SIZE * PART_SIZE;
    int size = PART_SIZE;
    while (nextSize < length) {
      size = nextSize;
      nextSize = Integer.MAX_VALUE / size < PART_SIZE
              ? size * PART_SIZE
              : Integer.MAX_VALUE;
    }
    final ExpandableDoubleIndexableNode[] nodes = new ExpandableDoubleIndexableNode[(length + size - 1) / size];
    int nodeIndex = 0;
    int pos;
    final int boundary = length - size;
    for (pos = 0;  pos < boundary  &&  pos >= 0;  pos += size) { // >=0 is for very large arrays
      nodes[nodeIndex++] = from(values, start + pos, size);
    }
    nodes[nodeIndex] = from(values, start + pos, length - pos);
    return new Node(nodes);
  }

  /**
   * Create an expandable double indexable node from the given values.
   * @param values double array with values
   * @return expandable double indexable
   */
  @NotNull
  static ExpandableDoubleIndexableNode from(double ... values)
  {
    final int length = values.length;
    if (length == 0) {
      return EMPTY;
    }
    if (length <= PART_SIZE) {
      return new Leaf(values);
    }
    else {
      return from(values, 0, length);
    }
  }

  /**
   * Create an expandable ouble indexable node from a given indexable.
   * @param indexable indexable defining the values of the returned indexable
   * @return expandable double indexable
   */
  @NotNull
  static ExpandableDoubleIndexableNode from(@NotNull DoubleIndexable indexable)
  {
    if (indexable.isEmpty()) {
      return EMPTY;
    }
    final int length = indexable.size();
    if (length <= PART_SIZE) {
      return new Leaf(indexable.toArray());
    }
    int nextSize = PART_SIZE * PART_SIZE;
    int size = PART_SIZE;
    while (nextSize < length) {
      size = nextSize;
      nextSize = Integer.MAX_VALUE / size < PART_SIZE
              ? size * PART_SIZE
              : Integer.MAX_VALUE;
    }
    final ExpandableDoubleIndexableNode[] nodes = new ExpandableDoubleIndexableNode[(length + size - 1) / size];
    int nodeIndex = 0;
    int pos;
    final int boundary = length - size;
    for (pos = 0;  pos >= 0   &&  pos < boundary;  pos += size) { // >=0 is for very large arrays
      nodes[nodeIndex++] = from(indexable.subSet(pos, pos + size));
    }
    nodes[nodeIndex] = from(indexable.tailSet(pos));
    return new Node(nodes);
  }
}
