// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.IntToDoubleFunction;

/**
 * Immutable expandable primitive double indexable.
 * <p>
 * This interface defines a double indexable which allows
 * to change (set, add, remove) its values but is immutable
 * nevertheless. All changing methods return copies, thus
 * guaranteeing immutability.
 * <p>
 * The internal implementation tries to keep copying a cheap
 * operation, but some access patterns may result in performance
 * degration. It is therefore recommended not to use this in
 * time-critical code.
 * <p>
 * Initial creation is done by using the following methods:
 * <ul>
 *   <li>{@link #empty()}</li>
 *   <li>{@link #zeroed(int)}</li>
 *   <li>{@link #from(double...)}</li>
 *   <li>{@link #from(double[], int, int)}</li>
 *   <li>{@link #from(DoubleIndexable)}</li>
 *   <li>{@link #from(Indexable)}</li>
 *   <li>{@link #from(Collection)}</li>
 *   <li>{@link #fromNumbers(Number[])}</li>
 * </ul>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 24, 2020
 */
public interface ExpandableDoubleIndexable
        extends DoubleIndexable
{
  /**
   * Return an exoandable double indexable Insert where the the given value
   * was inserted at the given index.
   * @param index index in the range {@code 0} (insert at first position)
   *              to {@code size()} (insert after last position)
   * @param value value to insert at the given position
   * @return new double indexable with the given value inserted
   */
  @NotNull
  ExpandableDoubleIndexable add(int index, double value);

  /**
   * Return an expandable double indexable where the value at the given index
   * was removed.
   * @param index index to remove, in the range {@code 0} to
   *              {@code size() - 1}
   * @return new expandable double indexable with the given index removed
   */
  @NotNull
  ExpandableDoubleIndexable remove(int index);

  /**
   * Get an expandable copy of this double indexable.
   * As this is basically immutable extending classes are expected to return {@code this}.
   * @return independent copy of this indexable
   */
  @NotNull
  ExpandableDoubleIndexable getCopy();

  /**
   * Return an expandable double indexable where the given
   * value was added to the end of this indexable.
   * @param value value to add
   * @return new double indexable with added value
   */
  @NotNull
  default ExpandableDoubleIndexable add(double value)
  {
    return add(size(), value);
  }

  /**
   * Pythonesque insertion allowing negative indexes to insert from the end.
   * @param index positive values behave the same way as {@link #add(int, double)},
   *              but negative values count from the back, so {@code -1}
   *              will insert before the last element
   * @param value value to insert
   * @return new double indexable with the given value inserted
   */
  @NotNull
  default ExpandableDoubleIndexable ydd(int index, double value)
  {
    if (index >= 0) {
      return add(index, value);
    }
    else {
      final int realIndex = index + size();
      if (realIndex < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           index, size()));
      }
      return add(realIndex, value);
    }
  }

  /**
   * Pythonesque remove allowing negative indexes to remove from the end.
   * @param index positive values behave the same as {@link #remove(int)},
   *              while negative indexes count from the back, so {@code -1}
   *              will remove the last element
   * @return new expandable double indexable with the given index removed
   */
  @NotNull
  default ExpandableDoubleIndexable remyve(int index)
  {
    if (index >= 0) {
      return remove(index);
    }
    else {
      final int realIndex = index + size();
      if (realIndex < 0) {
        throw  new IndexOutOfBoundsException(String.format("Cannot access index %d with %d elements!",
                                                           index, size()));
      }
      return remove(realIndex);
    }
  }

  /**
   * Convenience method to create an empty expandable int indexable.
   * @return empty expandable int indexable
   */
  @NotNull
  static ExpandableDoubleIndexable empty()
  {
    return from();
  }

  /**
   * Create an expandable int indexable of the given size initialized with {@code 0.0}.
   * @param size size of returned double indexable
   * @return expandable double indexable of the given size
   */
  @NotNull
  static ExpandableDoubleIndexable zeroed(int size)
  {
    return from(new DoubleIndexable.Base()
    {
      @Override
      public int size()
      {
        return size;
      }

      @Override
      public double get(int index)
      {
        return 0;
      }

      @Override
      public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
      {
        Arrays.fill(array, arrayPos, arrayPos + length, 0.0);
        return arrayPos + length;
      }
    });
  }

  /**
   * Create an expandable double indexable with a given size, and initialize its elements by index.
   * @param size     size of the indexable
   * @param producer element producer which will be called for each index
   * @return expandable double indexable with the given size containing the elements created by the producer
   */
  @NotNull
  static DoubleIndexable initByIndex(int size, @NotNull IntToDoubleFunction producer)
  {
    return from(DoubleIndexable.viewByIndex(size, producer));
  }

  /**
   * Create an expandable integer indexable from an initial array of values.
   * @param values array of values
   * @return expandable integer indexable
   */
  @NotNull
  static ExpandableDoubleIndexable from(@NotNull double ... values)
  {
    return ExpandableDoubleIndexableImpl.from(values);
  }

  /**
   * Create an expandable integer indexable from part of an int array.
   * @param array    integer array
   * @param start    start position in array
   * @param length   number of byte in array
   * @return expandable integer indexable
   */
  @NotNull
  static ExpandableDoubleIndexable from(@NotNull double[] array, int start, int length)
  {
    return ExpandableDoubleIndexableImpl.from(array, start, length);
  }

  /**
   * Get an expandable integer indexable from a normal
   * integer indexable.
   * @param indexable  standard integer indexable
   * @return expandable copy of the given indexable
   */
  @NotNull
  static ExpandableDoubleIndexable from(@NotNull DoubleIndexable indexable)
  {
    return ExpandableDoubleIndexableImpl.from(indexable);
  }

  /**
   * Get an expandable integer indexable from an array of numbers.
   * @param values number values
   * @param <N>    number type, prefarably {@link Integer}
   * @return expandable int indexable
   */
  @NotNull
  @SafeVarargs // because read only
  @SuppressWarnings("varargs")
  static <N extends Number> ExpandableDoubleIndexable fromNumbers(@NotNull N... values)
  {
    return from(DoubleIndexable.viewNumberArray(values));
  }

  /**
   * Get an expandable integer indexable from an indexable of numbers.
   * @param indexable numbers indexable
   * @return expandable int indexable
   */
  @NotNull
  static ExpandableDoubleIndexable from(@NotNull Indexable<? extends Number> indexable)
  {
    return from(DoubleIndexable.viewIndexable(indexable));
  }

  /**
   * Create an expandable integer indexable from a collection of numbers.
   *
   * @param values collection of numbers, {@link List} is strongly
   *               recommended for better performance
   * @return expandable integer indexable
   */
  @NotNull
  @SuppressWarnings("unchecked")
  static ExpandableDoubleIndexable from(@NotNull Collection<? extends Number> values)
  {
    if (values instanceof List) {
      return from(Indexable.viewList((List<? extends Number>)values));
    }
    // need fast access, so no way around converting
    final Number[] numbers = values.toArray(Empty.NUMBER_ARRAY);
    return from(DoubleIndexable.viewNumberArray(numbers));
  }
}
