// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Enumeration;
import java.util.Iterator;

/**
 * Iterator wrapper for an enumeration.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see de.caff.generics.Types#toIterator(java.util.Enumeration)
 */
public class EnumerationIterator<T>
        implements Iterator<T>
{
  @NotNull
  private final Enumeration<T> enumeration;

  /**
   * Constructor.
   * @param enumeration enumeration to wrap
   */
  public EnumerationIterator(@NotNull Enumeration<T> enumeration)
  {
    this.enumeration = enumeration;
  }

  /**
   * Returns <tt>true</tt> if the iteration has more elements. (In other
   * words, returns <tt>true</tt> if <tt>next</tt> would return an element
   * rather than throwing an exception.)
   *
   * @return <tt>true</tt> if the iterator has more elements.
   */
  @Override
  public boolean hasNext()
  {
    return enumeration.hasMoreElements();
  }

  /**
   * Returns the next element in the iteration.  Calling this method
   * repeatedly until the {@link #hasNext()} method returns false will
   * return each element in the underlying collection exactly once.
   *
   * @return the next element in the iteration.
   * @throws java.util.NoSuchElementException
   *          iteration has no more elements.
   */
  @Override
  public T next()
  {
    return enumeration.nextElement();
  }

  /**
   * Removes from the underlying collection the last element returned by the
   * iterator (optional operation).  This method can be called only once per
   * call to <tt>next</tt>.  The behavior of an iterator is unspecified if
   * the underlying collection is modified while the iteration is in
   * progress in any way other than by calling this method.
   *
   * @throws UnsupportedOperationException if the <tt>remove</tt>
   *                                       operation is not supported by this Iterator.
   * @throws IllegalStateException         if the <tt>next</tt> method has not
   *                                       yet been called, or the <tt>remove</tt> method has already
   *                                       been called after the last call to the <tt>next</tt>
   *                                       method.
   */
  @Override
  public void remove()
  {
    throw new UnsupportedOperationException("No remove for wrapped enumeration!");
  }
}
