// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.DoubleFunction2;
import de.caff.generics.function.DoubleOperator2;
import de.caff.generics.function.DoublePredicate2;
import de.caff.generics.tuple.Tuple;
import de.caff.generics.tuple.Tuple2;

import java.io.Serializable;
import java.util.Locale;
import java.util.Objects;

import static de.caff.generics.Primitives.areEqual;
import static de.caff.generics.Primitives.boxed;

/**
 * A pair of primitive double values.
 * <p>
 * This is similar to {@link Pair}, but for raw {@code double} values.
 * As these values are defined final, this class is also immutable.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 04, 2020
 */
public final class DoublePair
        implements Serializable,
                   DoubleIndexable
{
  private static final long serialVersionUID = 2437672519488765631L;
  /** The first value of this pair. */
  public final double first;
  /** The second value of this pair. */
  public final double second;

  /**
   * Constructor.
   * @param first    first value of pair
   * @param second   second value of pair
   */
  public DoublePair(double first, double second)
  {
    this.first = first;
    this.second = second;
  }

  /**
   * Get this pair with a different first value.
   * @param first new first value
   * @return pair with the given {@code first} value,
   *         and the {@link #second} value of this pair
   */
  @NotNull
  public DoublePair withFirst(double first)
  {
    if (areEqual(this.first, first)) {
      return this;
    }
    return new DoublePair(first, this.second);
  }

  /**
   * Get this pair with a different first value.
   * @param second new second value
   * @return pair with the {@link #first} value of this pair,
   *         and the given {@code second} value
   */
  @NotNull
  public DoublePair withSecond(double second)
  {
    if (areEqual(this.second, second)) {
      return this;
    }
    return new DoublePair(this.first, second);
  }

  /**
   * Get this pair with swapped values.
   * @return pair with swapped {@link #first} and {@link #second} values
   */
  @NotNull
  public DoublePair swapped()
  {
    if (areEqual(first, second)) {
      return this;
    }
    return new DoublePair(second, first);
  }

  @Override
  public int size()
  {
    return 2;
  }

  @Override
  public double get(int index)
  {
    switch (index) {
    case 0:
      return first;
    case 1:
      return second;
    default:
      throw new IndexOutOfBoundsException(String.format("Pair has only 2 elements, so %d is out of bounds!", index));
    }
  }

  /**
   * Apply a 2-double function on this pair.
   * @param function function to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @param <T> output type
   * @return result of applying the function
   */
  public <T> T applyAsDoubles(@NotNull DoubleFunction2<T> function)
  {
    return function.applyAsDouble(first, second);
  }

  /**
   * Call an 2-double operator on this pair.
   * @param operator operator to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @return result of applying the operator
   */
  public double operate(@NotNull DoubleOperator2 operator)
  {
    return operator.applyAsDouble(first, second);
  }

  /**
   * Call a 2-double predicate on this pair. 
   * @param predicate predicate to apply, {@link #first} becomes its first, {@link #second} its second argument 
   * @return result of testing this pair
   */
  public boolean test(@NotNull DoublePredicate2 predicate)
  {
    return predicate.testDoubles(first, second);
  }

  /**
   * Convert this into a tuple.
   * @return tuple created from this pair
   */
  @NotNull
  public Tuple2<Double, Double> toTuple()
  {
    return Tuple.of(first, second);
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final DoublePair that = (DoublePair)o;
    return areEqual(that.first, first) &&
           areEqual(that.second, second);
  }

  @Override
  public int hashCode()
  {
    return Objects.hash(boxed(first), boxed(second));
  }

  @Override
  @NotNull
  public String toString()
  {
    return String.format(Locale.US, "DoublePair(%s, %s)", first, second);
  }
}
