// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.*;
import de.caff.generics.range.Range;

import java.util.*;
import java.util.function.*;

import static de.caff.generics.Order.Ascending;
import static de.caff.generics.Order.Descending;

/**
 * Class which allows readonly access of raw doubles by index.
 * <p>
 * The {@link #EMPTY} constant or {@link #emptyIndexable()} method provide the same
 * useful return value to indicate emptiness.
 * <p>
 * Simple implementations should extend {@link Base} because
 * that provides useful implementations for standard Object methods and implements
 * {@link Comparable}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 7, 2020
 * @see de.caff.generics.algorithm.FastDoubleSequenceSearch
 */
public interface DoubleIndexable
        extends PrimitiveDoubleIterable,
                Sizeable
{
  /** Double indexable which only contains the single value {@code 0.0}. */
  DoubleIndexable SINGLE_0 = singleton(0.0);

  /**
   * Get the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @return element at the given index
   */
  double get(int index);

  /**
   * Pythonesque get.
   * This allows accessing elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @return element at the given index
   */
  default double gyt(int index)
  {
    return get(Pythonesque.mapX(index, this));
  }

  /**
   * Get an element modulo size.
   * <p>
   * This maps the given {@code index} into the range of this indexable
   * by applying a modulo {@link #size()} operation. For empty indexable
   * this will throw an {@link IndexOutOfBoundsException} as there is no
   * possible index to get.
   * @param index index, possibly out of range, possibly even negative
   *              for Pythonesque access
   * @return element at the given, possibly modulated, index
   */
  default double getMod(int index)
  {
    final int size = size();
    if (size == 0) {
      throw new IndexOutOfBoundsException("No element for empty indexable!");
    }
    return gyt(index % size);
  }

  /**
   * Get this indexable but with inverted order.
   * @return a view to this indexable which accesses the elements in reverse order
   */
  @NotNull
  default Base reverse()
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }

      @Override
      public double get(int index)
      {
        return DoubleIndexable.this.get(DoubleIndexable.this.size() - index - 1);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return DoubleIndexable.this.asBase();
      }
    };
  }

  /**
   * Is this indexable empty?
   * @return {@code true}: there are no elements in this indexable<br>
   *         {@code false}: this indexable has elements
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * Get an primitive double iterator.
   * @return double iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveIterator.OfDouble doubleIterator()
  {
    return doubleIterator(0, size());
  }

  /**
   * Get a double iterator from the given sub set.
   * @param from first index of iterator
   * @param to   one past last index of iterator
   * @return int iterator which iterates over the given part of this indexable
   */
  @NotNull
  default PrimitiveIterator.OfDouble doubleIterator(final int from, final int to)
  {
    return new PrimitiveIterator.OfDouble()
    {
      private int index = from;

      @Override
      public double nextDouble()
      {
        if (index >= to) {
          throw new NoSuchElementException(String.format("Index %s out of allowed range [%d, %d[!",
                                                         index, from, to));
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < to;
      }
    };
  }

  /**
   * Returns an iterator over elements of type {@code T}.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  default Iterator<Double> iterator()
  {
    return listIterator();
  }

  /**
   * Get the first value in this double indexable.
   * @return first value
   */
  default double firstValue()
  {
    return get(0);
  }

  /**
   * Get the last value in this double indexable.
   * @return last value
   */
  default double lastValue()
  {
    return gyt(-1);
  }

  /**
   * Returns a list iterator over elements of type {@code T}.
   *
   * @return a list iterator.
   */
  @NotNull
  default ListIterator<Double> listIterator()
  {
    return new ListIterator<Double>() {
      private int index = 0;

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public Double next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public Double previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(--index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(Double t)
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void add(Double t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  /**
   * Get an indexable subset.
   * The subset includes indices {@code fromIndex} to {@code toIndex - 1}.
   * @param fromIndex start index of sub set
   * @param toIndex   index after last index
   * @return indxable subset view of this indexable
   */
  @NotNull
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0)
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    if (toIndex > size())
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    if (fromIndex > toIndex)
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    final int length = toIndex - fromIndex;
    return length == 0
            ? EMPTY
            : new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public double get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException("No such element: "+index);
        }
        return DoubleIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return DoubleIndexable.this.subSet(fromIndex + fromIdx,
                                           fromIndex + toIdx);
      }
    };
  }

  /**
   * Get an indexable subset.
   * This is the Pythonesque version which allows negative indexes.
   * @param fromIndex start index of sub set
   * @param toIndex   end index of sub set
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param fromIndex index to start with, negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param toIndex index one after the end (equal to the length of the returned set),
   *                negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Get a view on this indexable as an unmodifiable collection.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   */
  @NotNull
  default Collection<Double> asCollection()
  {
    return new AbstractCollection<Double>()
    {
      @NotNull
      @Override
      public Iterator<Double> iterator()
      {
        return DoubleIndexable.this.iterator();
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable list.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   * @see #toList()
   */
  @NotNull
  default List<Double> asList()
  {
    return new AbstractList<Double>()
    {
      @Override
      public Double get(int index)
      {
        return DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }

      @NotNull
      @Override
      public Iterator<Double> iterator()
      {
        return DoubleIndexable.this.iterator();
      }
    };
  }

  /**
   * Create a list from this double indexable.
   * This creates an independent list to which the elements of this
   * indexable are copied.
   * @return list copy of this indexable
   * @see #asList()
   */
  @NotNull
  default ArrayList<Double> toList()
  {
    final ArrayList<Double> list = new ArrayList<>(size());
    addAllTo(list);
    return list;
  }

  /**
   * Add all elements of this indexable to the given collection.
   * @param collection collection to add to
   */
  default void addAllTo(@NotNull Collection<? super Double> collection)
  {
    for (Double entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Get a view of this indexable as a standard object-based indexable.
   * @return indexable with non-null {@code Double} values
   */
  @NotNull
  default Indexable<Double> asIndexable()
  {
    return new Indexable.Base<Double>()
    {
      @Override
      public Double get(int index)
      {
        return DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * Convert this indexable into a Java array.
   * @return array copy containing all values of this indexable
   */
  @NotNull
  default double[] toArray()
  {
    if (isEmpty()) {
      return Empty.DOUBLE_ARRAY;
    }
    final double[] array = new double[size()];
    addToArray(array, 0);
    return array;
  }

  /**
   * Add the content of this indexable to the given array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull double[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull double[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveIterator.OfDouble it = doubleIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextDouble();
    }
    return arrayPos;
  }

  /**
   * Get the indexes of this indexable as an iterable range.
   * @return iterable indexes
   * @see #intIndexes()
   * @see Range#indexes(int)
   */
  @NotNull
  default Iterable<Integer> indexes()
  {
    return Range.indexes(size());
  }

  /**
   * Get the indexes of this indexable as an integer indexable.
   * @return indexes as integer indexable
   * @see #indexes()
   * @see IntIndexable#rangeFromSize(int)
   */
  @NotNull
  default IntIndexable intIndexes()
  {
    return IntIndexable.rangeFromSize(size());
  }

  /**
   * Go over all values and do a cumulative calculation.
   * <p>
   * Eg calculate the maximum of this indexable use
   * <pre>{@code
   *   double max = indexable.foldLeft(Double.NEGATIVE_INFINITY, Math::max);
   * }</pre>
   *
   * @param initialValue  start value for the operation, will be returned if this iterable is empty
   * @param foldOperation operation applied to each value, will get the accumulated value as its
   *                      first and the current element value as its second argument
   * @return accumulated value, result of applying fold operation to all values of this indexable
   *         iteratively
   */
  default double foldLeft(double initialValue,
                          @NotNull DoubleBinaryOperator foldOperation)
  {
    double result = initialValue;
    for (PrimitiveIterator.OfDouble iterator = doubleIterator();
         iterator.hasNext();  ) {
      result = foldOperation.applyAsDouble(result, iterator.nextDouble());
    }
    return result;
  }

  /**
   * View the elements of this double indexable transformed.
   * @param operator transformer applied before doubles are returned
   * @return double indexable view with transformed elements
   */
  @NotNull
  default Base transformed(@NotNull DoubleOperator1 operator)
  {
    return new Base() {
      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }

      @Override
      public double get(int index)
      {
        return operator.applyAsDouble(DoubleIndexable.this.get(index));
      }
    };
  }

  /**
   * Get the next index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int nextMatch(int startIndex, @NotNull DoublePredicate check)
  {
    final int size = size();
    for (int i = Pythonesque.mapX(startIndex, this);  i < size;  ++i) {
      if (check.test(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the first index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return first index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int firstMatch(@NotNull DoublePredicate check)
  {
    return nextMatch(0, check);
  }

  /**
   * Get the previous index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int previousMatch(int startIndex, @NotNull DoublePredicate check)
  {
    for (int i = Pythonesque.mapX(startIndex, this);  i >= 0;  --i) {
      if (check.test(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the last index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return last index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int lastMatch(@NotNull DoublePredicate check)
  {
    return previousMatch(-1, check);
  }

  /**
   * Get a frozen version of this indexable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying colelction might
   * nevertheless change. This copies the current state of this indexable
   * into an unmodifiable state, and returns an Indexable which is
   * stable in size and will return always the same element for a given index.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, so you can safely call this method more than once.
   *
   * @return frozen version of this Indexable
   */
  @NotNull
  default DoubleIndexable frozen()
  {
    return IndexableHelper.frozenFromArray(toArray());
  }

  /**
   * Get a spliterator on this indexable.
   * The returned spliterator will not have its {@link Spliterator#IMMUTABLE} flag set,
   * see {@link #frozenDoubleSpliterator()} for an alternative.
   * @return a spliterator on this indexable
   */
  @NotNull
  default Spliterator.OfDouble doubleSpliterator()
  {
    return new DoubleIndexable.DoubleIndexableSpliterator(this);
  }

  /**
   * Get an immutable spliterator on a frozen copy of this indexable.
   * If this indexable is already immutable (i.e. if {@link #frozen()} returns {@code this})
   * then this is the same as {@link #spliterator()} with the difference that the spliterator
   * returned here will have its {@link Spliterator#IMMUTABLE} flag set. If this is not immutable
   * a frozen copy will be created, and the spliterator will be operating on that. Please refer to
   * {@link #frozen()} to understand what level of immutabiliy it will provide.
   * @return an "immutable" spliterator with the possible cost of copying this indexable
   */
  @NotNull
  default Spliterator.OfDouble frozenDoubleSpliterator()
  {
    final DoubleIndexable frozen = frozen();
    return new DoubleIndexable.DoubleIndexableSpliterator(frozen,
                                                      0, frozen.size(),
                                                      true);
  }

  /**
   * Spliterator for double indexables.
   */
  class DoubleIndexableSpliterator implements Spliterator.OfDouble
  {
    @NotNull
    private final DoubleIndexable indexable;
    private int index;
    private final int fence;
    private final int character;

    /**
     * Instantiate a mutable spliterator for an double indexable.
     * @param indexable indexable of this spliterator
     */
    public DoubleIndexableSpliterator(@NotNull DoubleIndexable indexable)
    {
      this(indexable, 0, indexable.size(), false);
    }

    /**
     * Instantiate a spliterator for part of an double indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param immutable is the underlying indexable immutable? Use with care!
     */
    public DoubleIndexableSpliterator(@NotNull DoubleIndexable indexable,
                                    int start,
                                    int fence,
                                    boolean immutable)
    {
      this(indexable, start, fence,
           immutable
                   ? IMMUTABLE | ORDERED | SIZED | SUBSIZED
                   : ORDERED | SIZED | SUBSIZED);
    }

    /**
     * Instantiate a spliterator for an double indexable.
     * @param indexable indexable of this spliterator
     * @param start     starting index of iteration
     * @param fence     ending index of iteration, not included
     * @param character characteristics of this spliterator
     */
    private DoubleIndexableSpliterator(@NotNull DoubleIndexable indexable, int start, int fence, int character)
    {
      this.indexable = indexable;
      this.index = start;
      this.fence = fence;
      this.character = character;
    }

    @Override
    public OfDouble trySplit()
    {
      final int here = index;
      final int mid = (here + fence) / 2;
      if (here < mid) {
        index = mid;
        return new DoubleIndexable.DoubleIndexableSpliterator(indexable,
                                                          here, mid,
                                                          character);
      }
      return null;
    }

    @Override
    public boolean tryAdvance(@NotNull DoubleConsumer action)
    {
      if (index < fence) {
        action.accept(indexable.get(index++));
        return true;
      }
      return false;
    }

    @Override
    public long estimateSize()
    {
      return fence - index;
    }

    @Override
    public int characteristics()
    {
      return character;
    }
  }

  /**
   * View this as an indexable of another type.
   * This returns a view of this double indexable where each access
   * invokes the {@code convert} function, but you can use
   * {@link Indexable#frozen()} on it to get a fix independent indexable object.
   *
   * @param convert converter between double and the required type
   * @param <T> required type
   * @return view of this double indexable
   */
  @NotNull
  default <T> Indexable<T> view(@NotNull DoubleFunction<? extends T> convert)
  {
    return new Indexable.Base<T>()
    {
      @Override
      public T get(int index)
      {
        return convert.apply(DoubleIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this double indexable with an inserted value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are inserted.
   * Use {@link ExpandableDoubleIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable might
   * result in a disaster when using the returned indexable!
   *
   * @param index index where the value is inserted before the current index
   *              (<b>not Pythonesque</b>, because it is possible to add an value at {@code index == size()},
   *              and it is expected that most insertion will happen at 0 or at the end)
   * @param value inserted value at that index
   * @return a view of this indexable with 1 more value at the given index,
   *         all values beyond are moved one index forward
   * @see #withAppendedValue(double)
   */
  @NotNull
  default DoubleIndexable withInsertedValueAt(int index, double value)
  {
    final int insertIndex = index;
    final int newSize = size() + 1;
    if (insertIndex == 0) {
      return new DoubleIndexable.Base() {
        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public double get(int index)
        {
          return index == 0
                  ? value
                  : DoubleIndexable.this.get(index - 1);
        }
      };
    }
    if (insertIndex == newSize - 1) {
      return new DoubleIndexable.Base() {

        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public double get(int index)
        {
          return index == newSize - 1
                  ? value
                  : DoubleIndexable.this.get(index);
        }
      };
    }
    return new DoubleIndexable.Base()
    {
      @Override
      public int size()
      {
        return newSize;
      }

      @Override
      public double get(int index)
      {
        if (index == insertIndex) {
          return value;
        }
        return DoubleIndexable.this.get(index < insertIndex
                                                ? index
                                                : index - 1);
      }
    };
  }

  /**
   * Create a view with of this indexable with another value added at the end.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are added.
   * @param value value to add
   * @return indexable view of this indexable, with the given value added at the end,
   *         so size is larger by 1
   * @see #withInsertedValueAt(int, double)
   */
  @NotNull
  default DoubleIndexable withAppendedValue(double value)
  {
    return withInsertedValueAt(size(), value);
  }

  /**
   * Create a view of this double indexable with an exchanged value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are exchanged.
   * Use {@link ExpandableDoubleIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable could
   * lead to unexpected results when using the returned indexable!
   *
   * @param index index where the element is exchanged, hiding the current element.
   *              (Pythonesque)
   * @param value exchanged value at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   */
  @NotNull
  default DoubleIndexable withExchangedValueAt(int index, double value)
  {
    final int insertIndex = Pythonesque.mapX(index, this);
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return index == insertIndex
                ? value
                : DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this indexable with one value removed.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few values are removed.
   *
   * @param index index of the removed value (Pythonesque)
   * @return a view of this indexable where the value at the given index
   *         is removed
   */
  @NotNull
  default DoubleIndexable withRemovedValueAt(int index)
  {
    final int removeIndex = Pythonesque.mapX(index, this);
    if (removeIndex == 0) {
      // remove first
      return tailSet(1);
    }
    final int sz = size() - 1;
    if (removeIndex == sz) {
      // remove last
      return headSet(-1);
    }
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return index < removeIndex
                ? DoubleIndexable.this.get(index)
                : DoubleIndexable.this.get(index + 1);
      }

      @Override
      public int size()
      {
        return sz;
      }
    };
  }

  /**
   * Create a view of this double indexable with two values swapped.
   * Note that this is not efficient, but handy in certain situations
   * where only a few items are swapped.
   *
   * @param index1 index of the first value (Pythonesque)
   * @param index2 index of the second value (Pythonesque)
   * @return a view of this indexable where the first and second value
   *         have exchanged places
   */
  @NotNull
  default DoubleIndexable withSwappedValuesAt(int index1, int index2)
  {
    final int swap1 = Pythonesque.mapX(index1, this);
    final int swap2 = Pythonesque.mapX(index2, this);
    if (swap1 == swap2) {
      return this;
    }
    return new Base()
    {
      @Override
      public double get(int index)
      {
        if (index == swap1) {
          return DoubleIndexable.this.get(swap2);
        }
        if (index == swap2) {
          return DoubleIndexable.this.get(swap1);
        }
        return DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * Get a rotated view of this double indexable.
   * The returned indexable will show this indexable with rotated indexes.
   * A positive {@code steps} will rotate left, i.e. any index will be accessed
   * as if the number of steps is added before extraction (modulo length).
   * E.g. a rotation of {@code -1} will return the last element when index {@code 0} is
   * requested, and the first element for index {@code 1}.
   *
   * @param steps steps to rotate
   * @return rotated view of this indexable, use  {@link #frozen()}
   *         to create an indexable which no longer depends on this one
   */
  @NotNull
  default DoubleIndexable rotated(int steps)
  {
    steps = steps % size();
    if (steps == 0) {
      return this;
    }
    if (steps < 0) {
      steps += size();
    }
    final int rotate = steps;
    return new Base()
    {
      @Override
      public double get(int index)
      {
        return DoubleIndexable.this.get((index + rotate) % size());
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }

      @NotNull
      @Override
      public DoubleIndexable rotated(int steps)
      {
        return DoubleIndexable.this.rotated(rotate + steps);
      }
    };
  }

  /**
   * View this double indexable transformed by an operator.
   * @param op transforming operator
   * @return view of this indexable with transformed values
   */
  @NotNull
  default DoubleIndexable viewOp(@NotNull DoubleUnaryOperator op)
  {
    return DoubleIndexable.viewByIndex(size(),
                                    idx -> op.applyAsDouble(DoubleIndexable.this.get(idx)));
  }

  /**
   * View this double indexable as a float indexable.
   * Note that this will result in a loss of precision and reduction of
   * the numeric range of the values.
   * This will downcast the values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a float indexable view of this indexable
   */
  @NotNull
  default FloatIndexable viewAsFloat()
  {
    return new FloatIndexable.Base()
    {
      @Override
      public float get(int index)
      {
        return (float)DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View this double indexable as a long integer indexable.
   * Note that this will usually result in a loss of precision and
   * reduction of the numeric range of the values.
   * This will cast the double values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a long indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsLong()
  {
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return (long)DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as a long integer indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return long indexable view of this indexable
   */
  @NotNull
  default LongIndexable viewAsLong(@NotNull DoubleToLongFunction convert)
  {
    return new LongIndexable.Base()
    {
      @Override
      public long get(int index)
      {
        return convert.applyAsLong(DoubleIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View this double indexable as a short integer indexable.
   * Note that this will usually result in a loss of precision and
   * reduction of the numeric range of the values.
   * This will cast the float values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return an integer indexable view of this indexable
   */
  @NotNull
  default IntIndexable viewAsInt()
  {
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return (int)DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as an integer indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return integer indexable view of this indexable
   */
  @NotNull
  default IntIndexable viewAsInt(@NotNull DoubleToIntFunction convert)
  {
    return new IntIndexable.Base()
    {
      @Override
      public int get(int index)
      {
        return convert.applyAsInt(DoubleIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View this double indexable as a short integer indexable.
   * Note that this will usually result in a loss of precision and
   * severe reduction of the numeric range of the values.
   * This will cast the float values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a short integer indexable view of this indexable
   */
  @NotNull
  default ShortIndexable viewAsShort()
  {
    return new ShortIndexable.Base()
    {
      @Override
      public short get(int index)
      {
        return (short)DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View this double indexable as a byte integer indexable.
   * Note that this will usually result in a loss of precision and
   * severe reduction of the numeric range.
   * This will cast the float values of this indexable on each request,
   * use {@link #frozen()} on the result to create an independent copy
   * if this seems like a problem to you.
   * @return a byte integer indexable view of this indexable
   */
  @NotNull
  default ByteIndexable viewAsByte()
  {
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        return (byte)DoubleIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View this indexable as a boolean indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return boolean indexable view of this indexable
   */
  @NotNull
  default BooleanIndexable viewAsBoolean(@NotNull DoublePredicate convert)
  {
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return convert.test(DoubleIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }
    };
  }

  /**
   * View the differences between consecutive elements as a new double indexable.
   * This will return an indexable which contains the differences between values
   * of this indexable. E.g. for index {@code 0} the returned indexable will return
   * the difference between the element with index {@code 1} and index {@code 0}
   * of this indexable.
   * @return differences indexable as described, empty if the size of this indexable
   *         is 1 or less. As the returned result is a view, use {@link #frozen()}
   *         if you plan to use it more than once.
   */
  @NotNull
  default DoubleIndexable deltas()
  {
    final int newSize = size() - 1;
    if (newSize <= 0) {
      return emptyIndexable();
    }
    return DoubleIndexable.viewByIndex(newSize,
                                       idx -> DoubleIndexable.this.get(idx + 1) - DoubleIndexable.this.get(idx));
  }

  /**
   * View a list of numbers as a double indexable.
   * This requires a list of non-null numbers as a base.
   * @param list list used as a base, required to have only non-null elements
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public double get(int index)
      {
        return list.get(index).doubleValue();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This allows to provide a fallback for null elements in the list.
   * @param list list used as a base, required to have only non-null elements
   * @param nullValue value returned for {@code null} elements in the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Number> list, double nullValue)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public double get(int index)
      {
        final Number number = list.get(index);
        return number != null
                ? number.doubleValue()
                : nullValue;
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic list as a double indexable.
   * @param list generic list
   * @param extractor extractor function which extracts a double from the elements of {@code list}
   * @param <T> element type of list
   * @return a double indexable view of {@code list}
   */
  @NotNull
  static <T> Base viewList(@NotNull List<T> list,
                           @NotNull ToDoubleFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public double get(int index)
      {
        return extractor.applyAsDouble(list.get(index));
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic Number indexable as double indexable.
   * @param indexable generic indexable with Number elements, required to have only non-null elements
   * @return double indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public double get(int index)
      {
        return indexable.get(index).doubleValue();
      }
    };
  }

  /**
   * View a generic Number indexable as double indexable.
   * @param indexable generic indexable with Number elements
   * @param nullFallback fallback for {@code null} elements
   * @return double indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Number> indexable,
                            double nullFallback)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public double get(int index)
      {
        final Number number = indexable.get(index);
        return number != null
                ? number.doubleValue()
                : nullFallback;
      }
    };
  }

  /**
   * View a generic indexable as a double indexable.
   * @param indexable generic indexable
   * @param extractor extractor function which extracts a double from the elements of {@code indexable}
   * @param <T> element type of indexable
   * @return a double indexable view of {@code indexable}
   */
  @NotNull
  static <T> Base viewIndexable(@NotNull Indexable<T> indexable,
                                @NotNull ToDoubleFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public double get(int index)
      {
        return extractor.applyAsDouble(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean isEmpty()
      {
        return indexable.isEmpty();
      }
    };
  }

  /**
   * View a Number array as an Indexable.
   * To view only part of the array use {@link #subSet(int, int)}.
   * @param array array used as base, required to hold only non-null values
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewNumberArray(@NotNull Number ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public double get(int index)
      {
        return array[index].doubleValue();
      }
    };
  }

  /**
   * View a double array as an Indexable.
   * To view only part of the array use {@link #viewArray(double[], int, int)}.
   * @param array array used as base
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull double ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public double get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * View part of a double array as an Indexable.
   * @param array array used as base
   * @param start index of first byte to use from the array
   * @param length number of elements to use from the array
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull double[] array,
                        int start,
                        int length)
  {
    if (start < 0  ||  length < 0  ||  start + length > array.length) {
      throw new IndexOutOfBoundsException("Start or end outside of range!");
    }
    if (length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public double get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("Index %d outside range [0, %d[!",
                                                            index, length));
        }
        return array[index + start];
      }
    };
  }

  /**
   * View an object array as a double indexable.
   * @param array         viewed array
   * @param valueExtract  converter from array elements to the double values of this indexable
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   * @param <T> array element type
   */
  @NotNull
  static <T> DoubleIndexable.Base viewArray(@NotNull T[] array,
                                            @NotNull ToDoubleFunction<? super T> valueExtract)
  {
    if (array.length == 0) {
      return EMPTY;
    }
    return new DoubleIndexable.Base()
    {
      @Override
      public double get(int index)
      {
        return valueExtract.applyAsDouble(array[index]);
      }

      @Override
      public int size()
      {
        return array.length;
      }
    };
  }

  /**
   * View an array of a different type.
   * @param extractor function used to extract the double value from the different type
   * @param array     array of a different type
   * @param <T>       array type
   * @return indexable view of the given array
   */
  @NotNull
  @SafeVarargs // because only read access is used
  static <T> Base viewArray(@NotNull ToDoubleFunction<? super T> extractor,
                            T... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public double get(int index)
      {
        return extractor.applyAsDouble(array[index]);
      }
    };
  }

  /**
   * Get a double indexable of a given size which always returns the same value.
   * @param size  size of the returned indexable
   * @param value value returned for each element
   * @return indexable of size {@code size} with always the same element
   */
  @NotNull
  static Base init(int size, double value)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new Base()
    {
      @Override
      public double get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return value;
      }

      @Override
      public int size()
      {
        return 0;
      }

      @NotNull
      @Override
      public DoubleIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a double indexable of a given size which returns elements created by index.
   * This will always call the {@code producer} when an element is requested.
   * Use {@link #frozen()} on the result to create an indexable which avoids this.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable view which calls the {@code producer} with the given index when an element is requested
   * @see #initByIndex(int, IntToDoubleFunction)
   */
  @NotNull
  static Base viewByIndex(int size, @NotNull IntToDoubleFunction producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new Base()
    {
      @Override
      public double get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return producer.applyAsDouble(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get a double indexable of a given size which returns elements created by index.
   * This will call the producer only once for each element during the call of this method,
   * and return the results on later requests.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable with elements initialized by the {@code producer}
   * @see #viewByIndex(int, IntToDoubleFunction)
   */
  @NotNull
  static DoubleIndexable initByIndex(int size, @NotNull IntToDoubleFunction producer)
  {
    return viewByIndex(size, producer).frozen();
  }

  /**
   * View a single double value as a double indexable of size 1.
   * @param value single value
   * @return indexable with one value
   */
  @NotNull
  static Base singleton(double value)
  {
    return new DoubleIndexable.Base() {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public double get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException(String.format("Index %d for indexable of size 1!", index));
        }
        return value;
      }

      @Override
      public double firstValue()
      {
        return value;
      }

      @Override
      public double lastValue()
      {
        return value;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return this;
      }

      @NotNull
      @Override
      public DoubleIndexable rotated(int steps)
      {
        return this;
      }

      @NotNull
      @Override
      public Collection<Double> asCollection()
      {
        return Collections.singleton(value);
      }

      @NotNull
      @Override
      public List<Double> asList()
      {
        return Collections.singletonList(value);
      }

      @NotNull
      @Override
      public double[] toArray()
      {
        return new double[] { value };
      }

      @Override
      public int addToArray(@NotNull double[] array, int pos)
      {
        array[pos] = value;
        return pos + 1;
      }

      @NotNull
      @Override
      public Iterable<Integer> indexes()
      {
        return IntIndexable.SINGLE_0;
      }

      @NotNull
      @Override
      public IntIndexable intIndexes()
      {
        return IntIndexable.SINGLE_0;
      }

      @Override
      public double sum()
      {
        return value;
      }

      @NotNull
      @Override
      public OptionalDouble average()
      {
        return OptionalDouble.of(value);
      }

      @Override
      public boolean isOrdered(@NotNull DoubleOrdering order)
      {
        return true;
      }

      @Override
      public boolean isStrictlyOrdered(@NotNull DoubleOrdering order)
      {
        return true;
      }
    };
  }

  /**
   * An empty indexable.
   */
  @NotNull
  Base EMPTY = new Base()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public double get(int index)
    {
      throw new IndexOutOfBoundsException("Empty indexable has no elements!");
    }

    @Override
    public double firstValue()
    {
      throw new IndexOutOfBoundsException("Empty indexable has no first value!");
    }

    @Override
    public double lastValue()
    {
      throw new IndexOutOfBoundsException("Empty indexable has no last value!");
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public double[] toArray()
    {
      return Empty.DOUBLE_ARRAY;
    }

    @Override
    public int addToArray(@NotNull double[] array, int pos)
    {
      return pos;
    }

    @Override
    public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
    {
      return arrayPos;
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @Override
    public double foldLeft(double initialValue, @NotNull DoubleBinaryOperator foldOperation)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public Base reverse()
    {
      return this;
    }

    @NotNull
    @Override
    public DoubleIndexable rotated(int steps)
    {
      return this;
    }

    @NotNull
    @Override
    public Iterator<Double> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public ListIterator<Double> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public PrimitiveIterator.OfDouble doubleIterator()
    {
      return Types.EMPTY_DOUBLE_ITERATOR;
    }

    @Override
    public int compareTo(@NotNull DoubleIndexable o)
    {
      return o.isEmpty() ? 0 : -1;
    }

    @Override
    public String toString()
    {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }

    @Override
    public int hashCode()
    {
      return 1;
    }

    @Override
    public boolean equals(Object obj)
    {
      return (obj instanceof DoubleIndexable  &&  ((DoubleIndexable)obj).isEmpty());
    }

    @NotNull
    @Override
    public Collection<Double> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Double> asList()
    {
      return Collections.emptyList();
    }

    @Override
    public void forEachDouble(@NotNull DoubleConsumer consumer)
    {
    }

    @Override
    public <E extends Exception> void forEachDoubleFragile(@NotNull FragileDoubleConsumer<E> action) throws E
    {
    }

    @Override
    public boolean containsDouble(double value)
    {
      return false;
    }

    @Override
    public boolean containsDouble(double value, double eps)
    {
      return false;
    }

    @NotNull
    @Override
    public Base transformed(@NotNull DoubleOperator1 operator)
    {
      return this;
    }

    @NotNull
    @Override
    public Base frozen()
    {
      return this;
    }

    @NotNull
    @Override
    public <T> Indexable<T> view(@NotNull DoubleFunction<? extends T> convert)
    {
      return Indexable.emptyIndexable();
    }

    @Override
    public void forEach(Consumer<? super Double> action)
    {
    }

    @Override
    public boolean isOrdered(@NotNull DoubleOrdering order)
    {
      return true;
    }

    @Override
    public boolean isStrictlyOrdered(@NotNull DoubleOrdering order)
    {
      return true;
    }

    @Override
    public int binarySearch(double value)
    {
      return -1;
    }

    @Override
    public int binarySearch(double value, @NotNull DoubleOrdering order)
    {
      return -1;
    }

    @Override
    public double sum()
    {
      return 0.0;
    }

    @NotNull
    @Override
    public OptionalDouble average()
    {
      return OptionalDouble.empty();
    }

    @NotNull
    @Override
    public Spliterator.OfDouble doubleSpliterator()
    {
      return Spliterators.emptyDoubleSpliterator();
    }
  };

  /**
   * Get an empty indexable set.
   * @return empty indexable
   */
  @NotNull
  static DoubleIndexable.Base emptyIndexable()
  {
    return EMPTY;
  }

  /**
   * Create a string representation of the given indexable.
   * @param indexable indexable
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull DoubleIndexable indexable)
  {
    if (indexable.isEmpty()) {
      return "[]";
    }
    final StringBuilder sb = new StringBuilder();
    sb.append('[').append(indexable.get(0));
    for (PrimitiveIterator.OfDouble it = indexable.tailSet(1).doubleIterator();  it.hasNext();  ) {
      sb.append(',').append(it.nextDouble());
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Are two double indexables equal?
   * <p>
   * This handles NaN values specially so that two of them are considered equal,
   * compare {@link DoublePredicate2#EQUAL_EVEN_NAN}.
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equal(@NotNull DoubleIndexable indexable1,
                       @NotNull DoubleIndexable indexable2)
  {
    return equal(indexable1, indexable2, DoublePredicate2.EQUAL_EVEN_NAN);
  }

  /**
   * Are two double indexables equal?
   * <p>
   * This handles NaN values according to the standard so that two of them are considered non-equal,
   * compare {@link DoublePredicate2#EQUAL_STANDARD}.
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equalStandard(@NotNull DoubleIndexable indexable1,
                               @NotNull DoubleIndexable indexable2)
  {
    return equal(indexable1, indexable2, DoublePredicate2.EQUAL_STANDARD);
  }

  /**
   * Are two double indexables equal?
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @param checker    equality checker, this can be {@link DoublePredicate2#EQUAL_STANDARD} pr
   *                   {@link DoublePredicate2#EQUAL_EVEN_NAN} which handle NaN differently,
   *                   or allow for a small epsilon difference
   * @return {@code true} if both indexables contain the same values as defined by the checker in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equal(@NotNull DoubleIndexable indexable1,
                       @NotNull DoubleIndexable indexable2,
                       @NotNull DoublePredicate2 checker)
  {
    if (indexable1 == indexable2) {
      return true;
    }
    if (indexable1.size() != indexable2.size()) {
      return false;
    }
    final PrimitiveIterator.OfDouble it1 = indexable1.doubleIterator();
    final PrimitiveIterator.OfDouble it2 = indexable2.doubleIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      if (!checker.testDoubles(it1.next(), it2.next())) {
        return false;
      }
    }
    return !(it1.hasNext()  ||  it2.hasNext());
  }

  /**
   * Are two double indexables equal?
   * This is a convenience method which may be called from
   * implementations for their {@link Object#equals(Object)}
   * method.
   *
   * @param indexable1  first indexable
   * @param indexable2  object expected to be an indexable itself
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if the second object is not an indexable, of if sizes or values differ
   */
  static boolean equal(@NotNull DoubleIndexable indexable1,
                       @Nullable Object indexable2)
  {
    if (indexable2 instanceof DoubleIndexable) {
      return equal(indexable1, (DoubleIndexable)indexable2);
    }
    return false;
  }

  /**
   * Compare two double indexables.
   * This compares the two indexables lexically element by element
   * using {@link Double#compareTo(Double)}
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compare(@NotNull DoubleIndexable indexable1,
                     @NotNull DoubleIndexable indexable2)
  {
    final PrimitiveIterator.OfDouble it1 = indexable1.doubleIterator();
    final PrimitiveIterator.OfDouble it2 = indexable2.doubleIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Double.compare(it1.nextDouble(),
                                         it2.nextDouble());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Calculate a hashcode for a double indexable.
   * @param indexable indexable for which the hash code is required
   * @return hash code for the given indexable
   */
  static int hash(@NotNull DoubleIndexable indexable)
  {
    // this follows Arrays.hashCode(), but takes care of +/-0.0 quirks
    int result = 1;

    for (PrimitiveIterator.OfDouble it = indexable.doubleIterator();  it.hasNext();  ) {
      result = 31 * result + Primitives.hash(it.next());
    }

    return result;
  }

  /**
   * Wrap an indexable with one which caches the hash value.
   * This is useful if indexables are used as hash keys because hash value calculation
   * is expensive.
   * <p>
   * The wrapped indexable must not change after it is wrapped, otherwise strange things are expected to happen.
   * @param indexable wrapped indexable, must not change
   * @return indexable which forwards most methods to {@code indexable}, but also provides
   *         useful implementations for {@link Object#hashCode()}, {@link Object#equals(Object)},
   *         and {@link Object#toString()}
   */
  @NotNull
  static DoubleIndexable.Base withCachedHash(@NotNull DoubleIndexable indexable)
  {
    final int hashCode = hash(indexable);
    return new DoubleIndexable.Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public double get(int index)
      {
        return indexable.get(index);
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfDouble doubleIterator()
      {
        return indexable.doubleIterator();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfDouble doubleIterator(int from, int to)
      {
        return indexable.doubleIterator(from, to);
      }

      @NotNull
      @Override
      public Iterator<Double> iterator()
      {
        return indexable.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Double> listIterator()
      {
        return indexable.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return indexable.subSet(fromIndex, toIndex);
      }

      @Override
      public int addToArray(@NotNull double[] array, int pos)
      {
        return indexable.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
      {
        return indexable.addToArray(array, arrayPos, index, length);
      }

      @Override
      public int hashCode()
      {
        return hashCode;
      }

      @Override
      public boolean equals(Object obj)
      {
        return equal(indexable, obj);
      }

      @Override
      public String toString()
      {
        return DoubleIndexable.toString(indexable);
      }
    };
  }

  /**
   * Is this indexable sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * See {@linkplain #isStrictlyOrdered(DoubleOrdering)} for a stricter alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isOrdered(@NotNull DoubleOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    double last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final double value = get(i);
      if (order.checkDouble(value, last) == Descending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered(DoubleOrdering)} for a more relaxed alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered(@NotNull DoubleOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    double last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final double value = get(i);
      if (order.checkDouble(value, last) != Ascending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Do a binary search in an indexable ordered in natural ascending order.
   * This requires this indexable to be ordered in
   * {@link DoubleOrdering#STANDARD_ASCENDING standard Java ascending order},
   * i.e. {@linkplain #isOrdered(DoubleOrdering)} has to return {@code true}
   * for this ordering.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(DoubleOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @return a positive integer defining the index where the given value was found,
   *         and a negative integer if the value is not contained. The latter defines
   *         the insertion point where the looked up value would match into this
   *         sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(double value)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final double dp = get(mid);

      if (dp < value) {
        low = mid + 1;
      }
      else if (dp > value) {
        hi = mid - 1;
      }
      else {
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Do a binary search in an ordered indexable.
   * This requires this indexable to be ordered in non-descending order
   * as defined by the given {@code order},
   * i.e. {@linkplain #isOrdered(DoubleOrdering)} has to return {@code true}
   * for this order.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(DoubleOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @param order sort order
   * @return a positive integer defining the index where the given value was found,
   * and a negative integer if the value is not contained. The latter defines
   * the insertion point where the looked up value would match into this
   * sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(double value, @NotNull DoubleOrdering order)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final double dp = get(mid);

      switch (order.checkDouble(dp, value)) {
      case Ascending:
        low = mid + 1;
        break;
      case Descending:
        hi = mid - 1;
        break;
      default:
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Create a (mutable) double indexable from the values of this indexable
   * which is ordered as defined by the given ordering.
   * <p>
   * If natural ordering is required prefer calling {@link #ordered()}
   * because that typically runs 1.5 times faster.
   * @param order sort order
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableDoubleIndexable ordered(@NotNull DoubleOrdering order)
  {
    final MutableDoubleIndexable result = MutableDoubleIndexable.fromDoubleIndexable(this);
    result.order(order);
    return result;
  }

  /**
   * Create a (mutable) double indexable from the values of this indexable
   * which is ordered in natural order.
   * <p>
   * This can be some 1.5x faster than using {@link #ordered(DoubleOrdering)}
   * with {@link DoubleOrdering#STANDARD_ASCENDING standard Java order}.
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableDoubleIndexable ordered()
  {
    final MutableDoubleIndexable result = MutableDoubleIndexable.fromDoubleIndexable(this);
    result.order();
    return result;
  }

  /**
   * Make this double indexable usable as a base class with defined standard Object methods.
   * @return based version of this indexable
   */
  @NotNull
  default Base asBase()
  {
    return new DoubleIndexable.Base() {
      @Override
      public int size()
      {
        return DoubleIndexable.this.size();
      }

      @Override
      public double get(int index)
      {
        return DoubleIndexable.this.get(index);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return DoubleIndexable.this.reverse();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfDouble doubleIterator()
      {
        return DoubleIndexable.this.doubleIterator();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfDouble doubleIterator(int from, int to)
      {
        return DoubleIndexable.this.doubleIterator(from ,to);
      }

      @NotNull
      @Override
      public Iterator<Double> iterator()
      {
        return DoubleIndexable.this.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Double> listIterator()
      {
        return DoubleIndexable.this.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return DoubleIndexable.this.subSet(fromIndex, toIndex).asBase();
      }

      @NotNull
      @Override
      public double[] toArray()
      {
        return DoubleIndexable.this.toArray();
      }

      @Override
      public int addToArray(@NotNull double[] array, int pos)
      {
        return DoubleIndexable.this.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull double[] array, int arrayPos, int index, int length)
      {
        return DoubleIndexable.this.addToArray(array, arrayPos, index, length);
      }
    };
  }


  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}. It also provides
   * {@link Comparable#compareTo(Object)}.
   * @see DoubleIndexable#asBase()
   */
  abstract class Base implements DoubleIndexable,
                                 Comparable<DoubleIndexable>
  {
    @Override
    public int compareTo(@NotNull DoubleIndexable o)
    {
      Objects.requireNonNull(o);
      return compare(this, o);
    }

    @Override
    public int hashCode()
    {
      return hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      return equal(this, obj);
    }

    @Override
    public String toString()
    {
      return DoubleIndexable.toString(this);
    }

    @NotNull
    @Override
    public Base asBase()
    {
      return this;
    }
  }
}
