// ============================================================================
// File:               Dict
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2021-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            29.09.21 14:07
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.FragileFunction2;
import de.caff.generics.function.FragileProcedure2;
import de.caff.generics.tuple.ITuple2;

import java.util.*;
import java.util.function.*;

/**
 * A dictionary.
 * This is basically the same as {@link java.util.Map}, but optimized for
 * read-only handling and non-zero values.
 * This is defined as an immutable interface, but depending on its usage
 * it might nevertheless change:
 * <ul>
 *   <li>
 *     As this is often used as a view of a {@link java.util.Map} changes of
 *     the underlying map will be reflected. If this is a problem
 *     use {@link #frozen()} to get a decoupled {@code Dict}.
 *   </li>
 *   <li>
 *     It is generally a questionable idea to use mutable objects as map
 *     keys, here or in standard {@link Map}s. Any guarantees are void if a key changes.
 *     Think twice before doing this.
 *   </li>
 *   <li>
 *     Changes to mutable values are also reflected, but this may be exactly
 *     what you want. Otherwise copy the values before you put them
 *     in the underlying map (only if they are still kept outside of the map)
 *     and use {@link #viewMap(Map, Function)} to create the {@code Dict}
 *     with a function which copies the value.
 *     Thus each {@link #get(Object)} will copy the value before returning it,
 *     so the internal value will never change.
 *     If you want to have both freezing and copying the correct calling order
 *     is {@code dict = Dict.viewMap(map).frozen().valueView(copyFunction)}
 *   </li>
 * </ul>
 *
 * Own on-the-fly implementations should extend {@link Base} as it provides some
 * standard object methods. Other implementations might want to make use of
 * {@link #equal(Dict, Object)} in their {@link Object#equals(Object)} method
 * (use {@code Types.hash(entries())} for your {@link Object#hashCode()} implementation
 * then), and {@link #toString(Dict)} in their {@link Object#toString()} method.
 * <p>
 * All methods of this interface which return a {@code Dict} are silently returning
 * a {@code Dict.Base} object.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 29, 2021
 */
public interface Dict<K, V>
{
  /**
   * Empty dictionary implementation.
   * Use {@link #empty()} instead.
   */
  Dict<?,?> EMPTY = new Dict.Base<Object, Object>()
  {
    @Nullable
    @Override
    public Object get(Object key)
    {
      return null;
    }

    @NotNull
    @Override
    public Object getNonNull(Object key)
    {
      throw new NullPointerException("Empty dict has no elements!");
    }

    @NotNull
    @Override
    public Object getOrDefault(Object key, @NotNull Object defaultValue)
    {
      return defaultValue;
    }

    @Nullable
    @Override
    public Object require(Object key)
    {
      throw new NoSuchElementException("Empty dict has no elements!");
    }

    @NotNull
    @Override
    public Countable<Entry<Object, Object>> entries()
    {
      return Countable.empty();
    }

    @Override
    public boolean hasKey(Object key)
    {
      return false;
    }

    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked") // as this is immutable
    public <NV> Dict<Object, NV> valueView(@NotNull Function<? super Object, ? extends NV> valueMapper)
    {
      return (Dict<Object, NV>)this;
    }

    @NotNull
    @Override
    public Countable<Object> values()
    {
      return Countable.empty();
    }

    @NotNull
    @Override
    public Countable<Object> keys()
    {
      return Countable.empty();
    }

    @NotNull
    @Override
    public Dict<Object, Object> frozen()
    {
      return this;
    }

    @Override
    public boolean equals(Object o)
    {
      return o instanceof Dict  &&  ((Dict<?,?>)o).isEmpty();
    }

    @Override
    public int hashCode()
    {
      return Types.EMPTY_HASH;
    }

    @Override
    @NotNull
    public String toString()
    {
      return "{}";
    }
  };

  /**
   * Create a dictionary from a bunch of tuples.
   * It is assumed that {@link ITuple2#_1()} defines the key, and {@link ITuple2#_2()} the value.
   * @param entries dictionary entries defined as 2-tuples
   * @return dictionary created from the given tuples
   * @param <TK> key type
   * @param <TV> value type
   */
  @NotNull
  static <TK, TV> Dict<TK, TV> fromTuples(@NotNull Iterable<? extends ITuple2<? extends TK, ? extends TV>> entries)
  {
    final Map<TK, TV> map = new LinkedHashMap<>();
    entries.forEach(entry -> map.put(entry._1(), entry._2()));
    return viewMap(map);
  }

  /**
   * Get the value for the given key.
   * @param key key
   * @return value associated with key, or {@code null} if there is none
   * @see #getNonNull(Object)
   */
  @Nullable
  V get(K key);

  /**
   * Get the entries of this dictionary.
   * @return entries of this dictionary
   */
  @NotNull
  Countable<Entry<K, V>> entries();

  /**
   * Get the value for the given key.
   * This method will never return {@code null}
   * but throw a {@link NullPointerException} instead.
   * Depending on the underlying mapping {@code null}
   * values might be acceptable, but will result in an exception
   * nevertheless.
   * @param key key
   * @return non-null value associated with key
   * @throws NullPointerException if the return value would have been {@code null}
   * @see #getOrDefault(Object, Object)
   */
  @NotNull
  default V getNonNull(K key)
  {
    return Objects.requireNonNull(get(key));
  }

  /**
   * Get the value for the given key.
   * Return a default if the key is not contained or
   * if the associated value is {@code null}.
   * @param key          key
   * @param defaultValue default value used as fallback
   * @return associated value or {@code defaultValue}
   */
  @NotNull
  default V getOrDefault(K key, @NotNull V defaultValue)
  {
    return Types.notNull(get(key), defaultValue);
  }

  /**
   * Get the value for the given key.
   * If value is not defined, create it with a provider.
   * @param key      key
   * @param provider provider which is called when the key is not contain or the associated value is {@code null},
   *                 called with the key to provide the value which is returned
   * @return associated value or a value created by {@code provider}
   */
  default V getOr(K key, @NotNull Function<? super K, ? extends V> provider)
  {
    final V value = get(key);
    return value != null
            ? value
            : Objects.requireNonNull(provider.apply(key));
  }

  /**
   * Get the value for the given key.
   * Throw a {@link NoSuchElementException} if the key is not contained.
   * @param key key
   * @return associated value, {@code null} if the associated value is {@code null}
   * @throws NoSuchElementException if there is no such key
   */
  @Nullable
  default V require(K key)
  {
    final V result = get(key);
    if (result == null  &&  !hasKey(key)) {
      throw new NoSuchElementException("No element for key "+key);
    }
    return result;
  }

  /**
   * Get the value for the given key, and make sure it is not {@code null}.
   * Throw a {@link NoSuchElementException} if the key is not contained.
   * @param key key
   * @return associated value
   * @throws NoSuchElementException if there is no such key
   * @throws NullPointerException if the key is valid, but the associated value is {@code null}
   */
  @NotNull
  default V requireNonNull(K key)
  {
    final V result = get(key);
    if (result == null) {
      if (!hasKey(key)) {
        throw new NoSuchElementException("No element for key " + key);
      }
      throw new NullPointerException("Value of key " + key + " is null");
    }
    return result;
  }

  /**
   * Get the size of this dictionary.
   * @return size
   */
  default int size()
  {
    return entries().size();
  }

  /**
   * Is this map empty?
   * @return {@code true} if this map has no entries<br>
   *         {@code false} otherwise
   */
  default boolean isEmpty()
  {
    return entries().isEmpty();
  }

  /**
   * Does this dictionary provide a value for the given key?
   * @param key key to check
   * @return {@code true} if the given key is contained in this dictionary<br>
   *         {@code false} if not
   */
  default boolean hasKey(K key)
  {
    return get(key) != null;
  }

  /**
   * Get a view of this dictionary which seems as it has a different value type.
   * @param valueMapper mapper which maps outgoing values toi the expected type.
   *                    The mapper may assume non-{@code null} values.
   * @param <NV> new value type
   * @return view of this dictionary with different value type
   */
  @NotNull
  default <NV> Dict<K, NV> valueView(@NotNull Function<? super V, ? extends NV> valueMapper)
  {
    return new Dict.Base<K, NV>()
    {
      @Nullable
      @Override
      public NV get(@NotNull K key)
      {
        final V v = Dict.this.get(key);
        return v != null
                ? valueMapper.apply(v)
                : null;
      }

      @Override
      public boolean hasKey(@NotNull K key)
      {
        return Dict.this.hasKey(key);
      }

      @NotNull
      @Override
      public Countable<Entry<K, NV>> entries()
      {
        return Dict.this.entries().view(e -> e.valueView(valueMapper));
      }

      @NotNull
      @Override
      public Countable<NV> values()
      {
        return Dict.this.values().view(valueMapper);
      }

      @Override
      public int size()
      {
        return Dict.this.size();
      }

      @Override
      public boolean isEmpty()
      {
        return Dict.this.isEmpty();
      }
    };
  }

  /**
   * Convert this into a {@link Dict.Base} to have useful implementations for
   * some standard Object methods.
   * @return this converted into a Base or this itself if this is already a base
   */
  @NotNull
  default Base<K, V> asBase()
  {
    return new Base<K, V>()
    {
      @Nullable
      @Override
      public V get(K key)
      {
        return Dict.this.get(key);
      }

      @NotNull
      @Override
      public Countable<Entry<K, V>> entries()
      {
        return Dict.this.entries();
      }

      @Override
      public int size()
      {
        return Dict.this.size();
      }

      @Override
      public boolean isEmpty()
      {
        return Dict.this.isEmpty();
      }

      @Override
      public boolean hasKey(K key)
      {
        return Dict.this.hasKey(key);
      }

      @NotNull
      @Override
      public <NV> Dict<K, NV> valueView(@NotNull Function<? super V, ? extends NV> valueMapper)
      {
        return Dict.this.valueView(valueMapper);
      }
    };
  }

  /**
   * Get a frozen version of this dictionary.
   * <p>
   * Often {@code Dict} is used as a wrapper to an underlying {@link Map}.
   * This means that changes to this map will be reflected by the wrapping Dict
   * which usually is counter-intuitive because this interface is immutable.
   * In cases where this might be a problem this method decouples the returned
   * {@code Dict} so it is immutable under all sane circumstances.
   * <p>
   * The implementation takes care to keep the order of {@link #entries()}.
   * @return frozen dictionary, or {@code this} if this is already frozen
   */
  @NotNull
  default Dict<K, V> frozen()
  {
    if (isEmpty()) {
      return empty();
    }
    final LinkedHashMap<K, V> storeMap = new LinkedHashMap<>(size());
    entries().forEach(e -> storeMap.put(e.getKey(), e.getValue()));
    return new Dict.Base<K, V>() {
      @Nullable
      @Override
      public V get(K key)
      {
        return storeMap.get(key);
      }

      @NotNull
      @Override
      public Countable<Entry<K, V>> entries()
      {
        final Set<Map.Entry<K, V>> entries = storeMap.entrySet();
        return Countable.viewCollection(entries, Entry::viewMapEntry);
      }

      @Override
      public int size()
      {
        return storeMap.size();
      }

      @Override
      public boolean isEmpty()
      {
        return storeMap.isEmpty();
      }

      @Override
      public boolean hasKey(K key)
      {
        return storeMap.containsKey(key);
      }

      @NotNull
      @Override
      public Dict<K, V> frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get the values of this dictionary.
   * @return dictionary values
   */
  @NotNull
  default Countable<V> values()
  {
    return entries().view(Entry::getValue);
  }

  /**
   * Get the keys of this dictionary.
   * @return key of this dictionary
   */
  @NotNull
  default Countable<K> keys()
  {
    return entries().view(Entry::getKey);
  }

  /**
   * Get a filtered version of this dictionary.
   * @param check check which decides which entries are kept
   * @return filtered dictionary which only has approved entries
   */
  @NotNull
  default Dict<K, V> filtered(@NotNull Predicate<? super Entry<? super K, ? super V>> check)
  {
    final Map<K, V> filtered = new LinkedHashMap<>();
    entries().forEach(e -> {
      if (check.test(e)) {
        filtered.put(e.getKey(), e.getValue());
      }});
    if (filtered.isEmpty()) {
      return empty();
    }
    // this is only nearly the same as viewMap() because of frozen()
    return new Base<K, V>()
    {
      @Nullable
      @Override
      public V get(K key)
      {
        return filtered.get(key);
      }

      @NotNull
      @Override
      public Countable<Entry<K, V>> entries()
      {
        return Countable.viewCollection(filtered.entrySet(),
                                        Entry::viewMapEntry);
      }

      @Override
      public int size()
      {
        return filtered.size();
      }

      @Override
      public boolean isEmpty()
      {
        return true; // as filtered cannot change and is not empty
      }

      @Override
      public boolean hasKey(K key)
      {
        return filtered.containsKey(key);
      }

      @NotNull
      @Override
      public Dict<K, V> frozen()
      {
        return this;
      }
    };
  }

  /**
   * Call a procedure for each entry.
   * @param consumer procedure called for each entry with key and value
   * @see #forEachEntryFragile(FragileProcedure2)
   */
  default void forEachEntry(@NotNull BiConsumer<? super K, ? super V> consumer)
  {
    entries().forEach(entry -> consumer.accept(entry.getKey(), entry.getValue()));
  }

  /**
   * Call a procedure which might throw an exception for each entry.
   * @param consumer procedure called for each entry with key and value
   * @param <E> exception which might be thrown by {@code consumer}
   * @throws E forwarded exception if {@code consumer} throws
   */
  default <E extends Exception> void forEachEntryFragile(@NotNull FragileProcedure2<E, ? super K, ? super V> consumer)
          throws E
  {
    entries().forEachFragile(entry -> consumer.apply(entry.getKey(), entry.getValue()));
  }

  /**
   * Add all entries in this dictionary to the given map.
   * @param map map to collect the key-value pairs of this dictionary
   */
  default void addAllTo(@NotNull Map<? super K, ? super V> map)
  {
    forEachEntry(map::put);
  }

  /**
   * Call a consumer for a key and its value if the key exists.
   * @param key      key to check
   * @param consumer called with the key and the associated value if the key exists,
   *                 not called if the key is not contained in this dict
   * @return {@code true}: if the key exists and the consumer was called,<br>
   *         {@code false} if the key is not present, and the consumer wasn't called
   */
  default boolean trySupply(@NotNull K key,
                            @NotNull BiConsumer<? super K, ? super V> consumer)
  {
    final V value = get(key);
    if (value == null) {
      // make sure value exists
      if (!hasKey(key)) {
        return false;
      }
    }
    consumer.accept(key, value);
    return true;
  }

  /**
   * Call a fragile consumer for a key and its value if the key exists.
   * @param key      key to check
   * @param consumer called with the key and the associated value if the key exists,
   *                 not called if the key is not contained in this dict
   * @return {@code true}: if the key exists and the consumer was called,<br>
   *         {@code false} if the key is not present, and the consumer wasn't called
   * @param <E> exception type
   * @throws E if consumer throws
   */
  default
  <E extends Exception>
  boolean trySupplyFragile(@NotNull K key,
                           @NotNull FragileProcedure2<E, ? super K, ? super V> consumer)
          throws E
  {
    final V value = get(key);
    if (value == null) {
      // make sure value exists
      if (!hasKey(key)) {
        return false;
      }
    }
    consumer.apply(key, value);
    return true;
  }

  /**
   * Call a consumer if a given key exists, and another one, if not.
   * @param key          key to check
   * @param consumer     consumer called with key and value if the key is present
   * @param elseConsumer consumer called with the key if the key is not present
   */
  default void supplyOrElse(@NotNull K key,
                            @NotNull BiConsumer<? super K, ? super V> consumer,
                            @NotNull Consumer<? super K> elseConsumer)
  {
    final V value = get(key);
    if (value == null) {
      // make sure value exists
      if (!hasKey(key)) {
        elseConsumer.accept(key);
      }
    }
    consumer.accept(key, value);
  }

  /**
   * Call a consumer if a given key exists, and another one, if not.
   * @param key          key to check
   * @param consumer     consumer called with key and value if the key is present
   * @param elseConsumer consumer called with the key if the key is not present
   * @param <E>          exception type
   * @throws E if {@code consumer} throws
   */
  default
  <E extends Exception>
  void supplyFragileOrElse(@NotNull K key,
                           @NotNull FragileProcedure2<E, ? super K, ? super V> consumer,
                           @NotNull Consumer<? super K> elseConsumer)
          throws E
  {
    final V value = get(key);
    if (value == null) {
      // make sure value exists
      if (!hasKey(key)) {
        elseConsumer.accept(key);
      }
    }
    consumer.apply(key, value);
  }

  /**
   * Call a function for the key-value pair if it exists,
   * or a function for the non-existing key.
   * @param key          key to checked
   * @param digester     called if the key exists with key and value
   * @param elseDigester called if the key does not exist, with only the key
   * @return result of either function
   * @param <T> return type
   */
  default <T> T digestOrElse(@NotNull K key,
                             @NotNull BiFunction<? super K, ? super V, ? extends T> digester,
                             @NotNull Function<? super K, ? extends T> elseDigester)
  {
    final V value = get(key);
    if (value == null) {
      // make sure value exists
      if (!hasKey(key)) {
        return elseDigester.apply(key);
      }
    }
    return digester.apply(key, value);
  }

  /**
   * Call a function for the key-value pair if it exists, or return {@code null} if not.
   * @param key          key to checked
   * @param digester     called if the key exists with key and value
   * @return result of the digester if the key-value pair is present, or {@code null} if not
   * @param <T> return type
   */
  @Nullable
  default <T> T digest(@NotNull K key,
                       @NotNull BiFunction<? super K, ? super V, ? extends T> digester)
  {
    return digestOrElse(key, digester, k -> null);
  }

  /**
   * Call a fragile function for the key-value pair if it exists,
   * or a function for the non-existing key.
   * @param key          key to checked
   * @param digester     called if the key exists with key and value
   * @param elseDigester called if the key does not exist, with only the key
   * @return result of either function
   * @param <T> return type
   * @param <E> exception type
   * @throws E if the digester throws
   */
  default <T, E extends Exception> T
  digestFragileOrElse(@NotNull K key,
                      @NotNull FragileFunction2<? extends T, E, ? super K, ? super V> digester,
                      @NotNull Function<? super K, ? extends T> elseDigester)
          throws E
  {
    final V value = get(key);
    if (value == null) {
      // make sure value exists
      if (!hasKey(key)) {
        return elseDigester.apply(key);
      }
    }
    return digester.apply(key, value);
  }

  /**
   * Call a fragile function for the key-value pair if it exists,
   * and return the result; or return null if it does not exist.
   * @param key          key to checked
   * @param digester     called if the key exists with key and value
   * @return result of the digester if the key-value pair is present, or {@code null} if not
   * @param <T> return type
   * @param <E> exception type
   * @throws E if the digester throws
   */
  @Nullable
  default <T, E extends Exception> T
  digestFragile(@NotNull K key,
                @NotNull FragileFunction2<? extends T, E, ? super K, ? super V> digester)
          throws E
  {
    return digestFragileOrElse(key, digester, k -> null);
  }

  /**
   * Get an empty dictionary.
   * @param <KT> key type
   * @param <VT> value type
   * @return empty dictionary
   */
  @SuppressWarnings("unchecked") // as EMPTY is immutatble
  @NotNull
  static <KT, VT> Dict<KT, VT> empty()
  {
    return (Dict.Base<KT, VT>)EMPTY;
  }

  /**
   * Get a dictionary with a single entry.
   * @param key    single key
   * @param value  associated value
   * @param <KT>   key type
   * @param <VT>   value type
   * @return dictionary with single entry
   */
  @NotNull
  static <KT, VT> Dict<KT, VT> singleton(@NotNull KT key,
                                         @NotNull VT value)
  {
    final Entry<KT, VT> entry = Entry.view(key, value);
    return new Base<KT, VT>()
    {
      @Nullable
      @Override
      public VT get(KT k)
      {
        return key.equals(k) ? value : null;
      }

      @NotNull
      @Override
      public Countable<Entry<KT, VT>> entries()
      {
        return Countable.singleton(entry);
      }

      @NotNull
      @Override
      public Countable<VT> values()
      {
        return Countable.singleton(value);
      }

      @NotNull
      @Override
      public Countable<KT> keys()
      {
        return Countable.singleton(key);
      }

      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @Override
      public boolean hasKey(KT k)
      {
        return k.equals(key);
      }

      @NotNull
      @Override
      public Dict<KT, VT> frozen()
      {
        return this;
      }

      @Override
      public String toString()
      {
        return "{" + entry + "}";
      }
    };
  }

  /**
   * Create an read-only view of a standard map as if it is a dictionary.
   * @param map  map which is expected not to change nor to have {@code null} values
   * @param <MK> map and dictionary key type
   * @param <MV> map and dictionary value type
   * @return dictionary view of the standard map
   */
  @NotNull
  static <MK, MV> Dict<MK, MV> viewMap(@NotNull Map<? extends MK, ? extends MV> map)
  {
    return new Dict.Base<MK, MV>()
    {
      @Nullable
      @Override
      public MV get(MK key)
      {
        return map.get(key);
      }

      @NotNull
      @Override
      public Countable<Entry<MK, MV>> entries()
      {
        return Countable.viewCollection(map.entrySet(), Entry::viewMapEntry);
      }

      @NotNull
      @Override
      public Countable<MV> values()
      {
        return Countable.viewCollection(map.values());
      }

      @NotNull
      @Override
      public Countable<MK> keys()
      {
        return Countable.viewCollection(map.keySet());
      }

      @Override
      public int size()
      {
        return map.size();
      }

      @Override
      public boolean isEmpty()
      {
        return map.isEmpty();
      }

      @Override
      public boolean hasKey(MK key)
      {
        return map.containsKey(key);
      }
    };
  }

  /**
   * Create a read-only view of a map which is possibly {@code null}.
   * If it is {@code null} return an empty dict, otherwise behave
   * like {@link #viewMap(Map)}.
   * @param map  map which may itself be {@code null} but is expected not to change nor to have {@code null} values
   * @param <MK> map and dictionary key type
   * @param <MV> map and dictionary value type
   * @return dictionary view of the standard map, empty if it is {@code null}
   */
  @NotNull
  static <MK, MV> Dict<MK, MV> viewNullableMap(@Nullable Map<? extends MK, ? extends MV> map)
  {
    return map == null
            ? empty()
            : viewMap(map);
  }


  /**
   * Create a read-only view of a standard map as if it is a dictionary and has a different value type.
   * @param map  map which is expected not to change nor to have {@code null} keys
   *             or values
   * @param valueMapper mapper which prepares values before they are returned
   * @param <MK> map and dictionary key type
   * @param <MV> map value type
   * @param <DV> dictionary value type
   * @return dictionary view of the standard map with adapted value type
   */
  @NotNull
  static <MK, MV, DV> Dict<MK, DV> viewMap(@NotNull Map<MK, MV> map,
                                           @NotNull Function<? super MV, ? extends DV> valueMapper)
  {
    return new Dict.Base<MK, DV>()
    {
      @Nullable
      @Override
      public DV get(MK key)
      {
        return valueMapper.apply(map.get(key));
      }

      @NotNull
      @Override
      public Countable<Entry<MK, DV>> entries()
      {
        final Set<Map.Entry<MK, MV>> entries = map.entrySet();
        return Countable.viewCollection(entries, e -> Entry.viewMapEntry(e).valueView(valueMapper));
      }

      @Override
      public int size()
      {
        return map.size();
      }

      @Override
      public boolean isEmpty()
      {
        return map.isEmpty();
      }

      @Override
      public boolean hasKey(MK key)
      {
        return map.containsKey(key);
      }

      @NotNull
      @Override
      public Countable<DV> values()
      {
        return Countable.viewCollection(map.values(), valueMapper);
      }

      @NotNull
      @Override
      public Countable<MK> keys()
      {
        return Countable.viewCollection(map.keySet());
      }
    };
  }

  /**
   *
   * Create a read-only view of a standard map as if it is a dictionary and has a different value type.
   * This takes care of {@code null} values, otherwise it calls {@link #viewMap(Map, Function)}.
   * @param map  map which itself may be {@code null} but is expected not to
   *             change nor to have {@code null} keys or values
   * @param valueMapper mapper which prepares values before they are returned
   * @param <MK> map and dictionary key type
   * @param <MV> map value type
   * @param <DV> dictionary value type
   * @return dictionary view of the standard map with adapted value type,
   *         empty if {@code map} is {@code null}
   */
  @NotNull
  static <MK, MV, DV> Dict<MK, DV> viewNullableMap(@Nullable Map<MK, MV> map,
                                                   @NotNull Function<? super MV, ? extends DV> valueMapper)
  {
    return map == null
            ? empty()
            : viewMap(map, valueMapper);
  }

  /**
   * View a standard 2-dimensional map as 2-dimensional dictionary.
   * @param map     standard 2d map
   * @param <MK1>   outer key type
   * @param <MK2>   inner key type
   * @param <MV>    value type
   * @return 2d dictionary
   */
  @NotNull
  static <MK1, MK2, MV> Dict<MK1, Dict<MK2, MV>> viewMap2(@NotNull Map<MK1, Map<MK2, MV>> map)
  {
    return viewMap(map, v -> v != null ? viewMap(v) : null);
  }

  /**
   * View a standard 2-dimensional map as 2-dimensional dictionary
   * with a different inner value type.
   * @param map     standard 2d map
   * @param valueMapper mapper which prepares values before they are returned
   * @param <MK1>   outer key type
   * @param <MK2>   inner key type
   * @param <MV>    inner map value type (incoming)
   * @param <DV>    inner dictionary value type (returned)
   * @return 2d dictionary with adapted inner value type
   */
  @NotNull
  static <MK1, MK2, MV, DV> Dict<MK1, Dict<MK2, DV>> viewMap2(@NotNull Map<MK1, Map<MK2, MV>> map,
                                                              @NotNull Function<? super MV, ? extends DV> valueMapper)
  {
    return viewMap(map, v -> v != null ? viewMap(v, valueMapper) : null);
  }

  /**
   * Useful implementation for equality which can be used in
   * {@link Dict} implementations for {@link Object#equals(Object)}.
   * <p>
   * It depends on the order of the entries returned by {@code entries},
   * so a {@link Object#hashCode()} should return {@code Types.hash(entries())}.
   * @param dict  dict
   * @param o     other object
   * @param <KK>  key type
   * @param <VV>  value type
   * @return {@code true}
   */
  static <KK, VV> boolean equal(@NotNull Dict<KK, VV> dict,
                                @NotNull Object o)
  {
    if (o == dict) {
      return true;
    }
    if (!(o instanceof Dict)) {
      return false;
    }
    final Dict<?, ?> other = (Dict<?, ?>)o;
    if (other.size() != dict.size()) {
      return false;
    }
    return Types.areEqual(dict.entries(), other.entries());
  }

  /**
   * Creat a string representation from the given dictionary.
   * @param dict dictionary
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull Dict<?,?> dict)
  {
    return "{" + Types.join(", ", Types.view(dict.entries(), Object::toString)) + "}";
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   */
  abstract class Base<KK, VV>
          implements Dict<KK, VV>
  {
    @Override
    public @NotNull Base<KK, VV> asBase()
    {
      return this;
    }

    /**
     * Equals implementation.
     * This implementation depends on the order of the entries
     * returned from {@link #entries()}, so two {@code Dict}s
     * with the same content differ when the order differs.
     * This is consistent with the implementation this
     * class provides for {@link #hashCode()}.
     * @param o other object which is checked for equality
     * @return {@code true} when o is equal to this<br>
     *         {@code false} otherwise
     */
    @Override
    public boolean equals(Object o)
    {
      return Dict.equal(this, o);
    }

    @Override
    public int hashCode()
    {
      return Types.hash(entries());
    }

    @Override
    public String toString()
    {
      return Dict.toString(this);
    }
  }

  /**
   * Entry access.
   * This provides the basic methods {@link #toString()}, {@link #equals(Object)},
   * and {@link #hashCode()}.
   */
  abstract class Entry<EK, EV>
          implements ITuple2<EK, EV>
  {
    /**
     * Get the key.
     * @return key
     */
    public abstract EK getKey();

    /**
     * Get the value.
     * @return value
     */
    public abstract EV getValue();

    /**
     * Get the value, but make sure it is not {@code null}.
     * Depending on the underlying mapping {@code null}
     * values might be acceptable, but will result in an exception
     * nevertheless.
     * @return non-null value associated with {@linkplain #getKey()}
     * @throws NullPointerException if the return value would have been {@code null}
     * @see #getValueOrDefault(Object)
     */
    @NotNull
    public EV getNonNullValue()
    {
      return Objects.requireNonNull(getValue());
    }

    /**
     * Get the value, or a default.
     * Return a default if the associated value is {@code null}.
     * @param defaultValue default value used as fallback
     * @return associated value or {@code defaultValue}
     */
    @NotNull
    public EV getValueOrDefault(@NotNull EV defaultValue)
    {
      return Types.notNull(getValue(), defaultValue);
    }

    @Override
    public EK _1()
    {
      return getKey();
    }

    @Override
    public EV _2()
    {
      return getValue();
    }

    /**
     * View this entry as if it has another key.
     * @param keyMapper mapper from internal key to expected key
     * @param <NEK> new key type
     * @return entry with adapted key and same value
     */
    @NotNull
    public <NEK> Entry<NEK, EV> keyView(@NotNull Function<? super EK, ? extends NEK> keyMapper)
    {
      return new Entry<NEK, EV>()
      {
        @Override
        public NEK getKey()
        {
          return keyMapper.apply(Entry.this.getKey());
        }

        @Override
        public EV getValue()
        {
          return Entry.this.getValue();
        }
      };
    }

    /**
     * View this entry as if it has a different value type.
     * @param valueMapper mapper from internal to expected value
     * @param <NEV> new value type
     * @return entry with same key and adapted value
     */
    @NotNull
    public <NEV> Entry<EK, NEV> valueView(@NotNull Function<? super EV, ? extends NEV> valueMapper)
    {
      return new Entry<EK, NEV>()
      {
        @Override
        public EK getKey()
        {
          return Entry.this.getKey();
        }

        @Override
        public NEV getValue()
        {
          return valueMapper.apply(Entry.this.getValue());
        }
      };
    }

    /**
     * View this entry as if it has a different value.
     * @param keyMapper   mapper from internal to expected key type
     * @param valueMapper mapper from internal to expected value type
     * @param <NEK> returned key type
     * @param <NEV> returned value type
     * @return entry with different key and different value
     */
    @NotNull
    public <NEK, NEV> Entry<NEK, NEV> viewMapped(@NotNull Function<? super EK, ? extends NEK> keyMapper,
                                                 @NotNull Function<? super EV, ? extends NEV> valueMapper)
    {
      return new Entry<NEK, NEV>()
      {
        @Override
        public NEK getKey()
        {
          return keyMapper.apply(Entry.this.getKey());
        }

        @Override
        public NEV getValue()
        {
          return valueMapper.apply(Entry.this.getValue());
        }
      };
    }

    @Override
    @NotNull
    public String toString()
    {
      return String.format("%s\u2192%s", getKey(), getValue());
    }

    @Override
    public boolean equals(Object o)
    {
      if (this == o) {
        return true;
      }
      if (!(o instanceof Dict)) {
        return false;
      }
      final Entry<?,?> other = (Entry<?,?>)o;
      return getKey().equals(other.getKey()) &&
             getValue().equals(other.getValue());
    }

    @Override
    public int hashCode()
    {
      return Objects.hash(getKey(), getValue());
    }

    /**
     * View a pair of key and value as an entry.
     * @param key   key
     * @param value value
     * @param <KT>  key type
     * @param <VT>  value type
     * @return entry with given key and value
     */
    @NotNull
    public static <KT, VT> Entry<KT, VT> view(KT key,
                                              @NotNull VT value)
    {
      return new Entry<KT, VT>()
      {
        @Override
        public KT getKey()
        {
          return key;
        }

        @Override
        public VT getValue()
        {
          return value;
        }
      };
    }

    @NotNull
    public static <MEK, MEV> Entry<MEK, MEV> viewMapEntry(@NotNull Map.Entry<? extends MEK, ? extends MEV> mapEntry)
    {
      final MEK key = mapEntry.getKey();
      final MEV value = mapEntry.getValue();
      if (value == null) {
        throw new IllegalArgumentException("entry has null value!");
      }
      return new Entry<MEK, MEV>()
      {
        @Override
        public MEK getKey()
        {
          return key;
        }

        @Override
        public MEV getValue()
        {
          return value;
        }
      };
    }
  }
}
