// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.util.Counter;
import de.caff.generics.util.SimpleCounter;

import java.util.*;
import java.util.function.Supplier;

/**
 * A sorted set which also counts how often any element was added.
 * <p>
 * Because it would be breaking the set semantics removal is not allowed.
 * All removal methods but {@link CountingTreeSet#clear()}
 * therefore will throw an {@link UnsupportedOperationException}.
 * <p>
 * The methods which return subsets will return standard sets, not
 * members of this class.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class CountingTreeSet<T>
        extends AbstractBasicCountingSet<T>
        implements SortedSet<T>
{
  /**
   * Constructor.
   * <p>
   * This constructor silently assumes that T is Comparable!
   *
   * @param creator counter creator used for counter creation,
   *                see {@link Counter#SIMPLE} or {@link Counter#THREAD_SAFE}
   *                for standard ones
   * @deprecated use {@link #create(Supplier)} instead because it handles comparability correctly
   */
  @Deprecated
  public CountingTreeSet(@NotNull Supplier<Counter> creator)
  {
    super(new TreeMap<>(), creator);
  }

  /**
   * Create a new non-threadsafe counting tree set.
   * @param creator creator for new elements
   * @return counting tree set
   * @param <TT> type of set keys
   */
  @NotNull
  public <TT extends Comparable<T>> CountingTreeSet<TT> create(@NotNull Supplier<Counter> creator)
  {
    return new CountingTreeSet<>(new TreeMap<>(), creator);
  }

  /**
   * Create a new non-threadsafe counting tree set.
   * @param creator creator for new elements
   * @return counting tree set
   * @param <TT> type of set keys
   */
  @NotNull
  public <TT extends Comparable<T>> CountingTreeSet<TT> createThreadSafe(@NotNull Supplier<Counter> creator)
  {
    return new CountingTreeSet<>(new TreeMap<>(), creator);
  }

  /**
   * Constructor.
   * @param comparator comparator for element comparision
   * @param creator counter creator used for counter creation,
   *                see {@link Counter#SIMPLE} or {@link Counter#THREAD_SAFE}
   *                for standard ones
   */
  public CountingTreeSet(@NotNull Comparator<? super T> comparator,
                         @NotNull Supplier<Counter> creator)
  {
    super(new TreeMap<>(comparator), creator);
  }

  /**
   * Constructor.
   * <p>
   * This will use {@link SimpleCounter simple counters}.
   * <p>
   * This constructor silently assumes that T is Comparable!
   * @deprecated use {@link #create()} instead because it handles comparability correctly
   */
  @Deprecated
  public CountingTreeSet()
  {
    this(Counter.SIMPLE);
  }

  /**
   * Create a new simple counting tree set.
   * @return counting tree set
   * @param <TT> type of tree set
   */
  @NotNull
  public <TT extends Comparable<TT>> CountingTreeSet<TT> create()
  {
    return new CountingTreeSet<>(new TreeMap<>(), Counter.SIMPLE);
  }

  /**
   * Constructor.
   * <p>
   * This will use {@link SimpleCounter simple counters}.
   * @param comparator comparator for element comparison
   */
  public CountingTreeSet(@NotNull Comparator<? super T> comparator)
  {
    this(comparator, Counter.SIMPLE);
  }

  /**
   * Constructor.
   * @param set     set used for keeping counters
   * @param creator creator for new counters
   */
  private CountingTreeSet(@NotNull TreeMap<T, Counter> set, @NotNull Supplier<Counter> creator)
  {
    super(set, creator);
  }

  /**
   * Returns the comparator associated with this sorted set, or
   * <tt>null</tt> if it uses its elements' natural ordering.
   *
   * @return the comparator associated with this sorted set, or
   *         <tt>null</tt> if it uses its elements' natural ordering.
   */
  @Override
  public Comparator<? super T> comparator()
  {
    return getTreeMap().comparator();
  }

  /**
   * Get the underlying tree map.
   * @return tree map
   */
  private TreeMap<T, Counter> getTreeMap()
  {
    return ((TreeMap<T, Counter>)map);
  }

  /**
   * Returns a view of the portion of this sorted set whose elements range
   * from <tt>fromElement</tt>, inclusive, to <tt>toElement</tt>, exclusive.
   * (If <tt>fromElement</tt> and <tt>toElement</tt> are equal, the returned
   * sorted set is empty.)  The returned sorted set is backed by this sorted
   * set, so changes in the returned sorted set are reflected in this sorted
   * set, and vice-versa.  The returned sorted set supports all optional set
   * operations that this sorted set supports.<p>
   *
   * The sorted set returned by this method will throw an
   * <tt>IllegalArgumentException</tt> if the user attempts to insert a
   * element outside the specified range.<p>
   *
   * Note: this method always returns a <i>half-open range</i> (which
   * includes its low endpoint but not its high endpoint).  If you need a
   * <i>closed range</i> (which includes both endpoints), and the element
   * type allows for calculation of the successor a given value, merely
   * request the subrange from <tt>lowEndpoint</tt> to
   * <tt>successor(highEndpoint)</tt>.  For example, suppose that <tt>s</tt>
   * is a sorted set of strings.  The following idiom obtains a view
   * containing all of the strings in <tt>s</tt> from <tt>low</tt> to
   * <tt>high</tt>, inclusive: <pre>
   * SortedSet sub = s.subSet(low, high+"\0");
   * </pre>
   *
   * A similar technique can be used to generate an <i>open range</i> (which
   * contains neither endpoint).  The following idiom obtains a view
   * containing all of the Strings in <tt>s</tt> from <tt>low</tt> to
   * <tt>high</tt>, exclusive: <pre>
   * SortedSet sub = s.subSet(low+"\0", high);
   * </pre>
   *
   * @param fromElement low endpoint (inclusive) of the subSet.
   * @param toElement   high endpoint (exclusive) of the subSet.
   * @return a view of the specified range within this sorted set.
   * @throws ClassCastException       if <tt>fromElement</tt> and
   *                                  <tt>toElement</tt> cannot be compared to one another using this
   *                                  set's comparator (or, if the set has no comparator, using
   *                                  natural ordering).  Implementations may, but are not required
   *                                  to, throw this exception if <tt>fromElement</tt> or
   *                                  <tt>toElement</tt> cannot be compared to elements currently in
   *                                  the set.
   * @throws IllegalArgumentException if <tt>fromElement</tt> is greater than
   *                                  <tt>toElement</tt>; or if this set is itself a subSet, headSet,
   *                                  or tailSet, and <tt>fromElement</tt> or <tt>toElement</tt> are
   *                                  not within the specified range of the subSet, headSet, or
   *                                  tailSet.
   * @throws NullPointerException     if <tt>fromElement</tt> or
   *                                  <tt>toElement</tt> is <tt>null</tt> and this sorted set does
   *                                  not tolerate <tt>null</tt> elements.
   */
  @Override
  @NotNull
  public SortedSet<T> subSet(T fromElement, T toElement)
  {
    return new TreeSet<>(this).subSet(fromElement, toElement);
  }

  /**
   * Returns a view of the portion of this sorted set whose elements are
   * strictly less than <tt>toElement</tt>.  The returned sorted set is
   * backed by this sorted set, so changes in the returned sorted set are
   * reflected in this sorted set, and vice-versa.  The returned sorted set
   * supports all optional set operations.<p>
   *
   * The sorted set returned by this method will throw an
   * <tt>IllegalArgumentException</tt> if the user attempts to insert a
   * element outside the specified range.<p>
   *
   * Note: this method always returns a view that does not contain its
   * (high) endpoint.  If you need a view that does contain this endpoint,
   * and the element type allows for calculation of the successor a given
   * value, merely request a headSet bounded by
   * <tt>successor(highEndpoint)</tt>.  For example, suppose that <tt>s</tt>
   * is a sorted set of strings.  The following idiom obtains a view
   * containing all of the strings in <tt>s</tt> that are less than or equal
   * to <tt>high</tt>:
   * <pre>    SortedSet head = s.headSet(high+"\0");</pre>
   *
   * @param toElement high endpoint (exclusive) of the headSet.
   * @return a view of the specified initial range of this sorted set.
   * @throws ClassCastException       if <tt>toElement</tt> is not compatible
   *                                  with this set's comparator (or, if the set has no comparator,
   *                                  if <tt>toElement</tt> does not implement <tt>Comparable</tt>).
   *                                  Implementations may, but are not required to, throw this
   *                                  exception if <tt>toElement</tt> cannot be compared to elements
   *                                  currently in the set.
   * @throws NullPointerException     if <tt>toElement</tt> is <tt>null</tt> and
   *                                  this sorted set does not tolerate <tt>null</tt> elements.
   * @throws IllegalArgumentException if this set is itself a subSet,
   *                                  headSet, or tailSet, and <tt>toElement</tt> is not within the
   *                                  specified range of the subSet, headSet, or tailSet.
   */
  @Override
  @NotNull
  public SortedSet<T> headSet(T toElement)
  {
    return new TreeSet<>(this).headSet(toElement);
  }

  /**
   * Returns a view of the portion of this sorted set whose elements are
   * greater than or equal to <tt>fromElement</tt>.  The returned sorted set
   * is backed by this sorted set, so changes in the returned sorted set are
   * reflected in this sorted set, and vice-versa.  The returned sorted set
   * supports all optional set operations.<p>
   *
   * The sorted set returned by this method will throw an
   * <tt>IllegalArgumentException</tt> if the user attempts to insert a
   * element outside the specified range.<p>
   *
   * Note: this method always returns a view that contains its (low)
   * endpoint.  If you need a view that does not contain this endpoint, and
   * the element type allows for calculation of the successor a given value,
   * merely request a tailSet bounded by <tt>successor(lowEndpoint)</tt>.
   * For example, suppose that <tt>s</tt> is a sorted set of strings.  The
   * following idiom obtains a view containing all of the strings in
   * <tt>s</tt> that are strictly greater than <tt>low</tt>:
   *
   * <pre>    SortedSet tail = s.tailSet(low+"\0");</pre>
   *
   * @param fromElement low endpoint (inclusive) of the tailSet.
   * @return a view of the specified final range of this sorted set.
   * @throws ClassCastException       if <tt>fromElement</tt> is not compatible
   *                                  with this set's comparator (or, if the set has no comparator,
   *                                  if <tt>fromElement</tt> does not implement <tt>Comparable</tt>).
   *                                  Implementations may, but are not required to, throw this
   *                                  exception if <tt>fromElement</tt> cannot be compared to elements
   *                                  currently in the set.
   * @throws NullPointerException     if <tt>fromElement</tt> is <tt>null</tt>
   *                                  and this sorted set does not tolerate <tt>null</tt> elements.
   * @throws IllegalArgumentException if this set is itself a subSet,
   *                                  headSet, or tailSet, and <tt>fromElement</tt> is not within the
   *                                  specified range of the subSet, headSet, or tailSet.
   */
  @Override
  @NotNull
  public SortedSet<T> tailSet(T fromElement)
  {
    return new TreeSet<>(this).tailSet(fromElement);
  }

  /**
   * Returns the first (lowest) element currently in this sorted set.
   *
   * @return the first (lowest) element currently in this sorted set.
   * @throws java.util.NoSuchElementException
   *          sorted set is empty.
   */
  @Override
  public T first()
  {
    return getTreeMap().firstKey();
  }

  /**
   * Returns the last (highest) element currently in this sorted set.
   *
   * @return the last (highest) element currently in this sorted set.
   * @throws java.util.NoSuchElementException
   *          sorted set is empty.
   */
  @Override
  public T last()
  {
    return getTreeMap().lastKey();
  }
}
