// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.FragileFunction1;
import de.caff.generics.function.FragileProcedure1;
import de.caff.generics.util.Counter;

import java.lang.reflect.Array;
import java.util.*;
import java.util.function.*;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * Something containing a defined number of arbitrary elements which can be iterated over.
 * <p>
 * Having a known size usually helps in optimization, e.g. when copying.
 * </p>
 * <p>
 *   This is meant to be used as a read-only interface, although {@link Iterable}
 *   basically exports mutability via {@link Iterator#remove()}.
 *   This interface extends {@code Iterable} nevertheless so
 *   implementations can be used in enhanced {@code for} loops.
 *   In an ideal world Java collections should implement this interface to provide
 *   a better compile-time read-only access compared to the
 *   {@code Collections#unmodifiableCollection()} which will result
 *   in runtime errors when mutating.
 * </p>
 * <p>
 *   To allow living in a standard Java environment this interface provides
 *   methods {@link Countable#asCollection()} to view it as an unmodifiable
 *   collection and {@link Countable#viewCollection(Collection)} allowing
 *   to view a collection as {@link Countable}.
 * </p>
 * <p>
 *   A countable can also view an array or part of it ({@link Countable#viewArray(Object[])}
 *   and {@link Countable#viewArray(Object[], int, int)}), and can be converted to
 *   an array.
 * </p>
 * <p>
 *   Last not least an {@link Countable#empty() empty countable} is useful sometimes.
 * </p>
 *
 * @param <T> object type of this countable
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 24, 2020
 */
public interface Countable<T>
        extends Iterable<T>,
                Sizeable
{
  /**
   * A basic implementation of an empty countable.
   * Don't use this, use {@link #empty()}.
   */
  Countable<Object> EMPTY = new Countable.Base<Object>()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @Override
    @NotNull
    public Iterator<Object> iterator()
    {
      return Collections.emptyIterator();
    }

    @NotNull
    @Override
    public Collection<Object> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public Object[] toArray()
    {
      return Empty.OBJECT_ARRAY;
    }

    @Override
    public int addToArray(@NotNull Object[] array, int arrayPos)
    {
      return arrayPos;
    }

    @NotNull
    @Override
    public <R> Countable<R> view(@NotNull Function<? super Object, ? extends R> mapper)
    {
      return Countable.empty();
    }

    @Override
    public @NotNull <R, E extends Exception> Countable<R> viewFragile(@NotNull FragileFunction1<? extends R, E, ? super Object> mapper)
    {
      return Countable.empty();
    }

    @Override
    public <V> V foldLeft(V initialValue, @NotNull BiFunction<? super V, ? super Object, ? extends V> folder)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public ArrayList<Object> toList()
    {
      return new ArrayList<>();
    }

    @Override
    public void addAllTo(@NotNull Collection<? super Object> collection)
    {
    }

    @NotNull
    @Override
    public Object[] toArray(@NotNull Class<Object> type)
    {
      return Empty.OBJECT_ARRAY;
    }

    @NotNull
    @Override
    public Iterable<Object> filtered(@NotNull Predicate<? super Object> filter)
    {
      return Types.emptyIterable();
    }

    @NotNull
    @Override
    public Countable<Object> filterToCountable(@NotNull Predicate<? super Object> filter)
    {
      return this;
    }

    @NotNull
    @Override
    public Indexable<Object> filteredToIndexable(@NotNull Predicate<? super Object> filter)
    {
      return Indexable.emptyIndexable();
    }

    @Override
    public <E extends Exception> void forEachFragile(@NotNull FragileProcedure1<E, ? super Object> consumer) throws E
    {
    }

    @Override
    public void forEach(Consumer<? super Object> action)
    {
    }

    @NotNull
    @Override
    public Indexable<Object> sorted(@NotNull Comparator<? super Object> order)
    {
      return Indexable.emptyIndexable();
    }

    @Override
    public boolean isSorted(@NotNull Comparator<? super Object> order)
    {
      return true;
    }

    @Override
    public boolean isStrictlySorted(@NotNull Comparator<? super Object> order)
    {
      return true;
    }

    @Override
    public Object first()
    {
      throw new NoSuchElementException("No first element in an empty countable!");
    }

    @Nullable
    @Override
    public Object firstOrNull()
    {
      return null;
    }

    @NotNull
    @Override
    public Optional<Object> optFirst()
    {
      return Optional.empty();
    }

    @Override
    public Object last()
    {
      throw new NoSuchElementException("No last element in an empty countable!");
    }

    @Override
    public Object lastOrNull()
    {
      return null;
    }

    @NotNull
    @Override
    public Optional<Object> optLast()
    {
      return Optional.empty();
    }

    @NotNull
    @Override
    public Countable<Object> withPrependedItem(Object item)
    {
      return Countable.singleton(item);
    }

    @NotNull
    @Override
    public Countable<Object> withAppendedItem(Object item)
    {
      return Countable.singleton(item);
    }

    @Override
    public boolean containsEq(Object elem)
    {
      return false;
    }

    @Override
    public boolean containsRef(Object elem)
    {
      return false;
    }

    @Override
    public boolean hasAny(@NotNull Predicate<? super Object> check)
    {
      return false;
    }

    @Nullable
    @Override
    public Object getAny(@NotNull Predicate<? super Object> check)
    {
      return null;
    }

    @Override
    public boolean hasAll(@NotNull Predicate<? super Object> check)
    {
      return true;
    }

    @NotNull
    @Override
    public <K> Dict<K, Indexable<Object>> groupingBy(@NotNull Function<? super Object, ? extends K> keyExtractor)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public <K, V> Dict<K, Indexable<V>> groupingBy(@NotNull Function<? super Object, ? extends K> keyExtractor,
                                                   @NotNull Function<? super Object, ? extends V> valueMapper)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public <K> Dict<K, Object> mappingBy(boolean firstWins, @NotNull Function<? super Object, ? extends K> keyExtractor)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public <K, V> Dict<K, V> mappingBy(boolean firstWins, @NotNull Function<? super Object, ? extends K> keyExtractor,
                                       @NotNull Function<? super Object, ? extends V> valueMapper)
    {
      return Dict.empty();
    }

    @NotNull
    @Override
    public Indexable<Object> frozen()
    {
      return Indexable.emptyIndexable();
    }

    @NotNull
    @Override
    public Indexable<Object> frozen(@NotNull Function<? super Object, ?> elementCloner, boolean cloneOnGet)
    {
      return Indexable.emptyIndexable();
    }

    @NotNull
    @Override
    public Spliterator<Object> spliterator()
    {
      return Spliterators.emptySpliterator();
    }

    @NotNull
    @Override
    public Stream<Object> stream()
    {
      return Stream.empty();
    }

    @NotNull
    @Override
    public Stream<Object> parellelStream()
    {
      return Stream.empty();
    }
  };

  /**
   * Is this countable empty?
   * An empty countable does not contain any elements which means that {@link #size()}
   * will return {@code 0}.
   * <p>
   * This default implementation just checks for {@link #size()} returning {@code 0}
   * @return {@code true} if this countable is empty, {@code false} otherwise.
   * Because calculating the size might be expensive sometimes in these cases
   * implementations should override this method if possible.
   * </p>
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * View this countable as an unmodifiable standard Java collection.
   * @return standard Java {@link Collection} view of this countable
   */
  @NotNull
  default Collection<T> asCollection()
  {
    return new AbstractCollection<T>()
    {
      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        return Countable.this.iterator();
      }

      @Override
      public int size()
      {
        return Countable.this.size();
      }
    };
  }

  /**
   * Get a list which is the copy of this countable.
   * @return array list with the elements of this countable
   */
  @NotNull
  default ArrayList<T> toList()
  {
    final ArrayList<T> result = new ArrayList<>(size());
    addAllTo(result);
    return result;
  }

  /**
   * Add all entries of this countable to the given collection.
   * @param collection collection where all entries are added
   */
  default void addAllTo(@NotNull Collection<? super T> collection)
  {
    for (T entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Copy the content of this countable to an array.
   * @return array with the copied content of this countabel
   */
  @NotNull
  @SuppressWarnings("unchecked")
  default Object[] toArray()
  {
    final int len = size();
    if (len == 0) {
      return Empty.OBJECT_ARRAY;
    }
    final Object[] array = new Object[len];
    addToArray((T[])array, 0);
    return array;
  }

  /**
   * Copy the elements of this countable into an array of typed objects and return the result.
   * @param type array element type, has to match the element type of this indexable
   * @return array of objects
   */
  @SuppressWarnings("unchecked")
  @NotNull
  default T[] toArray(@NotNull final Class<T> type)
  {
    final T[] array = (T[])Array.newInstance(type, size());
    addToArray(array, 0);
    return array;
  }

  /**
   * Add the complete content of this countable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  start position in hte array where the content is added
   * @return array position after the added content
   */
  default int addToArray(@NotNull T[] array, int arrayPos)
  {
    for (T elem : this) {
      array[arrayPos++] = elem;
    }
    return arrayPos;
  }

  /**
   * View this countable as if it has another type.
   * This is a view which just converts each element when accessed.
   * @param mapper converter from elements of this countable to elements of the returned countable
   * @param <R> resulting element type
   * @return countable view of this countable with mapped elements
   */
  @NotNull
  default <R> Countable<R> view(@NotNull Function<? super T, ? extends R> mapper)
  {
    return new Countable.Base<R>()
    {
      @Override
      public int size()
      {
        return Countable.this.size();
      }

      @NotNull
      @Override
      public Iterator<R> iterator()
      {
        return new IteratorConverter<>(Countable.this.iterator(), mapper);
      }

      @Override
      public boolean isEmpty()
      {
        return Countable.this.isEmpty();
      }
    };
  }

  /**
   * View this countable as if it has another type using a fragile mapping which might throw a checked exception.
   * <p>
   * The returned countable is evaluated lazily, so the exception might be triggered much later,
   * of even never if the element where the converted does throw an exception is never used.
   * <p>
   * If {@code mapper} is throwing an exception this exception is wrapped into a {@link WrappedFragileException}
   * which is thrown instead.
   * <p>
   * If the resulting countable is only temporarily used, you can wrap its usage with a
   * try-catch construction (see below). If it is returned from your code the only way
   * to make the exception appear early is {@link #frozen() freezing} with the price
   * of an internal copy.
   * <p>
   * Example of wrapping the usage (assuming that method {@code Foo foo(Bar)} converts {@code Foo} objects
   * to {@code Bar} objects with the occasional {@code CheckedException} exception):
   * <pre>{@code
   *   try {
   *     Countable<Bar> bars = // ...
   *     try {
   *       Countable<Foo> foos = bars.viewFragile(bar -> foo(bar));
   *       // use foos only inside this block or
   *   //  return foos.frozen();
   *       // to trigger possible exceptions inside this block.
   *     } catch(WrappedFragileException x) {
   *       x.rethrow(CheckedException.class); // this will throw the caught exception
   *       // throw new RuntimeException("Never will reach here!");
   *     }
   *   }
   * }</pre>
   * @param mapper mapper from incoming to outgoing objects which might throw an exception
   * @param <R>    target type of the elements of the returned countable
   * @param <E>    possible exception thrown by {@code mapper}
   * @return lazy view of this countable which will seem to contain target type elements
   */
  @NotNull
  default <R, E extends Exception> Countable<R> viewFragile(@NotNull FragileFunction1<? extends R, E, ? super T> mapper)
  {
    return view(mapper.nonFragile());
  }

  /**
   * Get a filtered view of this countable.
   * @param filter check which defines the elements which are included in the returned {@code Iterbble}.
   * @return iterable only providing the elements which {@code filter} accepts
   */
  @NotNull
  default Iterable<T> filtered(@NotNull Predicate<? super T> filter)
  {
    return () -> new FilteringIteratorWrapper<>(Countable.this.iterator(), filter);
  }

  /**
   * Get a countable which contains elements from this countable
   * which fulfill a given filter condition.
   * In difference to {@link Countable#filtered(Predicate)} this
   * will create an independent copy.
   * @param filter filter conditian used to select elements of the returned indexable
   * @return countable with all elements from this countable which fulfill the filter condition,
   *         in the same sequence as they appear in this countable
   */
  @NotNull
  default Countable<T> filterToCountable(@NotNull Predicate<? super T> filter)
  {
    return viewCollection(Types.filter(this, filter));
  }

  /**
   * Get an indexable which contains elements from this countable
   * which fulfill a given filter.
   * In difference to {@link Countable#filtered(Predicate)} this
   * will create an independent copy.
   * @param filter filter used to select elements of the returned indexable
   * @return indexable with all elements from this countable which fulfill the filter,
   *         in the same sequence as they appear in this indexable, already frozen
   */
  @NotNull
  default Indexable<T> filteredToIndexable(@NotNull Predicate<? super T> filter)
  {
    final ArrayList<T> result = new ArrayList<>(size());
    Types.filter(result, this, filter);
    result.trimToSize();
    return IndexableHelper.frozenFromList(result);
  }

  /**
   * Apply a fragile consumer to each element of this countable.
   * @param consumer fragile consumer which might throw an exception
   * @param <E> exception which the consumer might throw
   * @throws E forwarded exception of the consumer
   */
  default <E extends Exception> void forEachFragile(@NotNull FragileProcedure1<E, ? super T> consumer)
          throws E
  {
    for (T elem : this) {
      consumer.apply(elem);
    }
  }

  /**
   * Calculate a value from this countable by reapplying a 2-argument function for each
   * element, with the result of the latest application and the current element.
   * @param initialValue initial start value used with the first element
   * @param folder       folding function, will receive the accumulated value as its first argument
   *                     and the current element as its second argument
   * @param <V>          result value type
   * @return accumulated value
   */
  default <V> V foldLeft(V initialValue,
                         @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return Types.foldLeft(this, initialValue, folder);
  }

  /**
   * Get a frozen version of this countable.
   * <p>
   * Often countables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying collection might
   * nevertheless change. This method copies the current state of this countable
   * into an unmodifiable state, and returns a Countable which is
   * stable in size and will return always the same elements.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable. Use {@link #frozen(Function, boolean)} to take care
   * of this.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, so you can safely call this method more than once.
   *
   * @return frozen indexable version of this countable
   */
  @NotNull
  default Indexable<T> frozen()
  {
    final ArrayList<T> backup = toList();
    backup.trimToSize();
    return IndexableHelper.frozenFromList(backup);
  }

  /**
   * Freeze a countable to an indexable.
   * In difference to {@link #frozen()} this method allows to change
   * the element type to one of its super types. As the view is read-only
   * this is not a problem.
   * @param countable countable to freeze
   * @return frozen version of the countable
   * @param <TT> element type of returned countable
   */
  @NotNull
  @SuppressWarnings("unchecked")
  static <TT> Indexable<TT> freeze(@NotNull Countable<? extends TT> countable)
  {
    return (Indexable<TT>)countable.frozen();
  }

  /**
   * Get a sorted frozen version of this countable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying collection might
   * nevertheless change. This method copies the current state of this countable
   * into an unmodifiable state, and returns an Indexable which is
   * sorted, stable in size and will always return the same elements
   * in the given order.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable. Use {@link #frozen(Function, boolean)}
   * and sort afterwards to take care of this.
   * @param order ordering condition for the returned indexable
   * @return sorted frozen indexable version of this Countable
   */
  @NotNull
  default Indexable<T> sorted(@NotNull Comparator<? super T> order)
  {
    final ArrayList<T> backup = toList();
    backup.trimToSize();
    backup.sort(order);
    return IndexableHelper.frozenFromList(backup);
  }

  /**
   * Is this countable ordered according to the given sort order?
   * This method accepts equal elements and assumes them ordered.
   * See {@link #isStrictlySorted(Comparator)} for a check which does not accept that.
   * Empty or 1-element countables will always return {@code true}.
   * @param order order which is checked
   * @return {@code true} if the elements in this countable are ordered considering the given {@code order}<br>
   *         {@code false} if not
   */
  default boolean isSorted(@NotNull Comparator<? super T> order)
  {
    if (size() < 2) {
      return true;
    }
    final Iterator<T> it = iterator();
    T lastElem = it.next();
    while (it.hasNext()) {
      final T elem = it.next();
      if (order.compare(lastElem, elem) > 0) {
        return false;
      }
      lastElem = elem;
    }
    return true;
  }

  /**
   * Is this countable strictly ordered according to the given sort order?
   * This method does consider equal elements unordered.
   * See {@link #isSorted(Comparator)} for a check which is more relaxed.
   * Empty or 1-element countables will always return {@code true}.
   * @param order order which is checked
   * @return {@code true} if the elements in this countable are ordered considering the given {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlySorted(@NotNull Comparator<? super T> order)
  {
    if (size() < 2) {
      return true;
    }
    final Iterator<T> it = iterator();
    T lastElem = it.next();
    while (it.hasNext()) {
      final T elem = it.next();
      if (order.compare(lastElem, elem) >= 0) {
        return false;
      }
      lastElem = elem;
    }
    return true;
  }

  /**
   * Get a frozen version of this countable.
   * <p>
   * Often Countables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying collection might
   * nevertheless change. This copies the current state of this countable
   * into an unmodifiable state, and returns a Countable which is
   * stable in size and will returns always the same elements.
   * This method also takes care of possibly mutable elements, which are
   * copied when freezing and possibly also on getting.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, but calling this method again will always create a
   * new object.
   * @param elementCloner cloner called for each element of this countable to create
   *                      the same-position element in the frozen countable
   * @param cloneOnGet    make the returned countable clone an element each
   *                      time it is requested? If {@code true} elements inside
   *                      the returned countable will be protected from mutation with
   *                      the price of creating a copy on each request.
   *
   * @return frozen indexable version of this Countable
   */
  @NotNull
  default Indexable<T> frozen(@NotNull Function<? super T, ? extends T> elementCloner,
                              boolean cloneOnGet)
  {
    final ArrayList<T> backup = new ArrayList<>(size());
    for (T elem : this) {
      backup.add(elementCloner.apply(elem));
    }
    backup.trimToSize();

    return cloneOnGet
            ? IndexableHelper.frozenFromList(backup, elementCloner)
            : IndexableHelper.frozenFromList(backup);
  }

  /**
   * Get the first element.
   * @return first element in this countable
   * @throws NoSuchElementException if this countable is empty
   * @see #firstOrNull()
   * @see #optFirst()
   */
  default T first()
  {
    return iterator().next();
  }

  /**
   * Get the first element, or {@code null} if this countable is empty.
   * Note that this method is dubious for countables which could contain {@code null} elements.
   * @return first element, or {@code null} if this countable is empty
   * @see #first()
   * @see #optFirst()
   */
  @Nullable
  default T firstOrNull()
  {
    return isEmpty()
            ? null
            : first();
  }

  /**
   * Get the first element as an Optional.
   * Note that this method will throw a {@link NullPointerException} if the first element is {@code null} elements.
   * @return optional, empty if this countable is empty
   * @throws NullPointerException if first element is {@code null}
   * @see #first()
   * @see #firstOrNull()
   */
  @NotNull
  default Optional<T> optFirst()
  {
    return isEmpty()
            ? Optional.empty()
            : Optional.of(first());
  }

  /**
   * Get the last element.
   * In the base implementation iterates over all elements in this countable
   * to reach the last element.
   * @return last element in this countable
   * @see #lastOrNull()
   * @see #optLast()
   */
  default T last()
  {
    if (isEmpty()) {
      throw new NoSuchElementException("No last element because there are no elements!");
    }
    T latest = null;
    for (T t : this) {
      latest = t;
    }
    return latest;
  }

  /**
   * Get the last element, or {@code null} if this countable is empty.
   * In the base implementation iterates over all elements in this countable
   * to reach the last element.
   * Note that this method is dubious for countables which could contain {@code null} elements.
   * @return last element, or {@code null} if this countable is empty
   * @see #last()
   * @see #optLast()
   */
  default T lastOrNull()
  {
    return isEmpty()
            ? null
            : last();
  }

  /**
   * Get the first element as an Optional.
   * In the base implementation iterates over all elements in this countable
   * to reach the last element.
   * Note that this method will throw a {@link NullPointerException} if the last element is {@code null} elements.
   * @return optional, empty if this countable is empty
   * @throws NullPointerException if last element is {@code null}
   * @see #last()
   * @see #lastOrNull()
   */
  @NotNull
  default Optional<T> optLast()
  {
    return isEmpty()
            ? Optional.empty()
            : Optional.of(last());
  }

  /**
   * Return a view of this countable which has the given item prepended.
   * @param item item to appear first when iterating
   * @return a view of this countable with the given item prepended
   */
  @NotNull
  default Countable<T> withPrependedItem(T item)
  {
    return new Countable.Base<T>()
    {
      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        return new Iterator<T>()
        {
          Iterator<T> basicIterator;

          @Override
          public boolean hasNext()
          {
            return basicIterator == null
                   || basicIterator.hasNext();
          }

          @Override
          public T next()
          {
            if (basicIterator == null) {
              basicIterator = Countable.this.iterator();
              return item;
            }
            return basicIterator.next();
          }
        };
      }

      @Override
      public int size()
      {
        return Countable.this.size() + 1;
      }
    };
  }

  /**
   * Return a view of this countable which has the given item appended.
   * @param item item to appear last when iterating
   * @return a view of this countable with the given item appended
   */
  @NotNull
  default Countable<T> withAppendedItem(T item)
  {
    return new Countable.Base<T>()
    {
      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        return new Iterator<T>()
        {
          Iterator<T> basicIterator = Countable.this.iterator();

          @Override
          public boolean hasNext()
          {
            return basicIterator != null
                   && basicIterator.hasNext();
          }

          @Override
          public T next()
          {
            if (basicIterator == null) {
              throw new NoSuchElementException("No more elements available!");
            }

            if (basicIterator.hasNext()) {
              return basicIterator.next();
            }
            basicIterator = null;
            return item;
          }
        };
      }

      @Override
      public int size()
      {
        return Countable.this.size() + 1;
      }

    };
  }


  /**
   * Is the given element contained?
   * This uses {@link Objects#deepEquals(Object, Object)} for checking whether an
   * element is contained.
   * @param elem element to check for
   * @return {@code true} if the element is contained in this countable<br>
   *         {@code false} if not
   * @see #containsRef(Object)
   */
  default boolean containsEq(T elem)
  {
    return hasAny(itElem -> Objects.deepEquals(elem, itElem));
  }

  /**
   * Is the given element contained?
   * This checks for an identical reference.
   * @param elem element to check for
   * @return {@code true} if the element is contained in this countable<br>
   *         {@code false} if not
   * @see #containsEq(Object)
   */
  default boolean containsRef(T elem)
  {
    return hasAny(itElem -> elem == itElem);
  }

  /**
   * Check if any element fulfills a given predicate.
   * This uses shortcut evaluation and will return on the first hit.
   * @param check predicate to check
   * @return {@code true} if the predicate tested {@code true} on any element<br>
   *         {@code false} if it tested {@code false} on all elements
   */
  default boolean hasAny(@NotNull Predicate<? super T> check)
  {
    for (T elem : this) {
      if (check.test(elem)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Get the first element which fulfills the predicate.
   * @param check check for identifying matches
   * @return first match found
   */
  @Nullable
  default T getAny(@NotNull Predicate<? super T> check)
  {
    for (T elem : this) {
      if (check.test(elem)) {
        return elem;
      }
    }
    return null;
  }

  /**
   * Check if all elements fulfill a given predicate.
   * This uses shortcut evaluation and will return on the first miss.
   * <p>
   * Note that as this basically checks for misses an empty countable will return
   * always return {@code true}. If this is an issue, add a check for emptiness.
   *
   * @param check predicate to check
   * @return {@code true} if the predicate tested {@code true} on all element<br>
   *         {@code false} if it tested {@code false} on any element
   */
  default boolean hasAll(@NotNull Predicate<? super T> check)
  {
    for (T elem : this) {
      if (!check.test(elem)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Create a mapping of a key extracted from each element to groups of elements which belongs to this key.
   * @param keyExtractor key extractor which defines the key for a given element in this countable
   * @return mapping of keys to groups of elements belonging to this key. It is recommended to apply
   *         {@link Dict#frozen()} to the result if it is not temporary.
   * @param <K> grouping key type
   */
  @NotNull
  default <K> Dict<K, Indexable<T>> groupingBy(@NotNull Function<? super T, ? extends K> keyExtractor)
  {
    final Map<K, List<T>> groupMap = new HashMap<>();
    for (T t : this) {
      groupMap.computeIfAbsent(keyExtractor.apply(t),
                               k -> new ArrayList<>())
              .add(t);
    }
    return Dict.viewMap(groupMap, Indexable::viewList);
  }

  /**
   * Create a mapping of a key extracted from each element to groups of elements which belongs to this key.
   * @param keyExtractor key extractor which defines the key for a given element in this countable
   * @param valueMapper  mapper which converts the elements of this countable to the element type of the
   *                     returned groups
   * @return mapping of keys to groups of values belonging to this key. It is recommended to apply
   *         {@link Dict#frozen()} to the result if it is not temporary.
   * @param <K> grouping key type
   * @param <V> value type of the resulting groups
   */
  @NotNull
  default <K, V> Dict<K, Indexable<V>> groupingBy(@NotNull Function<? super T, ? extends K> keyExtractor,
                                                  @NotNull Function<? super T, ? extends V> valueMapper)
  {
    final Map<K, List<V>> groupMap = new HashMap<>();
    for (T t : this) {
      groupMap.computeIfAbsent(keyExtractor.apply(t),
                               k -> new ArrayList<>())
              .add(valueMapper.apply(t));
    }
    return Dict.viewMap(groupMap, Indexable::viewList);
  }

  /**
   * Create a mapping of a key extracted from each element to the element itself.
   * If two elements resolve to the same key, parameter {@code firstWins} defines
   * whether the first is kept or overwritten.
   * <p>
   * Use {@link #groupingBy(Function)} if you want to keep all values.
   *
   * @param firstWins    if {@code true} the first element which resolves to a given key
   *                     is kept, all others are discarded, if {@code false} the last
   *                     element which resolves to a key is kept
   * @param keyExtractor key extractor which defines the key for a given element in this countable
   * @return mapping of keys to associated elements
   * @param <K> key type
   */
  @NotNull
  default <K> Dict<K, T> mappingBy(boolean firstWins,
                                   @NotNull Function<? super T, ? extends K> keyExtractor)
  {
    final Map<K, T> resultMap = new HashMap<>();
    if (firstWins) {
      for (T t : this) {
        resultMap.putIfAbsent(keyExtractor.apply(t), t);
      }
    }
    else {
      for (T t : this) {
        resultMap.put(keyExtractor.apply(t), t);
      }
    }
    return Dict.viewMap(resultMap);
  }

  /**
   * Create a mapping of a key extracted from each element to the element itself.
   * If two elements resolve to the same key, parameter {@code firstWins} defines
   * whether the first is kept or overwritten.
   * <p>
   * Use {@link #groupingBy(Function, Function)} if you want to keep all values.
   *
   * @param firstWins    if {@code true} the first element which resolves to a given key
   *                     is kept, all others are discarded, if {@code false} the last
   *                     element which resolves to a key is kept
   * @param keyExtractor key extractor which defines the key for a given element in this countable
   * @param valueMapper  mapper which converts the elements of this countable to the element type of the
   *                     returned groups
   * @return mapping of keys to associated elements
   * @param <K> key type
   * @param <V> value type returned mapping
   */
  @NotNull
  default <K, V> Dict<K, V> mappingBy(boolean firstWins,
                                      @NotNull Function<? super T, ? extends K> keyExtractor,
                                      @NotNull Function<? super T, ? extends V> valueMapper)
  {
    final Map<K, V> resultMap = new HashMap<>();
    if (firstWins) {
      for (T t : this) {
        resultMap.putIfAbsent(keyExtractor.apply(t), valueMapper.apply(t));
      }
    }
    else {
      for (T t : this) {
        resultMap.put(keyExtractor.apply(t), valueMapper.apply(t));
      }
    }
    return Dict.viewMap(resultMap);
  }

  /**
   * Convert this into a base countable.
   * Base countables have useful default implementation for
   * {@link Object#equals(Object)}, {@link Object#hashCode()},
   * and {@link Object#toString()}.
   * @return base indexable
   */
  @NotNull
  default Countable.Base<T> asBase()
  {
    return new Countable.Base<T>() {
      @Override
      public Iterator<T> iterator()
      {
        return Countable.this.iterator();
      }

      @Override
      public int size()
      {
        return Countable.this.size();
      }
    };
  }

  /**
   * Does this countable equal another one?
   * <p>
   * This method is more flexible than standard {@link Object#equals(Object)}
   * because it allows to define the check for element equality.
   * <p>
   * It's using {@link #equal(Countable, Countable, BiPredicate)} to perform the check
   * and therefore follows its invariants.
   *
   * @param other          other countable
   * @param elementEquals  check for element equality
   * @return {@code true} if this and the other countable are considered equal regarding the given element check<br>
   *         {@code false} if not
   * @param <E> element type of other countable
   */
  default <E> boolean equals(@NotNull Countable<E> other,
                             @NotNull BiPredicate<? super T, ? super E> elementEquals)
  {
    return equal(this, other, elementEquals);
  }

  /**
   * Create a spliterator from this countable.
   * @return spliterator from this countable
   */
  @NotNull
  default Spliterator<T> spliterator()
  {
    return Spliterators.spliterator(iterator(), size(), 0);
  }

  /**
   * Create a stream from this countable.
   * @return stream from his countable
   */
  @NotNull
  default Stream<T> stream()
  {
    return StreamSupport.stream(spliterator(), false);
  }

  /**
   * Create a parallel stream from this countable.
   * @return parallel stream from this countable
   */
  @NotNull
  default Stream<T> parellelStream()
  {
    return StreamSupport.stream(spliterator(), true);
  }

  /**
   * View a collection as if it were a countable.
   * This creates read-only access to the underlying collection.
   * <p>
   * As this is a view any changes in the underlying collection
   * will be reflected from the returned {@code Countable}.
   * If this may pose a problem copy the collection first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(new ArrayList<>(collection));
   * }</pre></blockquote>
   * @param collection collection to view
   * @param <E> element type of the returned countable, which because of the
   *            read-only nature of this interface can be any super-type of
   *            the collection's element type
   * @return countable view of the given {@code collection}
   */
  @NotNull
  static <E> Countable<E> viewCollection(@NotNull Collection<? extends E> collection)
  {
    return new Countable.Base<E>()
    {
      @Override
      public int size()
      {
        return collection.size();
      }

      @NotNull
      @Override
      public Iterator<E> iterator()
      {
        // note: don't use the collection's iterator directly as it may implement remove()
        final Iterator<? extends E> iterator = collection.iterator();
        return new Iterator<E>()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public E next()
          {
            return iterator.next();
          }
        };
      }

      @NotNull
      @Override
      public Collection<E> asCollection()
      {
        return Collections.unmodifiableCollection(collection);
      }
    };
  }

  /**
   * View a collection which is potentially {@code null} as if it were a countable.
   * This creates read-only access to the underlying collection.
   * <p>
   * As this is a view any changes in the underlying collection
   * will be reflected from the returned {@code Countable}.
   * If this may pose a problem copy the collection first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(new ArrayList<>(collection));
   * }</pre></blockquote>
   * or call {@linkplain #frozen()} on the returned countable
   * @param collection collection to view
   * @param <E> element type of the returned countable, which because of the
   *            read-only nature of this interface can be any super-type of
   *            the collection's element type
   * @return countable view of the given {@code collection}, {@link #empty() empty}
   *         if {@code collection} is {@code null}
   */
  @NotNull
  static <E> Countable<E> viewCollectionN(@Nullable Collection<? extends E> collection)
  {
    return collection == null
            ? empty()
            : viewCollection(collection);
  }

  /**
   * Create a read-only view of a collection as if it were a countable of a different type.
   * This creates read-only access to the underlying collection which represents
   * its elements in a different way.
   * <p>
   * As this is a view any changes in the underlying collection
   * will be reflected from the returned {@code Countable}.
   * If this may pose a problem copy the collection first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(new ArrayList<>(collection), mapper);
   * }</pre></blockquote>
   * or call {@linkplain #frozen()} on the returned countable.
   * @param collection collection to view
   * @param mapper     mapper which turns the elements of the incoming collection into
   *                   the elements of the countable
   * @param <E> element type of the incoming collection
   * @param <T> element type of the returned countable
   * @return countable view of the given {@code collection}
   * @see #viewCollection(Collection)
   */
  @NotNull
  static <E, T> Countable<T> viewCollection(@NotNull Collection<E> collection,
                                            @NotNull Function<? super E, ? extends T> mapper)
  {
    if (collection.isEmpty()) {
      // avoids creating an object, and empty is faster for most methods
      return empty();
    }
    return new Countable.Base<T>()
    {
      @Override
      public int size()
      {
        return collection.size();
      }

      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        final Iterator<E> iterator = collection.iterator();
        return new Iterator<T>()
        {
          @Override
          public boolean hasNext()
          {
            return iterator.hasNext();
          }

          @Override
          public T next()
          {
            return mapper.apply(iterator.next());
          }
        };
      }
    };
  }

  /**
   * Create a read-only view of a collection which is possibly {@code null}
   * as if it were a countable of a different type.
   * This creates a read-only access to the underlying collection which represents
   * its elements in a different way.
   * Besides of the handling of {@code null} this behaves the same as
   * {@link #viewCollection(Collection, Function)}, for {@code null}
   * this returns the empty countable.
   *
   * @param collection collection to view
   * @param mapper     mapper which turns the elements of the incoming collection into
   *                   the elements of the countable
   * @param <E> element type of the incoming collection
   * @param <T> element type of the returned countable
   * @return countable view of the given {@code collection}, empty for {@code null}
   */
  @NotNull
  static <E, T> Countable<T> viewCollectionN(@Nullable Collection<E> collection,
                                             @NotNull Function<? super E, ? extends T> mapper)
  {
    return collection == null
            ? empty()
            : viewCollection(collection, mapper);
  }

  /**
   * View an array as if it were a countable.
   * <p>
   * As this is a view any changes in the underlying array
   * will be reflected by the returned {@code Countable}.
   * If this may pose a problem copy the array first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(array.clone());
   * }</pre></blockquote>
   * @param elements array of elements
   * @param <E>      array element type, also element type of returned countable
   * @return countable view of the given array
   */
  @NotNull
  @SafeVarargs
  static <E> Countable<E> viewArray(@NotNull E... elements)
  {
    if (elements.length == 0) {
      // avoids creating an object, and empty is faster for most methods
      return empty();
    }
    return new Countable.Base<E>()
    {
      @Override
      public int size()
      {
        return elements.length;
      }

      @NotNull
      @Override
      public Iterator<E> iterator()
      {
        return new ArrayIterator<>(elements);
      }
    };
  }

  /**
   * View a part of an array as if it were a countable.
   * <p>
   * As this is a view any changes in the underlying array
   * will be reflected by the returned {@code Countable}.
   * If this may pose a problem copy the array first:
   * <blockquote><pre>{@code
   * Countable.viewCollection(Arrays.copyOfRange(elements, start, start + length));
   * }</pre></blockquote>
   * @param elements array of elements
   * @param start    start index of represented elements of array
   * @param length   number of elements represented from array
   * @param <E>      array element type, also element type of returned countable
   * @return countable view of the given part of the array
   */
  @NotNull
  static <E> Countable<E> viewArray(@NotNull E[] elements,
                                    int start,
                                    int length)
  {
    if (start < 0) {
      throw new IllegalArgumentException("start has to be non-negative, but is "+start);
    }
    if (length < 0) {
      throw new IllegalArgumentException("length has to be non-negative, but is "+length);
    }
    if (start > elements.length - length) {
      throw new IllegalArgumentException(String.format("End %d+%d=%d lies outside range for array with %d elements!",
                                                       start, length, (long)start + length, elements.length));
    }
    if (length == 0) {
      // avoids creating an object, and empty is faster for most methods
      return empty();
    }

    return new Countable.Base<E>()
    {
      @Override
      public int size()
      {
        return length;
      }

      @NotNull
      @Override
      public Iterator<E> iterator()
      {
        return new ArrayIterator<>(elements, start, length);
      }
    };
  }

  /**
   * Downcast a countable of a given type so that it appears at a countable of a super type.
   * This method does nothing but return its argument, but using it keeps the compiler happy.
   * So it is so cheap that the JIT might even remove it.
   * Downcasting countables is no problem, as they don't allow adding new elements.
   * <p>
   * Example:
   * <blockquote><pre>{@code
   * Countable<String> a = Countable.viewArray("a", "b", "c");
   * Countable<Object> b = Countable.downCast(a); // no compiler warning here
   * assert a == b;
   * }</pre></blockquote>
   * @param countable countable to be downcasted, needs to have a tyoe which extends or implements {@code <E>}
   * @param <E> target type of returned countable
   * @return {@code countable}, but using a different element type
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Countable does not allow adding
  static <E> Countable<E> downCast(@NotNull Countable<? extends E> countable)
  {
    return (Countable<E>)countable;
  }

  /**
   * Get an empty countable.
   * This is often preferable as a return result to indicate <i>nothing</i>
   * instead of returning {@code null}.
   * @param <E> element type of the returned empty countable
   * @return empty countable
   */
  @NotNull
  @SuppressWarnings("unchecked") // okay because of read-only access to nothing
  static <E> Countable<E> empty()
  {
    return (Countable<E>)EMPTY;
  }

  /**
   * Get a countable with a single element.
   * @param singleElement single element of this countable
   * @param <E> element type of this countable
   * @return countable of size 1 with the given element
   */
  @NotNull
  static <E> Countable<E> singleton(E singleElement)
  {
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public Iterator<E> iterator()
      {
        return new SingletonIterator<>(singleElement);
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public Indexable<E> frozen()
      {
        return Indexable.singleton(singleElement);
      }

      @NotNull
      @Override
      public Indexable<E> sorted(@NotNull Comparator<? super E> order)
      {
        return Indexable.singleton(singleElement);
      }

      @Override
      public boolean isSorted(@NotNull Comparator<? super E> order)
      {
        return true;
      }

      @Override
      public boolean isStrictlySorted(@NotNull Comparator<? super E> order)
      {
        return true;
      }

      @Override
      public E first()
      {
        return singleElement;
      }

      @NotNull
      @Override
      public E firstOrNull()
      {
        return singleElement;
      }

      @NotNull
      @Override
      public Optional<E> optFirst()
      {
        return Optional.of(singleElement);
      }

      @Override
      public E last()
      {
        return singleElement;
      }

      @Override
      public E lastOrNull()
      {
        return singleElement;
      }

      @NotNull
      @Override
      public Optional<E> optLast()
      {
        return Optional.of(singleElement);
      }

      @NotNull
      @Override
      public <K> Dict<K, Indexable<E>> groupingBy(@NotNull Function<? super E, ? extends K> keyExtractor)
      {
        return Dict.singleton(keyExtractor.apply(singleElement), Indexable.singleton(singleElement));
      }

      @NotNull
      @Override
      public <K, V> Dict<K, Indexable<V>> groupingBy(@NotNull Function<? super E, ? extends K> keyExtractor,
                                                     @NotNull Function<? super E, ? extends V> valueMapper)
      {
        return Dict.singleton(keyExtractor.apply(singleElement),
                              Indexable.singleton(valueMapper.apply(singleElement)));
      }

      @NotNull
      @Override
      public <K> Dict<K, E> mappingBy(boolean firstWins, @NotNull Function<? super E, ? extends K> keyExtractor)
      {
        return Dict.singleton(keyExtractor.apply(singleElement), singleElement);
      }

      @NotNull
      @Override
      public <K, V> Dict<K, V> mappingBy(boolean firstWins, @NotNull Function<? super E, ? extends K> keyExtractor,
                                         @NotNull Function<? super E, ? extends V> valueMapper)
      {
        return Dict.singleton(keyExtractor.apply(singleElement),
                              valueMapper.apply(singleElement));
      }
    };
  }

  /**
   * Get a countable with a single or no element.
   *
   * @param optionalElement optional element
   * @param <E> element type
   * @return countable with size 1 with the given element if it is not {@code null},
   *         or an empty countable if it is {@code null}
   */
  @NotNull
  static <E> Countable<E> optional(@Nullable E optionalElement)
  {
    return optionalElement != null
            ? singleton(optionalElement)
            : empty();
  }

  /**
   * Convert a standard Optional to a Countable with either one or no element.
   * @param optional optional to convert to countable
   * @param <E> optional/countable type
   * @return countable with either 1 element (if the optional has an element)
   *         or 0 elements (if the optional has no element)
   */
  @NotNull
  static <E> Countable<E> fromOptional(@NotNull Optional<E> optional)
  {
    return optional.map(Countable::singleton).orElseGet(Countable::empty);
  }

  /**
   * Get a countable which is returning the same element multiple times.
   * This can be useful under special circumstances.
   * @param element element returned from this countable on each {@code next()} call of its iterator
   * @param count   number of times the same element is returned from the iterator, must not be negative
   * @param <E> element type
   * @return countable which returns {@code element} {@code count} times during iteration
   */
  @NotNull
  static <E> Countable<E> uniform(@NotNull E element, int count)
  {
    if (count < 0) {
      throw new IllegalArgumentException("Count has to be positive or 0, but is "+count);
    }
    if (count == 0) {
      return empty();
    }
    if (count == 1) {
      return singleton(element);
    }
    return new Base<E>()
    {
      @Override
      public int size()
      {
        return count;
      }

      @Override
      @NotNull
      public Iterator<E> iterator()
      {
        return new Iterator<E>()
        {
          private int index = 0;

          @Override
          public boolean hasNext()
          {
            return index < count;
          }

          @Override
          public E next()
          {
            if (index >= count) {
              throw new NoSuchElementException("No element "+index+" in this Countable!");
            }
            ++index;
            return element;
          }
        };
      }
    };
  }

  /**
   * Combine several countables into one.
   * @param parts countables to be combined
   * @param <E> element type of resulting countable
   * @return countable which is the combination of the parts,
   *         iterating will iterate over all elements as expected
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  static <E> Countable<E> combined(@NotNull Countable<? extends E> ... parts)
  {
    return combined(Countable.viewArray(parts));
  }

  /**
   * Combine several countables into one.
   * @param parts countables to be combined
   * @param <E> element type of resulting countable
   * @return countable which is the combination of the parts,
   *         iterating will iterate over all elements as expected
   */
  @NotNull
  @SuppressWarnings("unchecked") // because of read-only access
  static <E> Countable<E> combined(@NotNull Countable<? extends Countable<? extends E>> parts)
  {
    switch (parts.size()) {
    case 0:
      return empty();
    case 1:
      return (Countable<E>)parts.first();
    }
    return new Countable.Base<E>() {
      @Override
      public Iterator<E> iterator()
      {
        return new ConcatenatedIterators<>(parts.view(pt -> (Iterator<E>)pt.iterator()));
      }

      @Override
      public int size()
      {
        final Counter counter = Counter.SIMPLE.get();
        parts.forEach(p -> counter.add(p.size()));
        return counter.getValue();
      }
    };
  }

  /**
   * Create an indexable of pairs from two countables which is the combination of pairs from
   * both countables when ordered. Each pair will either contain a combination of elements
   * from the first and second countable if both have are considered equal according to the order,
   * or of a pair with a {@code null} {@link Pair#first} entry if an element from the second
   * countable does not have a counterpart, or a pair with a {@code null} {@link Pair#second}
   * entry if a given value from the first countable does not have a counterpart.
   * @param countable1 first countable
   * @param countable2 second countable
   * @param order order applied to the countables
   * @param <VV> value type
   * @return ordered indexable of pairs with matching counterparts from both countables, or one
   *         entry being {@code null} for unmatched entries
   */
  @NotNull
  static <VV> Indexable<Pair<VV>> orderedCombination(@NotNull Countable<VV> countable1,
                                                     @NotNull Countable<VV> countable2,
                                                     @NotNull Comparator<? super VV> order)
  {
    final Indexable<VV> sorted1 = countable1.sorted(order);
    final Indexable<VV> sorted2 = countable2.sorted(order);
    final int size1 = sorted1.size();
    final int size2 = sorted2.size();
    final ArrayList<Pair<VV>> result = new ArrayList<>(Math.max(size1, size2));
    int i1 = 0;
    int i2 = 0;
    while (i1 < size1 && i2 < size2) {
      final VV v1 = sorted1.get(i1);
      final VV v2 = sorted2.get(i2);
      final int compare = order.compare(v1, v2);
      if (compare == 0) {
        result.add(Pair.createPair(v1, v2));
        ++i1; ++i2;
      }
      else if (compare < 0) {
        result.add(Pair.createPair(v1, null));
        ++i1;
      }
      else {
        result.add(Pair.createPair(null, v2));
        ++i2;
      }
    }
    while (i1 < size1) {
      result.add(Pair.createPair(sorted1.get(i1++), null));
    }
    while (i2 < size2) {
      result.add(Pair.createPair(null, sorted2.get(i2++)));
    }
    result.trimToSize();
    return IndexableHelper.frozenFromList(result);
  }

  /**
   * Are two countables equal?
   * <p>
   * Equality is complex, because it depends. This equality check for
   * two countables allows to define how elements are compared. Regardless
   * from that it there are 2 invariants:
   * <ol>
   *   <li>Both countables are <b>equal</b> if they refer to the same object.</li>
   *   <li>Both countables are <b>not equal</b> if they have different sizes.</li>
   * </ol>
   * It is possible to compare two countables with different element types.
   *
   * @param countable1 first countable
   * @param countable2 second countable
   * @param equalityChecker checker for equality of both objects
   * @param <E1> element type of first countable
   * @param <E2> element type of second countable
   * @return {@code true} if both countables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static <E1, E2> boolean equal(@NotNull Countable<E1> countable1,
                                @NotNull Countable<E2> countable2,
                                @NotNull BiPredicate<? super E1, ? super E2> equalityChecker)
  {
    if (countable1 == countable2) {
      return true;
    }
    if (countable1.size() != countable2.size()) {
      return false;
    }
    return Types.areEqual(countable1.iterator(),
                          countable2.iterator(),
                          equalityChecker);
  }

  /**
   * Create a string representation of the given countable.
   * @param countable countable
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull Countable<?> countable)
  {
    if (countable.isEmpty()) {
      return "[]";
    }
    final StringBuilder sb = new StringBuilder("[");
    boolean first = true;
    for (Object elem: countable) {
      if (first) {
        first = false;
      }
      else {
        sb.append(',');
      }
      sb.append(elem);
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   * @param <TT> element type
   * @see Countable#asBase()
   */
  abstract class Base<TT> implements Countable<TT>
  {
    @Override
    public int hashCode()
    {
      return Types.hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (!(obj instanceof Indexable)) {
        return false;
      }
      return equal(this, (Indexable<?>)obj, Objects::deepEquals);
    }

    @Override
    public String toString()
    {
      return Countable.toString(this);
    }

    @Override
    @NotNull
    public Base<TT> asBase()
    {
      return this;
    }
  }
}
