// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

/**
 * Enhanced cloning support.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public interface Copyable<T>
{
  /**
   * Get a copy of this object.
   * Immutable objects are allowed to return themselves.
   * The copy needs not to be of the same class, it
   * just has to fulfill the type restriction.
   *
   * @return independent copy of this object
   */
  @NotNull
  T getCopy();

  /**
   * Copy an copyable object while taking care of {@code null} values.
   * @param source source object, possibly {@code null}
   * @param <U> source object type
   * @return copy of source object, or {@code null} if {@code source} was {@code null}
   */
  @Nullable
  static <U extends Copyable<? extends U>> U copy(@Nullable U source)
  {
    return source != null
            ? source.getCopy()
            : null;
  }
}
