// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.EmptyStackException;

/**
 * Implementation of a functional stack which takes care of
 * mutable elements by copying them.
 * <p>
 * See {@link FStack} for more information.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 22, 2020
 * @param <T> element type
 */
class CopierFStackImpl<T>
        implements FStack<T>
{
  /**
   * Empty copier stack implementation.
   * @param <T> element type
   */
  static class Empty<T>
          implements FStack<T>
  {
    private final Copier<T> copier;

    /**
     * Constructor.
     * @param copier copier of this item
     */
    public Empty(Copier<T> copier)
    {
      this.copier = copier;
    }

    @Override
    public T top()
    {
      throw new EmptyStackException();
    }

    @NotNull
    @Override
    public FStack<T> push(T elem)
    {
      return new CopierFStackImpl<>(copier, elem, this);
    }

    @NotNull
    @Override
    public FStack<T> pop()
    {
      throw new EmptyStackException();
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @Override
    public int size()
    {
      return 0;
    }
  }

  @NotNull
  private final Copier<T> copier;

  private final T element;
  @NotNull
  private final FStack<T> tail;

  /**
   * Constructor.
   * @param copier copier use on incoming and outgoing elements
   * @param element element at the top of this stack
   * @param tail tails elements of this stack
   */
  CopierFStackImpl(@NotNull Copier<T> copier, T element, @NotNull FStack<T> tail)
  {
    this.copier = copier;
    this.element = element;
    this.tail = tail;
  }

  @Override
  public T top()
  {
    return copier.nullableCopy(element);
  }

  @NotNull
  @Override
  public FStack<T> push(T elem)
  {
    return new CopierFStackImpl<>(copier, elem, this);
  }

  @NotNull
  @Override
  public FStack<T> pop()
  {
    return tail;
  }

  @Override
  public boolean isEmpty()
  {
    return false;
  }

  @Override
  public int size()
  {
    return 1 + tail.size();
  }
}
