// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.Function1;

/**
 * Get copies of items of a given class.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 29, 2020
 */
@FunctionalInterface
public interface Copier<T>
        extends Function1<T, T>
{
  /**
   * Get a copy of the given item.
   * Implementations may expect item not to be {@code null},
   * callers will have to take care of that. See
   * {@link #nullableCopy(Object)} and
   * {@link #getCopy(Copier, Object)} which help in
   * cases where {@code null} items may appear.
   * @param item item to be copied
   * @return copy of the given item
   */
  @NotNull
  T copy(@NotNull T item);

  /**
   * Apply this as a function.
   * This takes care of possible {@code null} items.
   * @param t item to be copied, possibly {@code null}
   * @return copied item, {@code null} if {@code t} was {@code  null}
   */
  @Override
  default T apply(T t)
  {
    return getCopy(this, t);
  }

  /**
   * Get a copy of the given item even if it is {@code null}.
   * @param item item to be copied, possibly {@code null}
   * @return copier item, {@code null} if item was {@code null}
   */
  @Nullable
  default T nullableCopy(@Nullable T item)
  {
    return getCopy(this, item);
  }

  /**
   * Get a copy of the given item while taking care of {@code null} values.
   * @param copier copier to apply to non-{@code null} items
   * @param item   item to be copied, possibly {@code null}
   * @param <U> item type
   * @return copied item, or {@code null} if item was {@code null}
   */
  @Nullable
  static <U> U getCopy(@NotNull Copier<U> copier,
                       @Nullable U item)
  {
    return item != null
            ? copier.copy(item)
            : null;
  }
}
