// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Concatenated iterators.
 * <p>
 * This allows to iterate over several iterators one after the other in one iteration.
 * It just provides a view and does no additional copying.
 * <p>
 * This class is <b>not thread-safe</b>!
 *
 * @param <T> type of iterator
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August.04, 2016
 */
public class ConcatenatedIterators<T>
        implements Iterator<T>
{
  @NotNull
  private final Iterator<? extends Iterator<? extends T>> iterableIterator;
  @Nullable
  private Iterator<? extends T> nextIterator;

  /**
   * Constructor.
   * @param iterators  iterators to iterate over
   * @see Types#concat(Iterator[])
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public ConcatenatedIterators(Iterator<? extends T> ... iterators)
  {
    this(Types.asList(iterators));
  }

  /**
   * Constructor.
   * @param iterators  iterators to iterate over
   * @see Types#concatIterators(Iterable)
   */
  public ConcatenatedIterators(@NotNull Iterable<? extends Iterator<? extends T>> iterators)
  {
    iterableIterator = iterators.iterator();
    forward();
  }

  /**
   * Step forward through the iterators until the next non-empty
   * iterator is found.
   */
  private void forward()
  {
    while (iterableIterator.hasNext()) {
      nextIterator = iterableIterator.next();
      if (nextIterator.hasNext()) {
        return;
      }
    }
    nextIterator = null;
  }

  /**
   * Returns {@code true} if the iteration has more elements.
   * (In other words, returns {@code true} if {@link #next} would
   * return an element rather than throwing an exception.)
   *
   * @return {@code true} if the iteration has more elements
   */
  @Override
  public boolean hasNext()
  {
    if (nextIterator == null) {
      return false;
    }
    return nextIterator.hasNext();
  }

  /**
   * Returns the next element in the iteration.
   *
   * @return the next element in the iteration
   * @throws NoSuchElementException if the iteration has no more elements
   */
  @Override
  public T next()
  {
    if (nextIterator == null) {
      throw new NoSuchElementException();
    }
    final T value = nextIterator.next();
    if (!nextIterator.hasNext()) {
      forward();
    }
    return value;
  }
}
