// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * Concatenated iterables.
 * <p>
 * This allows to iterate over several iterables one after the other in one iteration.
 * It just provides a view and does no additional copying.
 * <p>
 * This class is <b>not thread-safe</b>!
 *
 * @param <T> type of iterator
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August.04, 2016
 */
public class ConcatenatedIterables<T>
        implements Iterable<T>
{
  @NotNull
  private final Iterable<? extends Iterable<? extends T>> iterables;

  /**
   * Constructor.
   * @param iterables  iterables to iterate over
   * @see Types#concat(Iterable[])
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public ConcatenatedIterables(Iterable<? extends T>... iterables)
  {
    this(Types.asList(iterables));
  }

  /**
   * Constructor.
   * @param iterables  iterators to iterate over
   * @see Types#concatIterables(Iterable)
   */
  public ConcatenatedIterables(@NotNull Iterable<? extends Iterable<? extends T>> iterables)
  {
    this.iterables = iterables;
  }

  /**
   * Returns an iterator over a set of elements of type T.
   *
   * @return an Iterator.
   */
  @NotNull
  @Override
  @SuppressWarnings("unchecked") // cast to Iterator<T> is okay because downcasting a read-only interface
  public Iterator<T> iterator()
  {
    return new ConcatenatedIterators<>(Types.view(iterables, it -> (Iterator<T>)it.iterator()));
  }

  /**
   * Flatten iterables 2 levels deep.
   * @param iterables iterables to flatten
   * @param <U> iterator value type
   * @return concatenation of all iterables in depth 2
   */
  @NotNull
  public static <U> ConcatenatedIterables<U> flatten2(@NotNull Iterable<? extends Iterable< ? extends Iterable<U>>> iterables)
  {
    final List<Iterable<U>> depth1List = new LinkedList<>();
    for (Iterable<? extends Iterable<U>> it : iterables) {
      depth1List.add(new ConcatenatedIterables<>(it));
    }
    return new ConcatenatedIterables<>(depth1List);
  }
}
