// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.CharFunction2;
import de.caff.generics.function.CharOperator2;
import de.caff.generics.function.CharPredicate2;
import de.caff.generics.tuple.Tuple;
import de.caff.generics.tuple.Tuple2;

import java.io.Serializable;
import java.util.Locale;
import java.util.Objects;

/**
 * A pair of primitive char values.
 * <p>
 * This is similar to {@link Pair}, but for raw {@code char} values.
 * As these values are defined final, this class is also immutable.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 04, 2020
 */
public final class CharPair
        implements Serializable,
                   CharIndexable
{
  private static final long serialVersionUID = -4898033497781881954L;
  /** The first value of this pair. */
  public final char first;
  /** The second value of this pair. */
  public final char second;

  /**
   * Constructor.
   * @param first    first value of pair
   * @param second   second value of pair
   */
  public CharPair(char first, char second)
  {
    this.first = first;
    this.second = second;
  }

  /**
   * Create a character pair from a 2-character string.
   *
   * @param pair string with 2 characters
   * @throws IllegalArgumentException if string does not have 2 characters
   */
  public CharPair(@NotNull String pair)
  {
    if (pair.length() != 2) {
      throw new IllegalArgumentException(String.format("pair does not have 2 characters: \"%s\"", pair));
    }
    first = pair.charAt(0);
    second = pair.charAt(1);
  }

  /**
   * Convert this char pair into an integer pair.
   * @return integer pair with same values
   */
  @NotNull
  public IntPair toIntPair()
  {
    return new IntPair(first, second);
  }

  /**
   * Get this pair with a different first value.
   * @param first new first value
   * @return pair with the given {@code first} value,
   *         and the {@link #second} value of this pair
   */
  @NotNull
  public CharPair withFirst(char first)
  {
    if (first == this.first) {
      return this;
    }
    return new CharPair(first, this.second);
  }

  /**
   * Get this pair with a different first value.
   * @param second new second value
   * @return pair with the {@link #first} value of this pair,
   *         and the given {@code second} value
   */
  @NotNull
  public CharPair withSecond(char second)
  {
    if (second == this.second) {
      return this;
    }
    return new CharPair(this.first, second);
  }

  /**
   * Get this pair with swapped values.
   * @return pair with swapped {@link #first} and {@link #second} values
   */
  @NotNull
  public CharPair swapped()
  {
    if (first == second) {
      return this;
    }
    return new CharPair(second, first);
  }

  @Override
  public int size()
  {
    return 2;
  }

  @Override
  public char get(int index)
  {
    switch (index) {
    case 0:
      return first;
    case 1:
      return second;
    default:
      throw new IndexOutOfBoundsException(String.format("Pair has only 2 elements, so %d is out of bounds!", index));
    }
  }

  /**
   * Apply a 2-character function on this pair.
   * @param function function to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @param <T> output type
   * @return result of applying the function
   */
  public <T> T applyAsChars(@NotNull CharFunction2<T> function)
  {
    return function.applyAsChar(first, second);
  }

  /**
   * Call an 2-character operator on this pair.
   * @param operator operator to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @return result of applying the operator
   */
  public char operate(@NotNull CharOperator2 operator)
  {
    return operator.applyAsChar(first, second);
  }

  /**
   * Call a 2-character predicate on this pair.
   * @param predicate predicate to apply, {@link #first} becomes its first, {@link #second} its second argument
   * @return result of testing this pair
   */
  public boolean test(@NotNull CharPredicate2 predicate)
  {
    return predicate.testChars(first, second);
  }

  /**
   * Convert this into a tuple.
   * @return tuple created from this pair
   */
  @NotNull
  public Tuple2<Character, Character> toTuple()
  {
    return Tuple.of(first, second);
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final CharPair that = (CharPair)o;
    return that.first == first && that.second == second;
  }

  @Override
  public int hashCode()
  {
    return Objects.hash(first, second);
  }

  @Override
  @NotNull
  public String toString()
  {
    return String.format(Locale.US, "CharPair(%s, %s)", first, second);
  }
}
