// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.*;
import de.caff.generics.range.Range;

import java.util.*;
import java.util.function.Consumer;

import static de.caff.generics.Order.Ascending;
import static de.caff.generics.Order.Descending;

/**
 * Class which allows readonly access of raw chars by index.
 * <p>
 * The {@link #EMPTY} constant or {@link #emptyIndexable()} method provide the same
 * useful return value to indicate emptiness.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 7, 2020
 * @see de.caff.generics.algorithm.FastCharSequenceSearch
 */
public interface CharIndexable
        extends PrimitiveCharIterable,
                Sizeable
{
  /**
   * Get the element at the given index.
   * @param index index between {@code 0} and {@code size() - 1}
   * @return element at the given index
   */
  char get(int index);

  /**
   * Pythonesque get.
   * This allows to access elements from the back by using negative indexes,
   * e.g. {@code -1} references the last element, {@code -2} its predecessor, and so on.
   * @param index index between {@code -size()} and {@code size() - 1}
   * @return element at the given index
   */
  default char gyt(int index)
  {
    return get(Pythonesque.mapX(index, this));
  }

  /**
   * Get an element modulo size.
   * <p>
   * This maps the given {@code index} into the range of this indexable
   * by applying a modulo {@link #size()} operation. For empty indexable
   * this will throw an {@link IndexOutOfBoundsException} as there is no
   * possible index to get.
   * @param index index, possibly out of range, possibly even negative
   *              for Pythonesque access
   * @return element at the given, possibly modulated, index
   */
  default char getMod(int index)
  {
    final int size = size();
    if (size == 0) {
      throw new IndexOutOfBoundsException("No element for empty indexable!");
    }
    return gyt(index % size);
  }

  /**
   * Get this indexable but with inverted order.
   * @return a view to this indexable which accesses the elements in reverse order
   */
  @NotNull
  default Base reverse()
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }

      @Override
      public char get(int index)
      {
        return CharIndexable.this.get(CharIndexable.this.size() - index - 1);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return CharIndexable.this.asBase();
      }
    };
  }

  /**
   * Is this indexable empty?
   * @return {@code true}: there are no elements in this indexable<br>
   *         {@code false}: this indexable has elements
   */
  default boolean isEmpty()
  {
    return size() == 0;
  }

  /**
   * View this char indexable as an indexable of another content.
   * @param <T> element type of resulting indexable
   * @param mapper mapper from character to the result element type
   * @return a view which has another element type
   */
  @NotNull
  default <T> Indexable<T> viewChars(@NotNull CharFunction1<? extends T> mapper)
  {
    return new Indexable.Base<T>() {
      @Override
      public T get(int index)
      {
        return mapper.applyAsChar(CharIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }
    };
  }

  /**
   * Get an primitive int iterator.
   * @return int iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveIterator.OfInt intIterator()
  {
    return new PrimitiveIterator.OfInt()
    {
      private int index = 0;

      @Override
      public int nextInt()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < size();
      }
    };
  }

  /**
   * Get an primitive char iterator.
   * @return {@code char} iterator which iterates over this indexable
   */
  @NotNull
  default PrimitiveCharIterator charIterator()
  {
    return new PrimitiveCharIterator()
    {
      private int index = 0;

      @Override
      public char nextChar()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < size();
      }
    };
  }

  /**
   * Get a char iterator from the given sub set.
   * @param from first index of iterator
   * @param to   one past last index of iterator
   * @return int iterator which iterates over the given part of this indexable
   */
  @NotNull
  default PrimitiveCharIterator charIterator(final int from, final int to)
  {
    return new PrimitiveCharIterator()
    {
      private int index = from;

      @Override
      public char nextChar()
      {
        if (index >= to) {
          throw new NoSuchElementException(String.format("Index %s out of allowed range [%d, %d[!",
                                                         index, from, to));
        }
        return get(index++);
      }

      @Override
      public boolean hasNext()
      {
        return index < to;
      }
    };
  }

  /**
   * Returns an iterator over elements of type {@code T}.
   *
   * @return an iterator.
   */
  @NotNull
  @Override
  default Iterator<Character> iterator()
  {
    return listIterator();
  }

  /**
   * Returns a list iterator over elements of type {@code T}.
   *
   * @return a read-only list iterator.
   */
  @NotNull
  default ListIterator<Character> listIterator()
  {
    return new ListIterator<Character>() {
      private int index = 0;

      @Override
      public boolean hasNext()
      {
        return index < size();
      }

      @Override
      public Character next()
      {
        if (index >= size()) {
          throw new NoSuchElementException("index: "+index);
        }
        return get(index++);
      }

      @Override
      public boolean hasPrevious()
      {
        return index > 0;
      }

      @Override
      public Character previous()
      {
        if (index == 0) {
          throw new NoSuchElementException("index: -1");
        }
        return get(--index);
      }

      @Override
      public int nextIndex()
      {
        return index;
      }

      @Override
      public int previousIndex()
      {
        return index - 1;
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void set(Character t)
      {
        throw new UnsupportedOperationException();
      }

      @Override
      public void add(Character t)
      {
        throw new UnsupportedOperationException();
      }
    };
  }

  /**
   * Get an indexable subset.
   * The subset includes indices {@code fromIndex} to {@code toIndex - 1}.
   * @param fromIndex start index of sub set
   * @param toIndex   index after last index
   * @return indxable subset view of this indexable
   */
  @NotNull
  default Base subSet(int fromIndex, int toIndex)
  {
    if (fromIndex < 0)
      throw new IndexOutOfBoundsException("fromIndex = " + fromIndex);
    if (toIndex > size())
      throw new IndexOutOfBoundsException("toIndex = " + toIndex);
    if (fromIndex > toIndex)
      throw new IllegalArgumentException("fromIndex(" + fromIndex +
                                         ") > toIndex(" + toIndex + ")");
    final int length = toIndex - fromIndex;
    return length == 0
            ? EMPTY
            : new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public char get(int index)
      {
        if (index >= length) {
          throw new IndexOutOfBoundsException("No such element: "+index);
        }
        return CharIndexable.this.get(index + fromIndex);
      }

      @NotNull
      @Override
      public Base subSet(int fromIdx, int toIdx)
      {
        if (fromIdx < 0) {
          throw new IndexOutOfBoundsException("fromIndex = " + fromIdx);
        }
        if (toIdx > length) {
          throw new IndexOutOfBoundsException("toIndex = " + toIdx);
        }
        if (fromIdx > toIdx) {
          throw new IllegalArgumentException("fromIndex(" + fromIdx +
                                             ") > toIndex(" + toIdx + ")");
        }
        return CharIndexable.this.subSet(fromIndex + fromIdx,
                                         fromIndex + toIdx);
      }
    };
  }

  /**
   * Get an indexable subset.
   * This is the Pythonesque version which allows negative indexes.
   * @param fromIndex start index of sub set
   * @param toIndex   end index of sub set
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base sybSet(int fromIndex, int toIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param fromIndex index to start with, negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base tailSet(int fromIndex)
  {
    return subSet(Pythonesque.mapX(fromIndex, this),
                  size());
  }

  /**
   * Create an indexable subset from the last elements of this indexable set.
   * This already allows Pythonesquew indexing.
   * @param toIndex index one after the end (equal to the length of the returned set),
   *                negative counts from the back
   * @return indexable subset view of this indexable set
   */
  @NotNull
  default Base headSet(int toIndex)
  {
    return subSet(0,
                  Pythonesque.mapX(toIndex, this));
  }

  /**
   * Return a mapped view on this character indexable.
   * @param mapper mapper
   * @param <T> result type pf mapper
   * @return mapped indexable
   */
  @NotNull
  default <T> Indexable<T> view(@NotNull CharFunction1<? extends T> mapper)
  {
    return new Indexable.Base<T>()
    {
      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }

      @Override
      public T get(int index)
      {
        return mapper.applyAsChar(CharIndexable.this.get(index));
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable collection.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   */
  @NotNull
  default Collection<Character> asCollection()
  {
    return new AbstractCollection<Character>()
    {
      @NotNull
      @Override
      public Iterator<Character> iterator()
      {
        return CharIndexable.this.iterator();
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }
    };
  }

  /**
   * Get a view on this indexable as an unmodifiable list.
   * This is especially useful for conversion to standard collection classes, eg for adding all
   * elements via {@link Collection#addAll(Collection)}.
   * @return collection view
   * @see #toList()
   */
  @NotNull
  default List<Character> asList()
  {
    return new AbstractList<Character>()
    {
      @Override
      public Character get(int index)
      {
        return CharIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }

      @NotNull
      @Override
      public Iterator<Character> iterator()
      {
        return CharIndexable.this.iterator();
      }
    };
  }

  /**
   * Create a list from this char indexable.
   * This creates an independent list to which the elements of this
   * indexable are copied.
   * @return list copy of this indexable
   * @see #asList()
   */
  @NotNull
  default ArrayList<Character> toList()
  {
    final ArrayList<Character> list = new ArrayList<>(size());
    addAllTo(list);
    return list;
  }

  /**
   * Add all elements of this indexable to the given collection.
   * @param collection collection to add to
   */
  default void addAllTo(@NotNull Collection<? super Character> collection)
  {
    for (Character entry : this) {
      collection.add(entry);
    }
  }

  /**
   * Get a view of this indexable as a standard object-based indexable.
   * @return indexable with non-null {@code Character} values
   */
  @NotNull
  default Indexable<Character> asIndexable()
  {
    return new Indexable.Base<Character>()
    {
      @Override
      public Character get(int index)
      {
        return CharIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }
    };
  }

  /**
   * Convert this indexable into a Java array.
   * @return array copy containing all values of this indexable
   */
  @NotNull
  default char[] toArray()
  {
    if (isEmpty()) {
      return Empty.CHAR_ARRAY;
    }
    final char[] array = new char[size()];
    addToArray(array, 0);
    return array;
  }

  /**
   * Convert this char indexable into a Java String.
   * This creates a simple string consisting of the characters,
   * while the standard {@link #toString(CharIndexable)} methods reflects
   * the indexable nature of this object.
   * <p>
   * Note that in many cases a {@code java.lang.CharSequence} can be used instead
   * of a string, see {@link #viewAsCharSequence()} which is cheaper especially for
   * large sequences.
   * @return simple string built from concatenating the characters of this indexable
   */
  @NotNull
  default String makeString()
  {
    final StringBuilder sb = new StringBuilder(size());
    for (PrimitiveCharIterator it = charIterator();  it.hasNext();  ) {
      sb.append(it.nextChar());
    }
    return sb.toString();
  }

  /**
   * View this char indexable transformed by an operator.
   * @param op transforming operator
   * @return view of this indexable with transformed values
   */
  @NotNull
  default CharIndexable viewOp(@NotNull CharOperator1 op)
  {
    return CharIndexable.viewByIndex(size(),
                                       idx -> op.applyAsChar(CharIndexable.this.get(idx)));
  }

  /**
   * View this char indexable as a standard Java char sequence.
   * @return char sequence view
   * @see #makeString()
   */
  @NotNull
  default CharSequence viewAsCharSequence()
  {
    return new CharSequence()
    {
      @Override
      public int length()
      {
        return CharIndexable.this.size();
      }

      @Override
      public char charAt(int index)
      {
        return CharIndexable.this.get(index);
      }

      @Override
      @NotNull
      public CharSequence subSequence(int start, int end)
      {
        return CharIndexable.this.subSet(start, end).viewAsCharSequence();
      }

      @NotNull
      public String toString()
      {
        return CharIndexable.this.makeString();
      }
    };
  }

  /**
   * Add the content of this indexable to the given array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull char[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given array.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull char[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveCharIterator it = charIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextChar();
    }
    return arrayPos;
  }

  /**
   * Convert this indexable into an int array.
   * @return array copy containing all values of this indexable
   */
  @NotNull
  default int[] toIntArray()
  {
    if (isEmpty()) {
      return Empty.INT_ARRAY;
    }
    final int[] array = new int[size()];
    addToArray(array, 0);
    return array;
  }


  /**
   * Add the content of this indexable to the given int array.
   * @param array array where the content is added
   * @param pos   position in the array where the content of this indexable is added
   * @return array position after this indexable
   */
  default int addToArray(@NotNull int[] array, int pos)
  {
    return addToArray(array, pos, 0, size());
  }

  /**
   * Add a part of the content of this indexable to the given intarray.
   * @param array     array where the content is added
   * @param arrayPos  position in hte array where the content is added
   * @param index     start index of this indexable which is added first
   * @param length    number of entries of this indexable added to the array
   * @return array position after the added content
   */
  default int addToArray(@NotNull int[] array, int arrayPos,
                         int index, int length)
  {
    for (PrimitiveCharIterator it = charIterator(index, index + length);  it.hasNext();  ) {
      array[arrayPos++] = it.nextChar();
    }
    return arrayPos;
  }

  /**
   * Get the indexes of this indexable as an iterable range.
   * @return iterable indexes
   * @see #intIndexes()
   * @see Range#indexes(int)
   */
  @NotNull
  default Iterable<Integer> indexes()
  {
    return Range.indexes(size());
  }

  /**
   * Get the indexes of this indexable as an integer indexable.
   * @return indexes as integer indexable
   * @see #indexes()
   * @see IntIndexable#rangeFromSize(int)
   */
  @NotNull
  default IntIndexable intIndexes()
  {
    return IntIndexable.rangeFromSize(size());
  }

  /**
   * View this char indexable as an integer indexable.
   * @return integer indexable view
   */
  @NotNull
  default IntIndexable.Base asIntIndexable()
  {
    return new IntIndexable.Base()
    {
      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }

      @Override
      public int get(int index)
      {
        return CharIndexable.this.get(index);
      }
    };
  }

  /**
   * Go over all values and do a cumulative calculation.
   *
   * @param initialValue  start value for the operation, will be returned if this iterable is empty
   * @param foldOperation operation applied to each value, will get the accumulated value as its
   *                      first and the current element value as its second argument
   * @return accumulated value, result of applying fold operation to all values of this indexable
   *         iteratively
   */
  default char foldLeft(char initialValue,
                        @NotNull CharOperator2 foldOperation)
  {
    char result = initialValue;
    for (PrimitiveCharIterator iterator = charIterator();
         iterator.hasNext();  ) {
      result = foldOperation.applyAsChar(result, iterator.nextChar());
    }
    return result;
  }

  /**
   * Get the next index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int nextMatch(int startIndex, @NotNull CharPredicate1 check)
  {
    final int size = size();
    for (int i = Pythonesque.mapLB(startIndex, this);  i < size;  ++i) {
      if (check.testChar(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the first index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return first index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int firstMatch(@NotNull CharPredicate1 check)
  {
    return nextMatch(0, check);
  }

  /**
   * Get the previous index for which the given check is fulfilled.
   * @param startIndex start index for checking, {@link Pythonesque} indexing supported
   * @param check      check to perform on elements of this indexable until {@code true}
   * @return next index starting with {@code startIndex} for which the check returns {@code true},
   *         {@link Pythonesque#UNMAPPABLE} if none
   */
  default int previousMatch(int startIndex, @NotNull CharPredicate1 check)
  {
    for (int i = Pythonesque.mapUB(startIndex, this);  i >= 0;  --i) {
      if (check.testChar(get(i))) {
        return i;
      }
    }
    return Pythonesque.UNMAPPABLE;
  }

  /**
   * Get the last index for which the given check is fulfilled.
   * @param check check to perform on elements of this indexable until {@code true}
   * @return last index for which the check returns {@code true},
   *         {@code -1} if none
   */
  default int lastMatch(@NotNull CharPredicate1 check)
  {
    return previousMatch(-1, check);
  }

  /**
   * Get a frozen version of this indexable.
   * <p>
   * Often Indexables are used as a view to underlying collections.
   * Although this interface is immutable, the underlying colelction might
   * nevertheless change. This copies the current state of this indexable
   * into an unmodifiable state, and returns an Indexable which is
   * stable in size and will return always the same element for a given index.
   * Beware: it is still possible that any element itself changes when the
   * elements are mutable.
   * <p>
   * Calling {@code frozen()} again on the returned object will just return
   * the object itself, so you can safely call this method more than once.
   *
   * @return frozen version of this Indexable
   */
  @NotNull
  default CharIndexable frozen()
  {
    return IndexableHelper.frozenFromArray(toArray());
  }

  /**
   * Is this indexable sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * See {@linkplain #isStrictlyOrdered(CharOrdering)} for a stricter alternative.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isOrdered(@NotNull CharOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    char last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final char value = get(i);
      if (order.checkChar(value, last) == Descending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable sorted naturally?
   * This method tests that consecutive elements in this indexable
   * or either ascending or the same, but never descending.
   * <p>
   * See {@linkplain #isStrictlyOrdered()} for a stricter alternative.
   * @return {@code true} if this indexable is sorted naturally<br>
   *         {@code false} if not
   */
  default boolean isOrdered()
  {
    if (size() < 2) {
      return true;
    }
    int last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final int value = get(i);
      if (value < last) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted according to the given ordering?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered(CharOrdering)} for a more relaxed alternative.
   * <p>
   * Prefer {@linkplain #isOrdered()} to check for natural ordering because it is faster.
   * @param order expected ordering
   * @return {@code true} if this indexable is sorted as defined by {@code order}<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered(@NotNull CharOrdering order)
  {
    if (size() < 2) {
      return true;
    }
    char last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final char value = get(i);
      if (order.checkChar(value, last) != Ascending) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Is this indexable strictly sorted naturally?
   * This method tests that consecutive elements in this indexable
   * or strictly ascending or the same, but never descending.
   * See {@linkplain #isOrdered()} for a more relaxed alternative.
   * @return {@code true} if this indexable is sorted naturally<br>
   *         {@code false} if not
   */
  default boolean isStrictlyOrdered()
  {
    if (size() < 2) {
      return true;
    }
    int last = gyt(-1);
    for (int i = size() - 2;  i >= 0;  --i) {
      final int value = get(i);
      if (value >= last) {
        return false;
      }
      last = value;
    }
    return true;
  }

  /**
   * Do a binary search in an indexable ordered in natural ascending order.
   * This requires this indexable to be ordered in
   * {@link CharOrdering#ASCENDING natural ascending order},
   * i.e. {@linkplain #isOrdered(CharOrdering)} has to return {@code true}
   * for this ordering.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(CharOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @return a positive integer defining the index where the given value was found,
   *         and a negative integer if the value is not contained. The latter defines
   *         the insertion point where the looked up value would match into this
   *         sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(char value)
  {
    final int iValue = value; // use int as bytecode cannot compare chars
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final int dp = get(mid);

      if (dp < iValue) {
        low = mid + 1;
      }
      else if (dp > iValue) {
        hi = mid - 1;
      }
      else {
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Do a binary search in an ordered indexable.
   * This requires this indexable to be ordered in non-descending order
   * as defined by the given {@code order},
   * i.e. {@linkplain #isOrdered(CharOrdering)} has to return {@code true}
   * for this order.
   * For efficiency this prerequisite is not checked, but not fulfilling
   * it will make this method return bogus results. If this indexable is only ordered,
   * but not {@link #isStrictlyOrdered(CharOrdering) strictly ordered} it is not defined
   * which index is returned when the looked up value appears in a sequence of equal values.
   *
   * @param value value to look up
   * @param order sort order
   * @return a positive integer defining the index where the given value was found,
   * and a negative integer if the value is not contained. The latter defines
   * the insertion point where the looked up value would match into this
   * sorted indexable in the form {@code -(insertIndex + 1)}.
   */
  default int binarySearch(char value, @NotNull CharOrdering order)
  {
    int low = 0;
    int hi  = size() - 1;

    while (low <= hi) {
      final int mid = (low + hi) >>> 1;
      final char dp = get(mid);

      switch (order.checkChar(dp, value)) {
      case Ascending:
        low = mid + 1;
        break;
      case Descending:
        hi = mid - 1;
        break;
      default:
        return mid;
      }
    }
    return -(low + 1);
  }

  /**
   * Create a (mutable) char indexable from the values of this indexable
   * which is ordered as defined by the given ordering.
   * @param order sort order
   * @return independent indexable with sorted values
   */
  @NotNull
  default MutableCharIndexable ordered(@NotNull CharOrdering order)
  {
    final MutableCharIndexable result = MutableCharIndexable.fromCharIndexable(this);
    result.order(order);
    return result;
  }

  /**
   * Make this char indexable usable as a class with defined standard Object methods.
   * @return based version of this indexable
   */
  @NotNull
  default Base asBase()
  {
    return new CharIndexable.Base() {
      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }

      @Override
      public char get(int index)
      {
        return CharIndexable.this.get(index);
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return CharIndexable.this.reverse();
      }

      @NotNull
      @Override
      public PrimitiveIterator.OfInt intIterator()
      {
        return CharIndexable.this.intIterator();
      }

      @NotNull
      @Override
      public PrimitiveCharIterator charIterator()
      {
        return CharIndexable.this.charIterator();
      }

      @NotNull
      @Override
      public PrimitiveCharIterator charIterator(int from, int to)
      {
        return CharIndexable.this.charIterator(from, to);
      }

      @NotNull
      @Override
      public Iterator<Character> iterator()
      {
        return CharIndexable.this.iterator();
      }

      @NotNull
      @Override
      public ListIterator<Character> listIterator()
      {
        return CharIndexable.this.listIterator();
      }

      @NotNull
      @Override
      public Base subSet(int fromIndex, int toIndex)
      {
        return CharIndexable.this.subSet(fromIndex, toIndex);
      }

      @NotNull
      @Override
      public char[] toArray()
      {
        return CharIndexable.this.toArray();
      }

      @Override
      public int addToArray(@NotNull char[] array, int pos)
      {
        return CharIndexable.this.addToArray(array, pos);
      }

      @Override
      public int addToArray(@NotNull char[] array, int arrayPos, int index, int length)
      {
        return CharIndexable.this.addToArray(array, arrayPos, index, length);
      }

      @NotNull
      @Override
      public IntIndexable.Base asIntIndexable()
      {
        return CharIndexable.this.asIntIndexable();
      }
    };
  }

  /**
   * Create a view of this char indexable with an inserted value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are inserted.
   * Use {@link ExpandableIntIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable might 
   * result in a disaster when using the returned indexable!
   *
   * @param index index where the value is inserted before the current index
   *              (<b>not Pythonesque</b>, because it is possible to add an value at {@code index == size()},
   *              and it is expected that most insertion will happen at 0 or at the end)
   * @param value inserted value at that index
   * @return a view of this indexable with 1 more value at the given index,
   *         all values beyond are moved one index forward
   * @see #withAppendedValue(char) 
   */
  @NotNull
  default CharIndexable withInsertedValueAt(int index, char value)
  {
    final int insertIndex = index;
    final int newSize = size() + 1;
    if (insertIndex == 0) {
      return new CharIndexable.Base() {
        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public char get(int index)
        {
          return index == 0
                  ? value
                  : CharIndexable.this.get(index - 1);
        }
      };
    }
    if (insertIndex == newSize - 1) {
      return new CharIndexable.Base() {

        @Override
        public int size()
        {
          return newSize;
        }

        @Override
        public char get(int index)
        {
          return index == newSize - 1
                  ? value
                  : CharIndexable.this.get(index);
        }
      };
    }
    return new CharIndexable.Base()
    {
      @Override
      public int size()
      {
        return newSize;
      }

      @Override
      public char get(int index)
      {
        if (index == insertIndex) {
          return value;
        }
        return CharIndexable.this.get(index < insertIndex
                                                ? index
                                                : index - 1);
      }
    };
  }

  /**
   * Create a view with of this indexable with another value added at the end.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few items are added.
   * @param value value to add
   * @return indexable view of this indexable, with the given value added at the end,
   *         so size is larger by 1
   * @see #withInsertedValueAt(int, char)
   */
  @NotNull
  default CharIndexable withAppendedValue(char value)
  {
    return withInsertedValueAt(size(), value);
  }

  /**
   * Create a view of this char indexable with an exchanged value.
   * Note that this is not efficient, but handy in certain situations
   * where only 1 or a few values are exchanged.
   * Use {@link ExpandableIntIndexable} if you want to carry out more
   * stuff like this.
   * <p>
   * As this creates a view of this indexable, changes to this indexable could 
   * lead to unexpected results when using the returned indexable!
   *
   * @param index index where the element is exchanged, hiding the current element.
   *              (Pythonesque)
   * @param value exchanged value at that index
   * @return a view of this indexable with 1 more element at the given index,
   *         all elements beyond are moved one index forward
   */
  @NotNull
  default CharIndexable withExchangedValueAt(int index, char value)
  {
    final int insertIndex = Pythonesque.mapX(index, this);
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        return index == insertIndex
                ? value
                : CharIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }
    };
  }

  /**
   * Create a view of this indexable with one value removed.
   * Note that this is not efficient, but handy in certain situations
   * where only one or a few values are removed.
   *
   * @param index index of the removed value (Pythonesque)
   * @return a view of this indexable where the value at the given index
   *         is removed
   */
  @NotNull
  default CharIndexable withRemovedValueAt(int index)
  {
    final int removeIndex = Pythonesque.mapX(index, this);
    if (removeIndex == 0) {
      // remove first
      return tailSet(1);
    }
    final int sz = size() - 1;
    if (removeIndex == sz) {
      // remove last
      return headSet(-1);
    }
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        return index < removeIndex
                ? CharIndexable.this.get(index)
                : CharIndexable.this.get(index + 1);
      }

      @Override
      public int size()
      {
        return sz;
      }
    };
  }

  /**
   * Create a view of this char indexable with two values swapped.
   * Note that this is not efficient, but handy in certain situations
   * where only a few items are swapped.
   *
   * @param index1 index of the first value (Pythonesque)
   * @param index2 index of the second value (Pythonesque)
   * @return a view of this indexable where the first and second value
   *         have exchanged places
   */
  @NotNull
  default CharIndexable withSwappedValuesAt(int index1, int index2)
  {
    final int swap1 = Pythonesque.mapX(index1, this);
    final int swap2 = Pythonesque.mapX(index2, this);
    if (swap1 == swap2) {
      return this;
    }
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        if (index == swap1) {
          return CharIndexable.this.get(swap2);
        }
        if (index == swap2) {
          return CharIndexable.this.get(swap1);
        }
        return CharIndexable.this.get(index);
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }
    };
  }

  /**
   * Get a rotated view of this char indexable.
   * The returned indexable will show this indexable with rotated indexes.
   * A positive {@code steps} will rotate left, i.e. any index will be accessed
   * as if the number of steps is added before extraction (modulo length).
   * E.g. a rotation of {@code -1} will return the last element when index {@code 0} is
   * requested, and the first element for index {@code 1}.
   *
   * @param steps steps to rotate
   * @return rotated view of this indexable, use  {@link #frozen()}
   *         to create an indexable which no longer depends on this one
   */
  @NotNull
  default CharIndexable rotated(int steps)
  {
    steps = steps % size();
    if (steps == 0) {
      return this;
    }
    if (steps < 0) {
      steps += size();
    }
    final int rotate = steps;
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        return CharIndexable.this.get((index + rotate) % size());
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }

      @NotNull
      @Override
      public CharIndexable rotated(int steps)
      {
        return CharIndexable.this.rotated(rotate + steps);
      }
    };
  }

  /**
   * View this indexable as a boolean indexable while using a conversion for
   * each value.
   * <p>
   *   Use {@link #frozen()} on the result to decouple it from this indexable.
   * </p>
   * @param convert converter applied to each value of this indexable before
   *                it is returned from the resulting indexable
   * @return boolean indexable view of this indexable
   */
  @NotNull
  default BooleanIndexable viewAsBoolean(@NotNull CharPredicate1 convert)
  {
    return new BooleanIndexable.Base()
    {
      @Override
      public boolean get(int index)
      {
        return convert.testChar(CharIndexable.this.get(index));
      }

      @Override
      public int size()
      {
        return CharIndexable.this.size();
      }
    };
  }

  /**
   * Create a string representation of this char indexable.
   * Compared to {@link Object#toString ()} this method will always work,
   * {@code toString()} will only do what is expected when the implementation
   * extends {@link CharIndexable.Base} (which takes care of {@code toString()}
   * or provides the expected implementation itself.
   * <p>
   * Use {@link #asCharSequence()} to create a non-copy view of this char indexable,
   * and use {@link #subSet(int, int)}, {@link #headSet(int)}, or {@link #tailSet(int)}
   * first to create texts from parts of this indexable.
   * @return a string created from the characters in this char indexable
   * @see #asCharSequence() 
   */
  @NotNull
  default String toText()
  {
    return CharIndexable.toString(this);
  }

  /**
   * Get a view of this char indexable as a char sequence.
   * <p>
   * The main advantage over {@link Object#toString()} is that this method will
   * always work even if the implementation of this interface is neither
   * extending {@link CharIndexable.Base} nor takes care of creating a string
   * itself, and that this method does not copy the chars around which can
   * be preferable for large sizes.
   * @return char sequence view, i.e. any changes to this char indexable
   *         will be reflected in the view. Use {@link #frozen()}
   *         for an immutable char indexable first if this might create a problem.
   * @see #toText()
   */
  @NotNull
  default CharSequence asCharSequence()
  {
    return new CharSequence()
    {
      @Override
      public int length()
      {
        return CharIndexable.this.size();
      }

      @Override
      public char charAt(int index)
      {
        return CharIndexable.this.get(index);
      }

      @Override
      public CharSequence subSequence(int start, int end)
      {
        return CharIndexable.this.subSet(start, end).viewAsCharSequence();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This requires a list of non-null numbers as a base.
   * @param list list used as a base, required to have only non-null elements
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Character> list)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public char get(int index)
      {
        return list.get(index);
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a list as an Indexable.
   * This allows to provide a fallback for null elements in the list.
   * @param list list used as a base, required to have only non-null elements
   * @param nullValue value returned for {@code null} elements in the list
   * @return indexable view of the given list, so any changes of the list will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewList(@NotNull List<? extends Character> list, char nullValue)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public char get(int index)
      {
        final Character value = list.get(index);
        return value != null
                ? value
                : nullValue;
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic list as a char indexable.
   * @param list generic list
   * @param extractor extractor function which extracts a char from the elements of {@code list}
   * @param <T> element type of list
   * @return a char indexable view of {@code list}
   */
  @NotNull
  static <T> Base viewList(@NotNull List<T> list,
                                           @NotNull ToCharFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public char get(int index)
      {
        return extractor.applyAsChar(list.get(index));
      }

      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }
    };
  }

  /**
   * View a generic character indexable as char indexable.
   * @param indexable generic indexable with Character elements, required to have only non-null elements
   * @return char indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Character> indexable)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public char get(int index)
      {
        return indexable.get(index);
      }
    };
  }

  /**
   * View a generic character indexable as char indexable.
   * @param indexable generic indexable with Character elements
   * @param nullFallback fallback for {@code null} elements
   * @return char indexable view
   */
  @NotNull
  static Base viewIndexable(@NotNull Indexable<? extends Character> indexable,
                            char nullFallback)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public char get(int index)
      {
        final Character value = indexable.get(index);
        return value != null
                ? value
                : nullFallback;
      }
    };
  }

  /**
   * View a generic indexable as a char indexable.
   * @param indexable generic indexable
   * @param extractor extractor function which extracts a char from the elements of {@code indexable}
   * @param <T> element type of indexable
   * @return a char indexable view of {@code indexable}
   */
  @NotNull
  static <T> Base viewIndexable(@NotNull Indexable<T> indexable,
                                @NotNull ToCharFunction<? super T> extractor)
  {
    return new Base()
    {
      @Override
      public char get(int index)
      {
        return extractor.applyAsChar(indexable.get(index));
      }

      @Override
      public int size()
      {
        return indexable.size();
      }

      @Override
      public boolean isEmpty()
      {
        return indexable.isEmpty();
      }
    };
  }

  /**
   * View a string as an CharIndexable.
   * @param text text string
   * @return indexable view of the given text string
   */
  @NotNull
  static Base viewString(@NotNull CharSequence text)
  {
    if (text.length() == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return text.length();
      }

      @Override
      public char get(int index)
      {
        return text.charAt(index);
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public CharIndexable frozen()
      {
        return this; // already unchangeable
      }
    };
  }

  /**
   * View an int array as an Indexable.
   * To view only part of the array use {@link #viewArray(char[], int, int)}.
   * @param array array used as base
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull char ... array)
  {
    return new Base()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public char get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * View part of an integer array as an Indexable.
   * @param array array used as base
   * @param start index of first byte to use from the array
   * @param length number of elements to use from the array
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable
   */
  @NotNull
  static Base viewArray(@NotNull char[] array,
                        int start,
                        int length)
  {
    if (start < 0  ||  length < 0  ||  start + length > array.length) {
      throw new IndexOutOfBoundsException("Start or end outside of range!");
    }
    if (length == 0) {
      return EMPTY;
    }
    return new Base()
    {
      @Override
      public int size()
      {
        return length;
      }

      @Override
      public char get(int index)
      {
        if (index < 0  ||  index >= length) {
          throw new IndexOutOfBoundsException(String.format("Index %d outside range [0, %d[!",
                                                            index, length));
        }
        return array[index + start];
      }
    };
  }

  /**
   * View an object array as a char indexable.
   * @param array         viewed array 
   * @param valueExtract  converter from array elements to the char values of this indexable 
   * @return indexable view of the given array, any changes to the underlying array will be reflected by the returned Indexable 
   * @param <T> array element type
   */
  @NotNull
  static <T> CharIndexable.Base viewArray(@NotNull T[] array,
                                          @NotNull ToCharFunction<? super T> valueExtract)
  {
    if (array.length == 0) {
      return EMPTY;
    }
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        return valueExtract.applyAsChar(array[index]);
      }

      @Override
      public int size()
      {
        return array.length;
      }
    };
  }

  /**
   * View a single char value as a char indexable of size 1.
   * @param value single value
   * @return indexable with one value
   */
  @NotNull
  static CharIndexable.Base singleton(char value)
  {
    return new CharIndexable.Base() {
      @Override
      public int size()
      {
        return 1;
      }

      @Override
      public char get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException(String.format("Index %d for indexable of size 1!", index));
        }
        return value;
      }

      @Override
      public boolean isEmpty()
      {
        return false;
      }

      @NotNull
      @Override
      public Base reverse()
      {
        return this;
      }

      @NotNull
      @Override
      public CharIndexable rotated(int steps)
      {
        return this;
      }

      @NotNull
      @Override
      public Collection<Character> asCollection()
      {
        return Collections.singleton(value);
      }

      @NotNull
      @Override
      public List<Character> asList()
      {
        return Collections.singletonList(value);
      }

      @NotNull
      @Override
      public char[] toArray()
      {
        return new char[] { value };
      }

      @Override
      public int addToArray(@NotNull char[] array, int pos)
      {
        array[pos] = value;
        return pos + 1;
      }

      @NotNull
      @Override
      public Iterable<Integer> indexes()
      {
        return IntIndexable.SINGLE_0;
      }

      @NotNull
      @Override
      public IntIndexable intIndexes()
      {
        return IntIndexable.SINGLE_0;
      }
    };
  }

  /**
   * Get a char indexable of a given size which always returns the same value.
   * @param size  size of the returned indexable
   * @param value value returned for each element
   * @return indexable of size {@code size} with always the same element
   */
  @NotNull
  static CharIndexable.Base init(int size, char value)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return value;
      }

      @Override
      public int size()
      {
        return 0;
      }

      @NotNull
      @Override
      public CharIndexable frozen()
      {
        return this;
      }
    };
  }

  /**
   * Get a char indexable of a given size which returns elements created by index.
   * This will always call the {@code producer} when an element is requested.
   * Use {@link #frozen()} on the result to create an indexable which avoids this.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value              
   * @return an indexable with the given size which will return elements by calling the {@code producer}
   */
  @NotNull
  static CharIndexable.Base viewByIndex(int size, @NotNull IntToCharFunction1 producer)
  {
    if (size == 0) {
      return EMPTY;
    }
    if (size < 0) {
      throw new IndexOutOfBoundsException("Indexables with negative size are impossible: "+size);
    }
    return new CharIndexable.Base()
    {
      @Override
      public char get(int index)
      {
        if (index < 0  ||  index >= size) {
          throw new IndexOutOfBoundsException(String.format("Element %d requested from indexable with size %d!",
                                                            index, size));
        }
        return producer.applyAsChar(index);
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get a char indexable of a given size which returns elements created by index.
   * This will call the producer only once for each element during the call of this method,
   * and return the results on later requests.
   * @param size size of the returned indexable
   * @param producer producer which is called with an index and expected to return the associated value
   * @return indexable with elements initialized by the {@code producer}
   * @see #viewByIndex(int, IntToCharFunction1)
   */
  @NotNull
  static CharIndexable initByIndex(int size, @NotNull IntToCharFunction1 producer)
  {
    return viewByIndex(size, producer).frozen();
  }

  /**
   * An empty indexable.
   */
  @NotNull
  Base EMPTY = new Base()
  {
    @Override
    public int size()
    {
      return 0;
    }

    @Override
    public char get(int index)
    {
      throw new IndexOutOfBoundsException("Empty indexable has no elements!");
    }

    @Override
    public boolean isEmpty()
    {
      return true;
    }

    @NotNull
    @Override
    public Base reverse()
    {
      return this;
    }

    @NotNull
    @Override
    public CharIndexable rotated(int steps)
    {
      return this;
    }

    @NotNull
    @Override
    public Collection<Character> asCollection()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<Character> asList()
    {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public PrimitiveIterator.OfInt intIterator()
    {
      return Types.EMPTY_INT_ITERATOR;
    }

    @NotNull
    @Override
    public PrimitiveCharIterator charIterator()
    {
      return Types.EMPTY_CHAR_ITERATOR;
    }

    @NotNull
    @Override
    public Iterator<Character> iterator()
    {
      return Types.emptyIterator();
    }

    @NotNull
    @Override
    public ListIterator<Character> listIterator()
    {
      return Types.emptyListIterator();
    }

    @NotNull
    @Override
    public char[] toArray()
    {
      return Empty.CHAR_ARRAY;
    }

    @Override
    public int addToArray(@NotNull char[] array, int pos)
    {
      return pos;
    }

    @NotNull
    @Override
    public Iterable<Integer> indexes()
    {
      return Types.emptyIterable();
    }

    @NotNull
    @Override
    public IntIndexable intIndexes()
    {
      return IntIndexable.EMPTY;
    }

    @NotNull
    @Override
    public IntIndexable.Base asIntIndexable()
    {
      return IntIndexable.EMPTY;
    }

    @Override
    public char foldLeft(char initialValue, @NotNull CharOperator2 foldOperation)
    {
      return initialValue;
    }

    @NotNull
    @Override
    public CharIndexable frozen()
    {
      return this;
    }

    @Override
    public @NotNull <T> Indexable<T> viewChars(@NotNull CharFunction1<? extends T> mapper)
    {
      return Indexable.emptyIndexable();
    }

    @Override
    public void forEachChar(@NotNull CharConsumer consumer)
    {
    }

    @Override
    public <E extends Exception> void forEachCharFragile(@NotNull FragileCharConsumer<E> action) throws E
    {
    }

    @Override
    public boolean containsChar(char value)
    {
      return false;
    }

    @Override
    public void forEach(Consumer<? super Character> action)
    {
    }

    @NotNull
    @Override
    public String makeString()
    {
      return Empty.STRING;
    }

    @Override
    public int addToArray(@NotNull char[] array, int arrayPos, int index, int length)
    {
      return index;
    }
  };

  /**
   * Get an empty indexable set.
   * @return empty indexable
   */
  @NotNull
  static Base emptyIndexable()
  {
    return EMPTY;
  }

  /**
   * Create a string representation of the given indexable.
   * @param indexable indexable
   * @return string representation
   * @see #makeString()
   */
  @NotNull
  static String toString(@NotNull CharIndexable indexable)
  {
    if (indexable.isEmpty()) {
      return Indexable.EMPTY_INDEXABLE_STRING;
    }
    final StringBuilder sb = new StringBuilder();
    sb.append('[').append(indexable.get(0));
    for (PrimitiveCharIterator it = indexable.tailSet(1).charIterator(); it.hasNext();  ) {
      sb.append(',').append(it.nextChar());
    }
    sb.append(']');
    return sb.toString();
  }

  /**
   * Calculate a hashcode for a char indexable.
   * @param indexable indexable for which the hash code is required
   * @return hash code for the given indexable
   */
  static int hash(@NotNull CharIndexable indexable)
  {
    // this follows Arrays.hashCode()
    int result = 1;

    for (PrimitiveCharIterator it = indexable.charIterator();  it.hasNext();  ) {
      result = 31 * result + Character.hashCode(it.next());
    }

    return result;
  }

  /**
   * Are two char indexables equal?
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static boolean equal(@NotNull CharIndexable indexable1,
                       @NotNull CharIndexable indexable2)
  {
    if (indexable1 == indexable2) {
      return true;
    }
    if (indexable1.size() != indexable2.size()) {
      return false;
    }
    final PrimitiveCharIterator it1 = indexable1.charIterator();
    final PrimitiveCharIterator it2 = indexable2.charIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      if (it1.nextChar() != it2.nextChar()) {
        return false;
      }
    }
    return !(it1.hasNext()  ||  it2.hasNext());
  }

  /**
   * Are two char indexables equal?
   * This is a convenience method which may be called from
   * implementations for their {@link Object#equals(Object)}
   * method.
   * @param indexable1  first indexable
   * @param indexable2  object expected to be an indexable itself
   * @return {@code true} if both indexables contain the same values in the same sequence<br>
   *         {@code false} if the second object is not an indexable, of if sizes or values differ
   */
  static boolean equal(@NotNull CharIndexable indexable1,
                       @Nullable Object indexable2)
  {
    if (indexable2 instanceof CharIndexable) {
      return equal(indexable1, (CharIndexable)indexable2);
    }
    return false;
  }

  /**
   * Compare two char indexables.
   * This compares the two indexable lexically element by element
   * until either the first difference is found (smaller of the two
   * values define the lower indexable) or the end of one is met
   * (shorter indexable is less).
   *
   * @param indexable1 first indexable
   * @param indexable2 second indexable
   * @return {@code < 0>} if {@code indexable1 < indxable2},
   *         {@code 0} if {@code indexable1 == indexable2}, or
   *         {@code > 0} if {@code indexable1 > indexable2}
   */
  static int compare(@NotNull CharIndexable indexable1,
                     @NotNull CharIndexable indexable2)
  {
    final PrimitiveCharIterator it1 = indexable1.charIterator();
    final PrimitiveCharIterator it2 = indexable2.charIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int compare = Character.compare(it1.nextChar(),
                                            it2.nextChar());
      if (compare != 0) {
        return compare;
      }
    }
    if (it1.hasNext()) {
      return 1;
    }
    if (it2.hasNext()) {
      return -1;
    }
    return 0;
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}. It also provides
   * {@link Comparable#compareTo(Object)}.
   * @see CharIndexable#asBase() 
   */
  abstract class Base implements CharIndexable,
                                 Comparable<CharIndexable>
  {
    @Override
    public int compareTo(@NotNull CharIndexable o)
    {
      Objects.requireNonNull(o);
      return compare(this, o);
    }

    @Override
    public int hashCode()
    {
      return hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      return equal(this, obj);
    }

    /**
     * Get a string representation of this indexable.
     *
     * @return string representation representing the indexable nature
     *         of the contained characters, use {@link #makeString()}
     *         to get a simple concatenated string
     */
    @Override
    public String toString()
    {
      return CharIndexable.toString(this);
    }

    @NotNull
    @Override
    public Base asBase()
    {
      return this;
    }
  }
}
